package com.bizunited.platform.core.service.dataview;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import com.bizunited.platform.core.vo.dataview.DataviewExportExcelFieldVo;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.security.Principal;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

/**
 * DataViewEntity业务模型的服务层接口定义
 * @author saturn
 */
public interface DataViewService {
  /**
   * 在创建一个新的DataViewEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewEntity dataViewEntity);
  /**
   * 创建一个新的DataViewEntity模型对象，ignoreValidate参数表示是否在进行创建前进行属性验证
   */
  public DataViewEntity create(DataViewEntity dataViewEntity ,boolean ignoreValidate);
  /**
   * 更新一个已有的DataViewEntity模型对象，其主键属性必须有值
   */
  public DataViewEntity update(DataViewEntity dataViewEntity);
  /**
   * 在更新一个已有的DataViewEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewEntity dataViewEntity);
  /**
   * 按照数据源ID查询DataViewEntity
   */
  public List<DataViewEntity> findByDataSource(String dataSourceId);
  /**
   * 按照数据库分组查询DataViewEntity
   */
  public List<DataViewEntity> findByDataViewGroup(String dataViewGroupId);
  /**
   * 按照结构字段查询DataViewEntity
   */
  public DataViewEntity findByFields(String fieldsId);
  /**
   * 按照过滤筛选字段查询DataViewEntity
   */
  public DataViewEntity findByFilters(String filtersId);
  /**
   * 按照系统查询字段查询DataViewEntity
   */
  public DataViewEntity findBySystemFilters(String systemFiltersId);
  /**
   * 按照数据视图的唯一业务编号信息进行详情查询（包括关联信息），注意，这里的详细查询并不包括数据权限信息
   * @param code 数据视图的唯一业务编号信息
   */
  public DataViewEntity findDetailsByCode(String code);

  /**
   * 简单根据CODE查询
   * @param code
   * @return
   */
  DataViewEntity findByCode(String code);
  /**
   * 可以按照多个数据ID进行详情查询（包括关联信息）。注意，这里的详细查询并不包括数据权限信息
   * @param code 数据视图的唯一业务编号信息
   * */
  public Set<DataViewEntity> findDetailsByIds(String[] ids);
  /**
   * 按照数据视图的ID信息进行详情查询（不包括关联信息）
   * @param code 数据视图的ID信息
   */
  public DataViewEntity findById(String id);
  /**
   * 按照数据源code查询 视图、分组、字段等统计信息
   * @param dataSourceCode
   * @return
   */
  public JSONArray findByDataSourceCode(String dataSourceCode);
  /**
   * 根据已有的视图ID以及系统参数信息（json格式），查找已有的数据信息
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @param primaryKey 唯一性质的键名称
   * @param primaryValue 唯一性质的键值
   * @return
   */
  public JSONArray findByViewAndDataKey(DataViewEntity dataView, InvokeParams invokeParams,Principal principal,String primaryKey,JSONArray primaryValue);
  /**
   * 按照指定的数据视图ID信息，查询最终满足这些id编号的数据条目。<br>
   * 这个方法一般用于验证一批给定的id，是否都能在系统中找到对应的数据
   * @param ids 指定的id信息
   * @return 满足的条目数量将会被返回
   */
  public int countByIds(String[] ids);
  /**
   *  按照主键进行信息的真删除
   * @param id 主键
   */
  public void deleteById(String id);

  /**
   * 导出数据视图数据到excel，如果dataSourceCode参数为空，则执行主数据源查询，否则执行第三方数据源查询数据
   * @param dataSourceCode 数据源编码
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param inputParams 入参
   * @param principal 操作者信息
   * @param excelFields
   * @return
   */
  byte[] exportExcel(String dataSourceCode, DataViewEntity dataView, InvokeParams inputParams, Principal principal, List<DataviewExportExcelFieldVo> excelFields);

  /**
   * 使用一个指定的第三方数据源，进行数据视图中设定的最原始SQL的预执行。预执行后，将向调用者返回执行后的SQL输出结构、以及要求填写的系统参数信息</br>
   * 注意，该方法只会对原始SQL进行执行（自动替换原始SQL中的系统参数信息），并不会将预执行后的结构做持久化保存；要进行持久化保存，请使用本服务中的creat方法
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 这个对象包括了数据视图最原始的SQL设定信息。
   * @return
   */
  public DataViewEntity executeResource(String dataSourceCode , DataViewEntity dataView);

  /**
   * 使用本本进程的主数据源进行数据视图中设定的最原始SQL的保存。其它操作特性请参见:</br>
   * executeResource(String dataSourceCode , DataViewEntity dataView)方法
   * @param dataView
   * @return
   */
  public DataViewEntity executeResource(DataViewEntity dataView);

  /**
   * 使用指定的第三方数据源，进行正式的查询SQL执行，但是最多只返回一条记录
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public Future<?> executeTop1(String dataSourceCode , DataViewEntity dataView , InvokeParams invokeParams,Principal principal);

  /**
   * 使用本地进程的主数据源，进行正式的查询SQL执行，但是最多只返回一条记录
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public JSONArray executeTop1(DataViewEntity dataView , InvokeParams invokeParams,Principal principal);

  /**
   * 使用指定的第三方数据源，进行正式的分页SQL查询
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public Future<?> execute(String dataSourceCode , DataViewEntity dataView , InvokeParams invokeParams , Pageable pageable,Principal principal);

  /**
   * 使用指定的第三方数据源，进行正式的查询SQL执行
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public Future<?> execute(String dataSourceCode , DataViewEntity dataView , InvokeParams invokeParams,Principal principal);

  /**
   * 使用本地进程的主数据源，进行正式的分页SQL查询
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public Page<JSONObject> execute(DataViewEntity dataView , InvokeParams invokeParams , Pageable pageable,Principal principal);

  /**
   * 使用本地进程的主数据源，进行正式的查询SQL执行
   * @param dataView 完成的，带有原始SQL、执行输出结构、系统参数描述、用户参数描述的，已被持久化的数据视图描述信息
   * @param invokeParams 入参
   * @param principal 操作者信息
   * @return
   */
  public JSONArray execute(DataViewEntity dataView , InvokeParams invokeParams,Principal principal);

  /**
   * 通过数据视图编码，返回当前用户能查看到的所有字段。
   * @param dataViewCode
   * @param principal
   * @return
   */
   Set<String> findFieldsByDataViewCode(String dataViewCode, Principal principal);
}