package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthHorizontalRelationRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalRelationService;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DataViewAuthHorizontalRelationEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewAuthHorizontalRelationServiceImpl")
public class DataViewAuthHorizontalRelationServiceImpl implements DataViewAuthHorizontalRelationService {
  @Autowired
  private DataViewAuthHorizontalRelationRepository dataViewAuthHorizontalRelationRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Override
  @Transactional
  public void create(DataViewAuthHorizontalEntity dataViewAuthHorizontalEntity, Set<DataViewAuthHorizontalRelationEntity> authRelations) {
    Validate.notNull(dataViewAuthHorizontalEntity, "横向数据权限的对象信息必须传入!!");
    if(CollectionUtils.isEmpty(authRelations)) {
      return;
    }
    authRelations.forEach(x -> {
      x.setAuthHorizontal(dataViewAuthHorizontalEntity);
      this.createValidation(x);
    });
    dataViewAuthHorizontalRelationRepository.saveAll(authRelations);
  }

  /**
   * 在创建一个新的DataViewAuthHorizontalRelationEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewAuthHorizontalRelationEntity dataViewAuthHorizontalRelationEntity) {
    Validate.notNull(dataViewAuthHorizontalRelationEntity , "权限领域信息必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewAuthHorizontalRelationEntity.getId()), "添加权限领域时，（主键）不能有值！");
    dataViewAuthHorizontalRelationEntity.setId(null);

    Validate.notNull(dataViewAuthHorizontalRelationEntity.getAuthHorizontal(),"横向权限信息不能为空！");
    Validate.notBlank(dataViewAuthHorizontalRelationEntity.getAuthRelationIds(), "权限领域不能为空！");
    Validate.notNull(dataViewAuthHorizontalRelationEntity.getSortIndex(),"权限领域的排序信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(dataViewAuthHorizontalRelationEntity.getAuthRelationIds() == null || dataViewAuthHorizontalRelationEntity.getAuthRelationIds().length() <= 255 , "权限关系集,填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 在更新一个已有的DataViewAuthHorizontalRelationEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewAuthHorizontalRelationEntity dataViewAuthHorizontalRelationEntity) {
    Validate.notNull(dataViewAuthHorizontalRelationEntity , "权限领域信息必须传入!!");
    Validate.isTrue(StringUtils.isNotBlank(dataViewAuthHorizontalRelationEntity.getId()), "修改权限领域信息时，（主键）必须有值！");

    DataViewAuthHorizontalEntity authHorizontal = dataViewAuthHorizontalRelationEntity.getAuthHorizontal();
    Validate.notNull(authHorizontal,"传入的横向权限对象不能为空，请检查!!");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(dataViewAuthHorizontalRelationEntity.getAuthRelationIds(), "权限领域不能为空！");
    Validate.notNull(dataViewAuthHorizontalRelationEntity.getSortIndex(),"权限领域的排序信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewAuthHorizontalRelationEntity.getAuthRelationIds() == null || dataViewAuthHorizontalRelationEntity.getAuthRelationIds().length() <= 255 , "权限关系集,填入值超过了限定长度(255)，请检查!");
  }


  @Override
  public Set<DataViewAuthHorizontalRelationEntity> findDetailsByAuthHorizontal(String authHorizontal) { 
    if(StringUtils.isBlank(authHorizontal)) { 
      return Sets.newHashSet();
    }
    return this.dataViewAuthHorizontalRelationRepository.findDetailsByAuthHorizontal(authHorizontal);
  }
  @Override
  public DataViewAuthHorizontalRelationEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewAuthHorizontalRelationRepository.findDetailsById(id);
  }
  @Override
  public DataViewAuthHorizontalRelationEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    Optional<DataViewAuthHorizontalRelationEntity> op = this.dataViewAuthHorizontalRelationRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewAuthHorizontalRelationEntity> op = this.dataViewAuthHorizontalRelationRepository.findById(id);
    op.ifPresent(dataViewAuthHorizontalRelationEntity -> this.dataViewAuthHorizontalRelationRepository.delete(dataViewAuthHorizontalRelationEntity));
  }

  @Override
  @Transactional
  public void deleteByAuthHorizontal(String authHorizontal) {
    Validate.notBlank(authHorizontal, "进行删除时，必须给定横向数据权限信息!");
    this.dataViewAuthHorizontalRelationRepository.deleteByAuthHorizontal(authHorizontal);
  }

  /**
   * 处理可能的更新、增添、删除权限领域信息
   * @param dataViewAuthHorizontalEntity
   * @param authRelations
   */
  @Override
  @Transactional
  public void save(DataViewAuthHorizontalEntity dataViewAuthHorizontalEntity, Set<DataViewAuthHorizontalRelationEntity> authRelations){
    //这里的关系权限集可能包含新增、更新与删除，所以需分情况验证，但是删除的验证不用理会
    Set<DataViewAuthHorizontalRelationEntity> needInsertAuthRelations = null;
    Set<DataViewAuthHorizontalRelationEntity> needUpdateAuthRelations = null;
    if (!CollectionUtils.isEmpty(authRelations)) {
      needInsertAuthRelations = authRelations.stream().filter(x -> StringUtils.isBlank(x.getId())).collect(Collectors.toSet());
    }
    if (!CollectionUtils.isEmpty(authRelations)) {
      needUpdateAuthRelations = authRelations.stream().filter(x -> StringUtils.isNotBlank(x.getId())).collect(Collectors.toSet());
    }

    Set<DataViewAuthHorizontalRelationEntity> dbAuthHorizoncalRelations = this.findDetailsByAuthHorizontal(dataViewAuthHorizontalEntity.getId());
    Set<String> needDeleteAuthHorizoncalRelationKeys = nebulaToolkitService.collectionDiffent(dbAuthHorizoncalRelations, authRelations, DataViewAuthHorizontalRelationEntity::getId);


    //1.可能的新增
    this.create(dataViewAuthHorizontalEntity, needInsertAuthRelations);

    //2.可能的更新
    if(!CollectionUtils.isEmpty(needUpdateAuthRelations)){
      needUpdateAuthRelations.forEach(y -> {
        y.setAuthHorizontal(dataViewAuthHorizontalEntity);
        this.updateValidation(y);
      });
      dataViewAuthHorizontalRelationRepository.saveAll(needUpdateAuthRelations);
    }

    //3.可能的删除
    if (!CollectionUtils.isEmpty(needDeleteAuthHorizoncalRelationKeys)) {
      needDeleteAuthHorizoncalRelationKeys.forEach(x -> dataViewAuthHorizontalRelationRepository.deleteById(x));
    }
  }

}
