package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.PositionEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.entity.UuidEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthHorizontalRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserGroupService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DataViewAuthHorizontalEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewAuthHorizontalServiceImpl")
public class DataViewAuthHorizontalServiceImpl implements DataViewAuthHorizontalService {
  @Autowired
  private DataViewRepository dataViewRepository;
  @Autowired
  private DataViewAuthHorizontalRepository dataViewAuthHorizontalRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private RoleService roleService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private UserGroupService userGroupService;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private DataViewAuthHorizontalRelationService dataViewAuthHorizontalRelationService;
  /**
  private DataViewAuthHorizontalRelationRepository dataViewAuthHorizontalRelationRepository;

  /**
   * Kuiper表单引擎用于减少编码工作量的工具包
   */ 
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 批量保存横向数据权限
   * @param authHorizontals
   * @return
   */
  @Transactional
  public Set<DataViewAuthHorizontalEntity> create(Set<DataViewAuthHorizontalEntity> authHorizontals) {
    Validate.notEmpty(authHorizontals,"传入的横向权限信息不能为空！");
    //验证数据正确性以及常规校验
    for(DataViewAuthHorizontalEntity e : authHorizontals){
      this.createValidation(e);
    }

    //批量保存
    this.dataViewAuthHorizontalRepository.saveAll(authHorizontals);

    //当传入参数方式是固定值时
    Set<DataViewAuthHorizontalEntity> auths = authHorizontals.stream().filter(e -> e.getParamSourceType() == 2).collect(Collectors.toSet());
    if (!CollectionUtils.isEmpty(auths)) {
      for (DataViewAuthHorizontalEntity e : auths) {
        Set<DataViewAuthHorizontalRelationEntity> needCreates = e.getAuthRelations();
        dataViewAuthHorizontalRelationService.create(e, needCreates);
      }
    }
    // 返回结果
    return authHorizontals;
  }


  /**
   * 在创建一个新的DataViewAuthHorizontalEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewAuthHorizontalEntity dataViewAuthHorizontalEntity) {
    Validate.notNull(dataViewAuthHorizontalEntity , "横向数据权限的对象信息必须传入!!");
    Validate.isTrue(StringUtils.isBlank(dataViewAuthHorizontalEntity.getId()), "添加横向数据权限信息时，（主键）不能有值！");
    dataViewAuthHorizontalEntity.setId(null);

    Validate.notNull(dataViewAuthHorizontalEntity.getSortIndex(), "排序值不能为空!");
    Validate.notNull(dataViewAuthHorizontalEntity.getAuthType(), "权限类型不能为空！");
    Validate.notBlank(dataViewAuthHorizontalEntity.getOprtType(), "操作符类型不能为空！");
    Validate.notBlank(dataViewAuthHorizontalEntity.getParamType(), "参数类型不能为空！");
    Validate.notNull(dataViewAuthHorizontalEntity.getParamSourceType(), "参数来源类型不能为空！");

    DataViewFieldEntity field = dataViewAuthHorizontalEntity.getField();
    Validate.notNull(field,"添加横向数据权限信息时，必须指定权限字段，请检查!!");

    DataViewEntity dataView = dataViewAuthHorizontalEntity.getDataView();
    Validate.notNull(dataView,"添加横向数据权限信息时，必须传入数据视图信息，请检查!!");

    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewAuthHorizontalEntity.getOprtType() == null || dataViewAuthHorizontalEntity.getOprtType().length() <= 32 , "操作符类型,填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(dataViewAuthHorizontalEntity.getParamType() == null || dataViewAuthHorizontalEntity.getParamType().length() <= 64 , "参数类型,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewAuthHorizontalEntity.getParamKey() == null || dataViewAuthHorizontalEntity.getParamKey().length() <= 64 , "预制值标识,填入值超过了限定长度(64)，请检查!");
  }
  
  /**
   * 批量更新横向数据权限
   * @param authHorizontals
   * @return
   */
  @Transactional
  public Set<DataViewAuthHorizontalEntity> update(Set<DataViewAuthHorizontalEntity> authHorizontals , String dataViewId) {
    Validate.notBlank(dataViewId,"传入的视图ID不能为空!");
    Optional<DataViewEntity> op = dataViewRepository.findById(dataViewId);
    DataViewEntity dataView = op.orElse(null);
    Validate.notNull(dataView,"根据视图ID，没有获取到相应的视图信息，请检查!!");

    if(CollectionUtils.isEmpty(authHorizontals)){
      Set<DataViewAuthHorizontalEntity> authHors = dataViewAuthHorizontalRepository.findDetailsByDataViewCode(dataView.getCode());
      if(!CollectionUtils.isEmpty(authHors)){
        for(DataViewAuthHorizontalEntity x : authHors){
          dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(x.getId());
        }
        dataViewAuthHorizontalRepository.deleteByDataView(dataViewId);
      }
      return Sets.newHashSet();
    }

    Set<DataViewAuthHorizontalEntity> needInsertSets = authHorizontals.stream().filter(e -> StringUtils.isBlank(e.getId())).collect(Collectors.toSet());
    Set<DataViewAuthHorizontalEntity> needUpdateSets = authHorizontals.stream().filter(e -> StringUtils.isNotBlank(e.getId())).collect(Collectors.toSet());
    Set<DataViewAuthHorizontalEntity> dbAuthHorizoncals = dataViewAuthHorizontalRepository.findDetailsByDataViewCode(dataView.getCode());
    Set<String> needDeleteFieldKeys = nebulaToolkitService.collectionDiffent(dbAuthHorizoncals,authHorizontals, DataViewAuthHorizontalEntity::getId);

    //可能需要增加的
    if(!CollectionUtils.isEmpty(needInsertSets)){
      needInsertSets.forEach(h -> h.setDataView(dataView));
      this.create(needInsertSets);
    }
    //可能需要删除的
    if(!CollectionUtils.isEmpty(needDeleteFieldKeys)){
      for(String authHorizontalKey : needDeleteFieldKeys){
        //删除明细
        dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(authHorizontalKey);
        //删除主数据
        dataViewAuthHorizontalRepository.deleteById(authHorizontalKey);
      }
    }
    if(CollectionUtils.isEmpty(needUpdateSets)){
      return authHorizontals;
    }

    //可能需要更新的
    //更新前需过验证
    for(DataViewAuthHorizontalEntity e : needUpdateSets){
      this.updateValidation(e);
      // ===================基本信息
      String currentId = e.getId();
      Optional<DataViewAuthHorizontalEntity> opt = dataViewAuthHorizontalRepository.findById(currentId);
      DataViewAuthHorizontalEntity currentDataViewAuthHorizontalEntity = opt.orElse(null);
      Validate.notNull(currentDataViewAuthHorizontalEntity ,"更新时，未发现指定的横向权限对象信息");
      // 开始重新赋值——一般属性
      currentDataViewAuthHorizontalEntity.setAuthType(e.getAuthType());
      currentDataViewAuthHorizontalEntity.setOprtType(e.getOprtType());
      currentDataViewAuthHorizontalEntity.setParamSourceType(e.getParamSourceType());
      currentDataViewAuthHorizontalEntity.setParamType(e.getParamType());
      currentDataViewAuthHorizontalEntity.setParamKey(e.getParamKey());
      currentDataViewAuthHorizontalEntity.setField(e.getField());
      currentDataViewAuthHorizontalEntity.setDataView(dataView);
      currentDataViewAuthHorizontalEntity.setSortIndex(e.getSortIndex());


      //当参数来源传入方式不是固定值时，清空权限关系
      if(e.getParamSourceType() != 2){
        e.setAuthRelations(null);
      }
      this.dataViewAuthHorizontalRepository.save(currentDataViewAuthHorizontalEntity);

      Set<DataViewAuthHorizontalRelationEntity> authRelations = e.getAuthRelations();
      if(CollectionUtils.isEmpty(authRelations) && e.getParamSourceType() == 3){
        continue;
      }

      dataViewAuthHorizontalRelationService.save(currentDataViewAuthHorizontalEntity,authRelations);

    }
    return authHorizontals;
  }
  
  /**
   * 在更新一个已有的DataViewAuthHorizontalEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewAuthHorizontalEntity dataViewAuthHorizontalEntity) {
    Validate.notNull(dataViewAuthHorizontalEntity,"更新时，横向权限信息必须传入！！");
    Validate.isTrue(StringUtils.isNotBlank(dataViewAuthHorizontalEntity.getId()), "修改横向数据权限信息时，（主键）必须有值！");

    Validate.notNull(dataViewAuthHorizontalEntity.getSortIndex(), "排序字段不能为空");
    Validate.notNull(dataViewAuthHorizontalEntity.getParamSourceType(), "参数来源类型不能为空！");
    Validate.notNull(dataViewAuthHorizontalEntity.getAuthType(), "权限类型不能为空！");
    Validate.notBlank(dataViewAuthHorizontalEntity.getOprtType(), "操作符类型不能为空！");
    Validate.notBlank(dataViewAuthHorizontalEntity.getParamType(), "参数类型不能为空！");

    DataViewFieldEntity field = dataViewAuthHorizontalEntity.getField();
    Validate.notNull(field,"更新横向数据权限信息时，必须指定权限字段，请检查!!");

    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewAuthHorizontalEntity.getOprtType() == null || dataViewAuthHorizontalEntity.getOprtType().length() < 32 , "操作符类型,填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(dataViewAuthHorizontalEntity.getParamType() == null || dataViewAuthHorizontalEntity.getParamType().length() < 64 , "参数类型,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewAuthHorizontalEntity.getParamKey() == null || dataViewAuthHorizontalEntity.getParamKey().length() < 64 , "预制值标识,填入值超过了限定长度(64)，请检查!");
  }


  /**
   * 根据指定的字段信息和数据视图ID，查询对应的横向权限信息
   * @param field 关联的 指定的用户权限筛选字段
   * @param dataViewId 视图ID
   * @return
   */
  @Override
  public DataViewAuthHorizontalEntity findDetailsByFieldAndDataView(String field , String dataViewId) {
    if(StringUtils.isBlank(field) || StringUtils.isBlank(dataViewId)) {
      return null;
    }
    return this.dataViewAuthHorizontalRepository.findDetailsByFieldAndDataView(field,dataViewId);
  }
  @Override
  public Set<DataViewAuthHorizontalEntity> findDetailsByDataViewCode(String dataView) {
    if(StringUtils.isBlank(dataView)) { 
      return Sets.newHashSet();
    }
    Set<DataViewAuthHorizontalEntity> auths = this.dataViewAuthHorizontalRepository.findDetailsByDataViewCode(dataView);
    Collection<DataViewAuthHorizontalEntity> collection = this.nebulaToolkitService.copyCollectionByWhiteList(auths,
        DataViewAuthHorizontalEntity.class,DataViewAuthHorizontalEntity.class, LinkedHashSet.class, ArrayList.class,
        "field","dataView","authRelations","authRelations.authHorizontal", "authRelations.relation");
    this.loadRelation(collection);
    return (Set<DataViewAuthHorizontalEntity>)collection;
  }

  /**
   * 加载关联的依赖信息
   * @param collection
   */
  private void loadRelation(Collection<DataViewAuthHorizontalEntity> collection) {
    if(CollectionUtils.isEmpty(collection)) {
      return;
    }
    for (DataViewAuthHorizontalEntity horizontal : collection) {
      Set<DataViewAuthHorizontalRelationEntity> authRelations = horizontal.getAuthRelations();
      if(CollectionUtils.isEmpty(authRelations)) {
        continue;
      }
      for (DataViewAuthHorizontalRelationEntity authRelation : authRelations) {
        UuidEntity relation = this.getRelation(horizontal, authRelation);
        authRelation.setRelation(relation);
      }
    }
  }

  /**
   * 获取关联的依赖信息
   * @param horizontal
   * @param authRelation
   * @return
   */
  private UuidEntity getRelation(DataViewAuthHorizontalEntity horizontal, DataViewAuthHorizontalRelationEntity authRelation) {
    if(StringUtils.isBlank(authRelation.getAuthRelationIds())) {
      return null;
    }
    UuidEntity relation = null;
    switch (horizontal.getAuthType()) {
      case 1:
        // 用户
        UserVo user = userService.findByAccount(authRelation.getAuthRelationIds());
        if(user != null) {
          relation = nebulaToolkitService.copyObjectByWhiteList(user, UserEntity.class, HashSet.class, ArrayList.class);
        }
        break;
      case 2:
        // 角色
        RoleVo role = roleService.findByCode(authRelation.getAuthRelationIds());
        if(role != null) {
          relation = nebulaToolkitService.copyObjectByWhiteList(role, RoleEntity.class, HashSet.class, ArrayList.class);
        }
        break;
      case 3:
        // 岗位
        PositionVo position = positionService.findByCode(authRelation.getAuthRelationIds());
        if(position != null) {
          relation = nebulaToolkitService.copyObjectByWhiteList(position, PositionEntity.class, HashSet.class, ArrayList.class);
        }
        break;
      case 4:
        // 用户组
        UserGroupVo userGroup = userGroupService.findByGroupName(authRelation.getAuthRelationIds());
        if(userGroup != null) {
          relation = nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupEntity.class, HashSet.class, ArrayList.class);
        }
        break;
      case 5:
        // 组织
        OrganizationVo organization = organizationService.findByCode(authRelation.getAuthRelationIds());
        if(organization != null) {
          relation = nebulaToolkitService.copyObjectByWhiteList(organization, OrganizationEntity.class, HashSet.class, ArrayList.class);
        }
        break;
      default:
        break;
    }
    return relation;
  }
  @Override
  public Set<DataViewAuthHorizontalEntity> findDetailsByDataViewIds(String[] dataViewIds) {
    if(dataViewIds == null || dataViewIds.length == 0) { 
      return Sets.newHashSet();
    }
    
    Set<DataViewAuthHorizontalEntity> auths = this.dataViewAuthHorizontalRepository.findDetailsByDataViewIds(dataViewIds);
    Collection<DataViewAuthHorizontalEntity> collection = this.nebulaToolkitService.copyCollectionByWhiteList(auths,DataViewAuthHorizontalEntity.class,DataViewAuthHorizontalEntity.class, HashSet.class, ArrayList.class, "field","dataView","authRelations","authRelations.authHorizontal");
    return (Set<DataViewAuthHorizontalEntity>)collection;
  }
  @Override
  public DataViewAuthHorizontalEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewAuthHorizontalRepository.findDetailsById(id);
  }
  @Override
  public DataViewAuthHorizontalEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    Optional<DataViewAuthHorizontalEntity> op = dataViewAuthHorizontalRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewAuthHorizontalEntity> op = dataViewAuthHorizontalRepository.findById(id);
    op.ifPresent(dataViewAuthHorizontalEntity -> this.dataViewAuthHorizontalRepository.delete(dataViewAuthHorizontalEntity));
  }

  @Override
  @Transactional
  public void deleteByDateView(DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity, "数据视图为空，请检查！");
    Validate.notBlank(dataViewEntity.getCode(), "数据视图编号不能为空！");
    Validate.notBlank(dataViewEntity.getId(), "修改信息时，当前信息的数据编号（主键）必须有值！");
    Set<DataViewAuthHorizontalEntity> hors = dataViewAuthHorizontalRepository.findDetailsByDataViewCode(dataViewEntity.getCode());
    if(!CollectionUtils.isEmpty(hors)){
      for(DataViewAuthHorizontalEntity v : hors){
        dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(v.getId());
      }
      dataViewAuthHorizontalRepository.deleteByDataView(dataViewEntity.getId());
    }
  }

  @Override
  @Transactional
  public void deleteByField(String fieldId) {
    Validate.notBlank("数据视图字段ID不能为空！");
    Set<DataViewAuthHorizontalEntity> horizontals = dataViewAuthHorizontalRepository.findByFieldId(fieldId);
    if(CollectionUtils.isEmpty(horizontals)) {
      return;
    }
    for (DataViewAuthHorizontalEntity horizontal : horizontals) {
      dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(horizontal.getId());
      dataViewAuthHorizontalRepository.delete(horizontal);
    }
  }
} 