package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthInterceptorRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthInterceptorService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;

/**
 * 数据视图权限拦截器服务实现
 * @Author: Paul Chan
 * @Date: 2019-12-23 10:24
 */
@Service("DataViewAuthInterceptorServiceImpl")
public class DataViewAuthInterceptorServiceImpl implements DataViewAuthInterceptorService {

  @Autowired
  private DataViewAuthInterceptorRepository dataViewAuthInterceptorRepository;
  @Autowired
  private DataViewService dataViewService;

  @Override
  @Transactional
  public DataViewAuthInterceptorEntity create(DataViewAuthInterceptorEntity interceptor) {
    this.createValidation(interceptor);
    return dataViewAuthInterceptorRepository.save(interceptor);
  }

  @Override
  @Transactional
  public DataViewAuthInterceptorEntity update(DataViewAuthInterceptorEntity interceptor) {
    this.updateValidation(interceptor);
    DataViewAuthInterceptorEntity interceptorEntity = dataViewAuthInterceptorRepository.findById(interceptor.getId()).orElse(null);
    Validate.notNull(interceptorEntity, "拦截器对象不存在");
    interceptorEntity.setClassType(interceptor.getClassType());
    interceptorEntity.setInterceptor(interceptor.getInterceptor());
    return dataViewAuthInterceptorRepository.save(interceptorEntity);
  }

  @Override
  public DataViewAuthInterceptorEntity findByDataView(String dataViewCode) {
    if(StringUtils.isBlank(dataViewCode)) {
      return null;
    }
    return dataViewAuthInterceptorRepository.findByDataViewCode(dataViewCode);
  }

  /**
   * 新增前数据校验
   * @param interceptor
   */
  private void createValidation(DataViewAuthInterceptorEntity interceptor) {
    Validate.notNull(interceptor, "新增对象不能为空！！");
    Validate.isTrue(interceptor.getId() == null, "新增对象ID不能有值");
    Validate.notBlank(interceptor.getInterceptor(), "拦截器不能为空！！");
    Validate.notNull(interceptor.getClassType(), "拦截器类类型不能为空");
    Validate.notNull(interceptor.getDataView(), "数据视图对象不能为空");
    Validate.notNull(interceptor.getDataView().getId(), "数据视图对象ID不能为空");
    DataViewEntity dataView = dataViewService.findById(interceptor.getDataView().getId());
    Validate.notNull(dataView, "未找到数据视图对象");
    DataViewAuthInterceptorEntity oldInterceptor = dataViewAuthInterceptorRepository.findByDataView(dataView.getId());
    Validate.isTrue(oldInterceptor == null, "该数据视图权限已存在拦截器，请更新");
  }

  /**
   * 更新前数据校验
   * @param interceptor
   */
  private void updateValidation(DataViewAuthInterceptorEntity interceptor) {
    Validate.notNull(interceptor, "更新对象不能为空！！");
    Validate.notBlank(interceptor.getId(), "更新对象ID不能为空");
    Validate.notNull(interceptor.getClassType(), "拦截器类类型不能为空");
    Validate.notNull(interceptor.getInterceptor(), "拦截器不能为空！！");
  }

}
