package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.repository.dataview.DataViewFieldRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.bizunited.platform.core.service.dataview.DataViewFieldService;
import com.bizunited.platform.core.service.dataview.DataViewFilterService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * DataViewFieldEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewFieldEntityServiceImpl")
public class DataViewFieldServiceImpl implements DataViewFieldService {
  @Autowired
  private DataViewFieldRepository dataViewFieldEntityRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DataViewFilterService dataViewFilterService;
  @Autowired
  private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Autowired
  private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  /**
   * 在创建一个新的DataViewFieldEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewFieldEntity dataViewFieldEntity) {
    Validate.notNull(dataViewFieldEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewFieldEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dataViewFieldEntity.setId(null);
    Validate.notBlank(dataViewFieldEntity.getFieldName(), "字段名（英文）不能为空！");
    Validate.notBlank(dataViewFieldEntity.getFieldType(), "字段类型不能为空！");
    Validate.notNull(dataViewFieldEntity.getSortIndex(), "字段排序不能为空！");
    Validate.notNull(dataViewFieldEntity.getPhysical(), "是否物理字段不能为空！");
    Validate.notNull(dataViewFieldEntity.getTstatus(), "是否有效不能为空！");
    Validate.notNull(dataViewFieldEntity.getDisplayName(), "显示中文名不能为空");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewFieldEntity.getFieldName() == null || dataViewFieldEntity.getFieldName().length() < 64 , "字段名（英文）,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getFieldType() == null || dataViewFieldEntity.getFieldType().length() < 64 , "字段类型,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getSchemaName() == null || dataViewFieldEntity.getSchemaName().length() < 64 , "数据分库,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getTargetTable() == null || dataViewFieldEntity.getTargetTable().length() < 64 , "物理数据表,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getTargetField() == null || dataViewFieldEntity.getTargetField().length() < 64 , "物理字段,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getDisplayName().length() < 64 , "显示中文名,填入值超过了限定长度(64)，请检查!");
  }
  @Transactional
  @Override
  public DataViewFieldEntity create(DataViewFieldEntity dataViewFieldEntity , boolean ignoreValidate) {
    if(!ignoreValidate) {
      this.createValidation(dataViewFieldEntity);
    }
    // 新增的时候增加UUID
    dataViewFieldEntity.setCode(UUID.randomUUID().toString());
    
    this.dataViewFieldEntityRepository.save(dataViewFieldEntity);
    
    return dataViewFieldEntity;
  }

  @Transactional
  @Override
  public DataViewFieldEntity update(DataViewFieldEntity dataViewFieldEntity) { 
    this.updateValidation(dataViewFieldEntity);
    // ===================基本信息
    String currentId = dataViewFieldEntity.getId();
    Optional<DataViewFieldEntity> op = this.dataViewFieldEntityRepository.findById(currentId);
    DataViewFieldEntity currentDataViewFieldEntity = op.orElse(null);
    Validate.notNull(currentDataViewFieldEntity,"未发现指定的原始模型对象信息");

    // 开始重新赋值，首先是一般对象，updateable = true形式的
    currentDataViewFieldEntity.setFieldName(dataViewFieldEntity.getFieldName());
    currentDataViewFieldEntity.setFieldType(dataViewFieldEntity.getFieldType());
    currentDataViewFieldEntity.setSortIndex(dataViewFieldEntity.getSortIndex());
    currentDataViewFieldEntity.setPhysical(dataViewFieldEntity.getPhysical());
    currentDataViewFieldEntity.setSchemaName(dataViewFieldEntity.getSchemaName());
    currentDataViewFieldEntity.setTargetTable(dataViewFieldEntity.getTargetTable());
    currentDataViewFieldEntity.setTargetField(dataViewFieldEntity.getTargetField());
    currentDataViewFieldEntity.setTstatus(dataViewFieldEntity.getTstatus());
    currentDataViewFieldEntity.setDisplayName(dataViewFieldEntity.getDisplayName());
    currentDataViewFieldEntity.setEnumsource(dataViewFieldEntity.getEnumsource());
    currentDataViewFieldEntity.setAuthType(dataViewFieldEntity.getAuthType());
    this.dataViewFieldEntityRepository.save(currentDataViewFieldEntity);
    // =============
    // ManyToMany 结构的关联关系，需要开发人员自行处理
    // =============
    return currentDataViewFieldEntity;
  }
  /**
   * 在更新一个已有的DataViewFieldEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewFieldEntity dataViewFieldEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dataViewFieldEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewFieldEntity.getFieldName(), "字段名（英文）不能为空！");
    Validate.notBlank(dataViewFieldEntity.getFieldType(), "字段类型不能为空！");
    Validate.notNull(dataViewFieldEntity.getSortIndex(), "字段排序不能为空！");
    Validate.notNull(dataViewFieldEntity.getPhysical(), "是否物理字段不能为空！");
    Validate.notBlank(dataViewFieldEntity.getTargetField(), "物理字段不能为空！");
    Validate.notBlank(dataViewFieldEntity.getDisplayName(), "显示中文名称不能为空！");
    Validate.notNull(dataViewFieldEntity.getTstatus(), "是否有效不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewFieldEntity.getFieldName() == null || dataViewFieldEntity.getFieldName().length() < 64 , "字段名（英文）,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getDisplayName() == null || dataViewFieldEntity.getDisplayName().length() < 64 , "字段显示中文名,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getFieldType() == null || dataViewFieldEntity.getFieldType().length() < 64 , "字段类型,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getSchemaName() == null || dataViewFieldEntity.getSchemaName().length() < 64 , "数据分库,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getTargetTable() == null || dataViewFieldEntity.getTargetTable().length() < 64 , "物理数据表,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewFieldEntity.getTargetField() == null || dataViewFieldEntity.getTargetField().length() < 64 , "物理字段,填入值超过了限定长度(64)，请检查!");
  }
  @Override
  public List<DataViewFieldEntity> findByDataView(String dataViewId) { 
    if(StringUtils.isBlank(dataViewId)) { 
      return Lists.newArrayList();
    }
    return this.dataViewFieldEntityRepository.findByDataView(dataViewId);
  }

  @Override
  public Set<DataViewFieldEntity> findAuthTypeFieldsByDataView(String dataViewId) {
    if(StringUtils.isBlank(dataViewId)) {
      return Collections.emptySet();
    }
    List<DataViewFieldEntity> dbFields = this.dataViewFieldEntityRepository.findByDataView(dataViewId);
    return dbFields.stream().filter(e -> e.getAuthType() != null).collect(Collectors.toSet());
  }

  @Override
  public DataViewFieldEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewFieldEntityRepository.findDetailsById(id);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewFieldEntity> op = this.dataViewFieldEntityRepository.findById(id);
    op.ifPresent(dataViewFieldEntity -> this.dataViewFieldEntityRepository.deleteById(id));
  }

  @Override
  public DataViewFieldEntity findByCode(String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    return dataViewFieldEntityRepository.findByCode(code);
  }

  @Transactional
  @Override
  public void create(Set<DataViewFieldEntity> fields, DataViewEntity dataViewEntity) {
    Validate.notEmpty(fields, "视图结构的输出字段信息不能为空，请检查!!");
    fields.forEach(
            e -> {
              this.createValidation(e);
              e.setCode(UUID.randomUUID().toString());
              e.setDataView(dataViewEntity);
            });
    this.dataViewFieldEntityRepository.saveAll(fields);
  }

  @Override
  @Transactional
  public Set<DataViewFieldEntity> update(DataViewEntity dataView, Set<DataViewFieldEntity> fields) {
    Validate.notNull(dataView, "数据视图不能为空！！！");
    Validate.notBlank(dataView.getId(), "数据视图ID不能为空！！！");
    this.updateValidation(fields);
    Map<String, DataViewFieldEntity> fieldsMap = fields.stream().collect(Collectors.toMap(DataViewFieldEntity::getFieldName, v -> v));
    List<DataViewFieldEntity> oldFields = dataViewFieldEntityRepository.findByDataView(dataView.getId());
    Set<DataViewFieldEntity> createFields = new HashSet<>();
    Set<DataViewFieldEntity> updateFields = new HashSet<>();
    Set<DataViewFieldEntity> deleteFields = new HashSet<>();
    nebulaToolkitService.collectionDiscrepancy(fields, oldFields, DataViewFieldEntity::getFieldName, deleteFields, updateFields, createFields);
    // 新增
    for (DataViewFieldEntity field : createFields) {
      field.setDataView(dataView);
      // 新增的时候增加UUID
      field.setCode(UUID.randomUUID().toString());
      dataViewFieldEntityRepository.save(field);
    }
    // 更新
    for (DataViewFieldEntity field : updateFields) {
      DataViewFieldEntity newField = fieldsMap.get(field.getFieldName());
      field.setDisplayName(newField.getDisplayName());
      field.setFieldType(newField.getFieldType());
      field.setPhysical(newField.getPhysical());
      field.setTargetField(newField.getTargetField());
      field.setTargetTable(newField.getTargetTable());
      field.setTstatus(newField.getTstatus());
      dataViewFieldEntityRepository.save(field);
    }
    // 删除
    for (DataViewFieldEntity field : deleteFields) {
      dataViewFilterService.deleteByField(field.getId());
      dataViewAuthHorizontalService.deleteByField(field.getId());
      dataViewAuthVerticalService.deleteByField(field.getId());
      dataViewFieldEntityRepository.delete(field);
    }
    createFields.addAll(updateFields);
    return createFields;
  }

  /**
   * 验证更新数据
   * @param fields
   */
  private void updateValidation(Set<DataViewFieldEntity> fields) {
    Validate.notEmpty(fields, "数据视图字段不能为空");
    for (DataViewFieldEntity field : fields) {
      // 基础信息判断，基本属性，需要满足not null 且 updateable == true
      Validate.notBlank(field.getFieldName(), "字段名（英文）不能为空！");
      Validate.notBlank(field.getFieldType(), "字段类型不能为空！");
      Validate.notNull(field.getSortIndex(), "字段排序不能为空！");
      Validate.notNull(field.getPhysical(), "是否物理字段不能为空！");
      Validate.notBlank(field.getTargetField(), "物理字段不能为空！");
      Validate.notBlank(field.getDisplayName(), "显示中文名称不能为空！");
      Validate.notNull(field.getTstatus(), "是否有效不能为空！");
      // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
      Validate.isTrue(field.getFieldName() == null || field.getFieldName().length() < 64 , "字段名（英文）,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(field.getDisplayName() == null || field.getDisplayName().length() < 64 , "字段显示中文名,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(field.getFieldType() == null || field.getFieldType().length() < 64 , "字段类型,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(field.getSchemaName() == null || field.getSchemaName().length() < 64 , "数据分库,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(field.getTargetTable() == null || field.getTargetTable().length() < 64 , "物理数据表,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(field.getTargetField() == null || field.getTargetField().length() < 64 , "物理字段,填入值超过了限定长度(64)，请检查!");
    }
  }

} 
