package com.bizunited.platform.core.service.dataview.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.enums.DataSourceTypeEnum;
import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.repository.dataview.analysis.MysqlAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.OracleSqlAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.SQLAuthVerticalAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.SQLPresetValueAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.SqlAnalysis;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthInterceptorService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.bizunited.platform.core.service.dataview.DataViewFieldService;
import com.bizunited.platform.core.service.dataview.DataViewFilterService;
import com.bizunited.platform.core.service.dataview.DataViewGroupService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.dataview.DataViewSystemService;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import com.bizunited.platform.core.vo.dataview.DataviewExportExcelFieldVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.AsyncResult;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import javax.transaction.Transactional;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

/**
 * DataViewEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewEntityServiceImpl")
public class DataViewServiceImpl implements DataViewService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewServiceImpl.class);
  private static final String ERROR_MESS_SQL = "原始SQL不能为空！";
  /**
   * 默认时间格式
   */
  private static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
  /**
   * 超时时间（秒）
   */
  private static final Integer TIME_OUT = 10;
  /**
   * 默认列宽
   */
  private static final int DEFAULT_COLUMN_WIDTH = 5000;

  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DataSourceProperties dataSourceProperties;
  @Autowired
  private DataViewRepository dataViewEntityRepository;
  @Autowired
  private DataViewFieldService dataViewFieldService;
  @Autowired
  private DataViewFilterService dataViewFilterService;
  @Autowired
  private DataViewSystemService dataViewSystemService;
  @Autowired
  private DataSourceService dataSourceService;
  @Autowired
  private DataViewGroupService dataViewGroupService;
  @Autowired
  private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  @Autowired
  private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Lazy
  @Autowired
  private DataViewAuthInterceptorService dataViewAuthInterceptorService;
  @Autowired
  private SQLPresetValueAnalysis sqlPresetValueAnalysis;
  private DataViewService proxySelf;

  @PostConstruct
  public void setProxySelf() {
    this.proxySelf = applicationContext.getBean(DataViewService.class);
  }

  /**
   * 在创建一个新的DataViewEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewEntity.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    dataViewEntity.setId(null);
    Validate.notBlank(dataViewEntity.getCode(), "数据视图编号不能为空！");
    DataViewEntity currentEntity = dataViewEntityRepository.findByCode(dataViewEntity.getCode());
    Validate.isTrue(null==currentEntity,"该CODE视图已存在，请改变编码！");
    Validate.notBlank(dataViewEntity.getName(), "视图中文名不能为空！");
    Validate.isTrue(!ChineseCharUtil.hasChinese(dataViewEntity.getCode()),"视图编码不能含有中文，请检查！");
    Validate.notBlank(dataViewEntity.getSourceSql(), ERROR_MESS_SQL);
    Validate.notNull(dataViewEntity.getTstatus(), "视图状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(dataViewEntity.getCode() == null || dataViewEntity.getCode().length() < 64 , "数据视图编号,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getName() == null || dataViewEntity.getName().length() < 64 , "视图中文名,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getSourceSql() == null || dataViewEntity.getSourceSql().length() < 4096 , "原始SQL,填入值超过了限定长度(4096)，请检查!");
  }

  @Transactional
  @Override
  public DataViewEntity create(DataViewEntity dataViewEntity , boolean ignoreValidate) {
    if(!ignoreValidate) {
      this.createValidation(dataViewEntity);
    }

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================
    DataSourceEntity dataSource = dataViewEntity.getDataSource();
    if(dataSource != null && StringUtils.isNotBlank(dataSource.getId())) {
      DataSourceEntity currentDataSource = dataSourceService.findById(dataSource.getId());
      Validate.notNull(currentDataSource,"未找到该数据源!");
    }

    DataViewGroupEntity dataViewGroup = dataViewEntity.getDataViewGroup();
    Validate.notNull(dataViewGroup, "视图分组信息不能为空，请检查!!");
    Validate.notBlank(dataViewGroup.getId(), "视图所选分组id不能为空，请检查!!");

    this.dataViewEntityRepository.save(dataViewEntity);

    Set<DataViewFieldEntity> fields = dataViewEntity.getFields();
    this.dataViewFieldService.create(fields, dataViewEntity);

    Set<DataViewSystemEntity> systemFilters = dataViewEntity.getSystemFilters();
    this.dataViewSystemService.create(systemFilters, dataViewEntity);

    Set<DataViewFilterEntity> filters = dataViewEntity.getFilters();
    this.dataViewFilterService.create(filters, fields, dataViewEntity);

    dataViewEntity.setFields(fields);
    dataViewEntity.setFilters(filters);
    dataViewEntity.setSystemFilters(systemFilters);
    return dataViewEntity;
  }

  @Override
  @Transactional
  public DataViewEntity update(DataViewEntity dataView) {
    this.updateValidation(dataView);
    DataViewEntity dbDataView = dataViewEntityRepository.findById(dataView.getId()).orElse(null);
    Validate.notNull(dbDataView, "未找到数据视图对象");
    dbDataView.setCode(dataView.getCode());
    dbDataView.setName(dataView.getName());
    dbDataView.setSourceSql(dataView.getSourceSql());
    dbDataView.setTstatus(dataView.getTstatus());
    dataViewEntityRepository.save(dbDataView);
    Set<DataViewFieldEntity> fields = dataViewFieldService.update(dbDataView, dataView.getFields());
    dbDataView.setFields(fields);
    Set<DataViewFilterEntity> filters = dataViewFilterService.update(dbDataView, dataView.getFilters());
    dbDataView.setFilters(filters);
    Set<DataViewSystemEntity> systemFilters = dataViewSystemService.update(dbDataView, dataView.getSystemFilters());
    dbDataView.setSystemFilters(systemFilters);
    return dbDataView;
  }

  /**
   * 在更新一个已有的DataViewEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  @Override
  public void updateValidation(DataViewEntity dataViewEntity) {
    Validate.isTrue(!StringUtils.isBlank(dataViewEntity.getId()), "修改信息时，当前信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewEntity.getCode(), "数据视图编号不能为空！");
    Validate.notBlank(dataViewEntity.getName(), "视图中文名不能为空！");
    Validate.notBlank(dataViewEntity.getSourceSql(), ERROR_MESS_SQL);
    Validate.notNull(dataViewEntity.getTstatus(), "视图状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewEntity.getCode() == null || dataViewEntity.getCode().length() < 64 , "数据视图编号,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getName() == null || dataViewEntity.getName().length() < 64 , "视图中文名,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getSourceSql() == null || dataViewEntity.getSourceSql().length() < 4096 , "原始SQL,填入值超过了限定长度(4096)，请检查!");
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewEntity> op = this.dataViewEntityRepository.findById(id);
    op.ifPresent(dataViewEntity -> this.dataViewEntityRepository.delete(dataViewEntity));
  }

  @Async("dynamicExecutor")
  @Override
  public Future<?> executeTop1(String dataSourceCode, DataViewEntity dataView, InvokeParams invokeParams,Principal principal) {
    Future<?> future;
    try {
      SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,dataSourceCode,principal,invokeParams,null);
      JSONArray jsonArr = dataViewEntityRepository.executeSQLTop1(dataSourceCode, sqlAnalysis);
      future = new AsyncResult<>(jsonArr);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(),e);
      future = new AsyncResult<>(e);
    }
    return future;
  }


  @Override
  public JSONArray executeTop1(DataViewEntity dataView, InvokeParams invokeParams,Principal principal) {
    SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,null,principal,invokeParams,PageRequest.of(0,1));
    return dataViewEntityRepository.executeSQLTop1(sqlAnalysis);
  }


  @Async("dynamicExecutor")
  @Override
  public Future<?> execute(String dataSourceCode, DataViewEntity dataView, InvokeParams invokeParams,Principal principal) {
    Future<?> future;
    try {
      SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,dataSourceCode,principal,invokeParams,null);
      JSONArray jsonArr = dataViewEntityRepository.executeSQL(dataSourceCode, sqlAnalysis , dataView);
      future = new AsyncResult<>(jsonArr);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(),e);
      future = new AsyncResult<>(e);
    }
    return future;
  }


  @Async("dynamicExecutor")
  @Override
  public Future<?> execute(String dataSourceCode, DataViewEntity dataView, InvokeParams invokeParams, Pageable pageable,Principal principal) {
    Future<?> future;
    try {
      SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,dataSourceCode,principal,invokeParams,pageable);
      Page<JSONObject> page = dataViewEntityRepository.executeSQLPageAble(dataSourceCode, sqlAnalysis , pageable);
      future = new AsyncResult<>(page);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(),e);
      future = new AsyncResult<>(e);
    }
    return future;
  }


  @Override
  public Page<JSONObject> execute(DataViewEntity dataView, InvokeParams invokeParams, Pageable pageable,Principal principal) {
    SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,null,principal,invokeParams,pageable);
    return this.dataViewEntityRepository.executeSQLPageAble(sqlAnalysis);
  }

  @Override
  public JSONArray execute(DataViewEntity dataView, InvokeParams invokeParams,Principal principal) {
    SqlAnalysis sqlAnalysis = this.dataViewLogic(dataView,null,principal,invokeParams,null);
    return this.dataViewEntityRepository.executeSQL(sqlAnalysis);
  }
  @Override
  public Set<String> findFieldsByDataViewCode(String dataViewCode, Principal principal) {
    DataViewEntity dataView = dataViewEntityRepository.findByCode(dataViewCode);
    Validate.notNull(dataView, "没有查到指定的数据视图信息");
    Set<DataViewAuthVerticalEntity> auths = dataViewAuthVerticalService.findDetailsByDataViewCode(dataViewCode);
    Map<Integer, Object> verticalPresets = sqlPresetValueAnalysis.authVerticalPresetValues(auths, principal);
    return SQLAuthVerticalAnalysis.buildFields(auths, verticalPresets);
  }

  @Override
  public byte[] exportExcel(String dataSourceCode, DataViewEntity dataView, InvokeParams inputParams, Principal principal, List<DataviewExportExcelFieldVo> excelFields) {
    if(CollectionUtils.isEmpty(excelFields)) {
      return new byte[0];
    }
    JSONArray jsonArray;
    if(StringUtils.isNotBlank(dataSourceCode)) {
      Future<?> future = proxySelf.execute(dataSourceCode, dataView, inputParams, principal);
      try {
        Object result = future.get(TIME_OUT, TimeUnit.SECONDS);
        jsonArray = (JSONArray) result;
      } catch (InterruptedException | ExecutionException | TimeoutException e) {
        throw new IllegalArgumentException("查询数据错误", e);
      }
    } else {
      jsonArray = this.execute(dataView, inputParams, principal);
    }
    Workbook workbook = this.buildWorkbook(jsonArray, excelFields);
    if(workbook == null) {
      return new byte[0];
    }
    byte[] bytes = new byte[]{};
    try(ByteArrayOutputStream os = new ByteArrayOutputStream();) {
      workbook.write(os);
      bytes = os.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
    return bytes;
  }

  /**
   * 构建excel
   * @param jsonArray
   * @param excelFields
   * @return
   */
  private Workbook buildWorkbook(JSONArray jsonArray, List<DataviewExportExcelFieldVo> excelFields) {
    Workbook workbook = new XSSFWorkbook();
    Sheet sheet = workbook.createSheet();
    List<DataviewExportExcelFieldVo> sortedExcelFields = excelFields.stream().sorted(Comparator.comparing(DataviewExportExcelFieldVo::getSortIndex)).collect(Collectors.toList());
    if(CollectionUtils.isEmpty(jsonArray)) {
      this.initTitle(sheet, sortedExcelFields);
      return workbook;
    }
    this.initColumnWidth(sheet, jsonArray.size());
    this.initTitle(sheet, sortedExcelFields);
    this.initRows(sheet, jsonArray, sortedExcelFields);
    return workbook;
  }

  /**
   * 初始化行
   * @param sheet
   * @param version
   * @param project
   */
  private void initRows(Sheet sheet, JSONArray jsonArray, List<DataviewExportExcelFieldVo> fields){
    for (int i = 0; i < jsonArray.size(); i++) {
      JSONObject jsonObject = jsonArray.getJSONObject(i);
      Row row = sheet.createRow(i + 1);
      this.initRow(row, jsonObject, fields);
    }
  }

  /**
   * 初始化行
   * @param row
   * @param version
   * @param project
   */
  private void initRow(Row row, JSONObject jsonObject, List<DataviewExportExcelFieldVo> fields) {
    for (int i = 0; i < fields.size(); i++) {
      DataviewExportExcelFieldVo field = fields.get(i);
      Cell cell = row.createCell(i);
      this.setCellValue(cell, jsonObject, field);
    }
  }

  /**
   * 获取数据字典显示
   * @param dicts
   * @param obj
   * @return
   */
  private String getCellDictKey(Map<String, String> dicts, Object obj) {
    String[] values = obj.toString().split(",");
    List<String> keys = new ArrayList<>();
    Map<String, String> vDicts = dicts.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (a, b) -> b, () -> new HashMap<>(16)));
    for (String value : values) {
      String key = vDicts.get(value);
      if(key == null) {
        keys.add(value);
      } else {
        keys.add(key);
      }
    }
    return StringUtils.join(keys, "、");
  }

  /**
   * 设置单元格的值
   * @param cell
   * @param jsonObject
   * @param field
   */
  private void setCellValue(Cell cell, JSONObject jsonObject, DataviewExportExcelFieldVo field) {
    Object obj = jsonObject.get(field.getFieldName());
    if(obj == null) {
      cell.setCellValue("");
      return;
    }
    Map<String, String> dicts = field.getDicts();
    if(dicts != null && dicts.size() > 0) {
      String key = this.getCellDictKey(dicts, obj);
      cell.setCellValue(key);
      return;
    }
    if(obj instanceof Integer) {
      cell.setCellValue((Integer)obj);
    } else if (obj instanceof Double) {
      cell.setCellValue((Double)obj);
    } else if (obj instanceof Float) {
      cell.setCellValue((Float)obj);
    } else if (obj instanceof String) {
      cell.setCellValue((String) obj);
    } else if (obj instanceof Date) {
      String format = DEFAULT_DATE_FORMAT;
      if(StringUtils.isNotBlank(field.getFormat())) {
        format = field.getFormat();
      }
      SimpleDateFormat dateFormat = new SimpleDateFormat(format);
      cell.setCellValue(dateFormat.format((Date)obj));
    } else if (obj instanceof Long) {
      Long data = (Long)obj;
      if(Objects.equals("java.sql.Timestamp", field.getFieldType())
        || Objects.equals("java.util.Date", field.getFieldType())) {
        String format = DEFAULT_DATE_FORMAT;
        if(StringUtils.isNotBlank(field.getFormat())) {
          format = field.getFormat();
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        cell.setCellValue(dateFormat.format(data));
      } else {
        cell.setCellValue((Long)obj);
      }
    } else if (obj instanceof Boolean) {
      cell.setCellValue((Boolean) obj);
    } else if (obj instanceof BigDecimal) {
      cell.setCellValue(((BigDecimal)obj).toString());
    } else if (obj instanceof Byte) {
      cell.setCellValue((Byte)obj);
    } else if (obj instanceof Short) {
      cell.setCellValue((Short)obj);
    } else {
      cell.setCellValue(obj.toString());
    }
  }

  /**
   * 初始化列宽
   * @param sheet
   */
  private void initColumnWidth(Sheet sheet, int columnCount) {
    for (int i = 0; i < columnCount; i++) {
      sheet.setColumnWidth(i, DEFAULT_COLUMN_WIDTH);
    }
  }

  /**
   * 初始化标题
   * @param sheet
   */
  private void initTitle(Sheet sheet, List<DataviewExportExcelFieldVo> fields) {
    Row row = sheet.createRow(0);
    for (int i = 0; i < fields.size(); i++) {
      Cell cell = row.createCell(i);
      cell.setCellValue(fields.get(i).getDisplayName());
    }
  }


  /**
   * 数据视图所涉及的公共部分
   * @param dataViewParam
   * @param dataSourceCode
   * @param principal
   * @param invokeParams
   * @param pageable
   */
  private SqlAnalysis dataViewLogic(DataViewEntity dataViewParam,String dataSourceCode,Principal principal,InvokeParams invokeParams,Pageable pageable){
    Validate.notNull(dataViewParam,"传入的数据视图不能为空，请检查!!");
    DataViewEntity dataView;
    String dataSourceType;
    //简单验证
    if(StringUtils.isBlank(dataSourceCode)){
      dataView = this.validateMainInfo(dataViewParam.getCode());
      DataSourceTypeEnum dataSourceTypeEnum = DataSourceTypeEnum.valueOfDriver(dataSourceProperties.getDriverClassName());
      Validate.notNull(dataSourceTypeEnum, String.format("不支持的数据库驱动类：%s", dataSourceProperties.getDriverClassName()));
      dataSourceType = dataSourceTypeEnum.getDatabase();
    }else{
      dataView = this.validateThirdInfo(dataSourceCode, dataViewParam.getCode());
      DataSourceEntity dataSource = dataSourceService.findDetailsByCode(dataSourceCode);
      Validate.notNull(dataSource, "根据传入的code值，未能获取到第三方数据源信息，请检查!!");
      dataSourceType = dataSource.getType();
    }
    Validate.notBlank(dataSourceType, "未知数据类型，请检查");
    Validate.notNull(dataView,"根据传入的视图Id，未能查询到视图相关信息，请检查!!");
    //横向权限
    Set<DataViewAuthHorizontalEntity> authHorizontals = dataViewAuthHorizontalService.findDetailsByDataViewCode(dataView.getCode());
    //竖向权限
    Set<DataViewAuthVerticalEntity> authVerticals = dataViewAuthVerticalService.findDetailsByDataViewCode(dataView.getCode());
    DataViewAuthInterceptorEntity interceptor = dataViewAuthInterceptorService.findByDataView(dataView.getCode());
    //系统参数涉及的预制值
    Map<String,Object> sysPresets = sqlPresetValueAnalysis.systemPresetValues(dataView.getSystemFilters(),principal);
    //竖向权限涉及的预制值
    Map<Integer,Object> verticalPresets = sqlPresetValueAnalysis.authVerticalPresetValues(authVerticals,principal);
    //横向权限涉及的预制值
    Map<String,Object> authHorizontalPresets = sqlPresetValueAnalysis.authHorizontalPresetValues(authHorizontals,principal);
    //所有字段信息
    Set<DataViewFieldEntity> allFileds = dataView.getFields();
    //SQL解析器构造函数
    SqlAnalysis sqlAnalysis;
    if(DataSourceTypeEnum.MYSQL.getDatabase().equalsIgnoreCase(dataSourceType)) {
      sqlAnalysis = new MysqlAnalysis(dataView,dataView.getFilters(),dataView.getSystemFilters(),
          authHorizontals, authVerticals,invokeParams,pageable,allFileds, interceptor);
    } else if (DataSourceTypeEnum.ORACLE.getDatabase().equalsIgnoreCase(dataSourceType)) {
      sqlAnalysis = new OracleSqlAnalysis(dataView,dataView.getFilters(),dataView.getSystemFilters(),
          authHorizontals, authVerticals,invokeParams,pageable,allFileds, interceptor);
    } else {
      throw new UnsupportedOperationException(String.format("不支持的数据源:%s", dataSourceType));
    }
    //设置预制值的值信息到SQL解析器中临时存储，留备后续使用
    sqlAnalysis.setPresets(sysPresets, authHorizontalPresets, verticalPresets);
    return sqlAnalysis;
  }



  /**
   * 使用一个指定的第三方数据源，进行数据视图中设定的最原始SQL的预执行。预执行后，将向调用者返回执行后的SQL输出结构、以及要求填写的系统参数信息</br>
   * 注意，该方法只会对原始SQL进行执行（自动替换原始SQL中的系统参数信息），并不会将预执行后的结构做持久化保存；要进行持久化保存，请使用本服务中的creat方法
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 这个对象包括了数据视图最原始的SQL设定信息。
   */
  @Override
  public DataViewEntity executeResource(String dataSourceCode, DataViewEntity dataView) {
    this.checkSQL(dataView);
    DataViewEntity dataViewEntity = dataViewEntityRepository.executeResource(dataSourceCode, dataView);
    return this.handleDataView(dataViewEntity);
  }

  /**
   * 处理数据视图
   * @param dataView
   * @return
   */
  private DataViewEntity handleDataView(DataViewEntity dataView) {
    if(dataView == null) {
      return null;
    }
    if (dataView.getFilters() == null){
      dataView.setFilters(Sets.newHashSet());
    }
    return dataView;
  }


  /**
   * 使用本本进程的主数据源进行数据视图中设定的最原始SQL的保存。其它操作特性请参见:</br>
   * executeResource(String dataSourceCode , DataViewEntity dataView)方法
   */
  @Override
  public DataViewEntity executeResource(DataViewEntity dataView) {
    this.checkSQL(dataView);
    DataViewEntity dataViewEntity = dataViewEntityRepository.executeResource(dataView);
    return this.handleDataView(dataViewEntity);
  }


  private void checkSQL(DataViewEntity dataView){
    String[] notAllowed = new String[]{"delete","insert","drop","create","alter",";"};
    List<String> notAllowedList = Arrays.asList(notAllowed);
    Validate.notNull(dataView,"数据视图不能为空！");
    Validate.notBlank(dataView.getSourceSql(), ERROR_MESS_SQL);
    String sql = dataView.getSourceSql();
    List<String> words = Arrays.asList(StringUtils.split(sql," "));
    if(!CollectionUtils.isEmpty(words)){
      Validate.isTrue("select".equals(StringUtils.lowerCase(words.get(0))),"查询语句只允许以'select'起始的查询语句");
      words.forEach(o -> Validate.isTrue(!notAllowedList.contains(o),String.format("语句中不能含有字符%s",o)));
      //最后一个词不能跟';'
      Validate.isTrue(!words.get(words.size()-1).contains(";"),"最后一个词后不能接分号");
    }
  }



  /**
   * 简单验证第三方数据源与视图
   * @param dataSourceCode
   * @param dataViewCode
   */
  private DataViewEntity validateThirdInfo(String dataSourceCode , String dataViewCode) {
    DataSourceEntity dataSourceEntity = dataSourceService.findDetailsByCode(dataSourceCode);
    Validate.notNull(dataSourceEntity, "根据传入的code值，未能获取到第三方数据源信息，请检查!!");
    Validate.isTrue(dataSourceEntity.getTstatus() == 1, "数据源状态是禁用状态，请检查!!");
    Validate.notBlank(dataViewCode, "传入的视图信息code不能为空，请检查!!");
    DataViewEntity dataViewEntity = dataViewEntityRepository.findDetailsByCode(dataViewCode);
    Validate.notNull(dataViewEntity, "根据视图code信息未能获取到视图信息，请检查!!");
    Validate.isTrue(dataViewEntity.getTstatus() == 1, "视图状态是禁用状态，请检查!!");
    Validate.notNull(dataViewEntity.getDataSource(), "当前数据视图信息有误，未能获取到第三方数据源信息，请检查!!");
    Validate.isTrue(StringUtils.equals(dataSourceCode, dataViewEntity.getDataSource().getCode()),"传入的数据源code值%s与当前数据视图不匹配，请检查!!");
    return dataViewEntity;
  }

  /**
   * 简单验证主数据源视图信息
   * @param dataViewCode
   */
  private DataViewEntity validateMainInfo(String dataViewCode) {
    Validate.notBlank(dataViewCode, "传入的视图信息code不能为空，请检查!!");
    DataViewEntity dataViewEntity = dataViewEntityRepository.findByCode(dataViewCode);
    Validate.notNull(dataViewEntity, "根据视图code信息未能获取到视图信息，请检查!!");
    Validate.isTrue(dataViewEntity.getTstatus() == 1, "视图状态是禁用状态，请检查!!");
    Validate.isTrue(dataViewEntity.getDataSource() == null,"传入的数据视图不是主数据源类型，请检查!!");
    return dataViewEntity;
  }

  @Override
  public JSONArray findByDataSourceCode(String dataSourceCode) {
    //1 先查询统计视图相关信息
    List<Object[]> jsonList;
    if(StringUtils.isBlank(dataSourceCode)) {
      jsonList = dataViewEntityRepository.findByDataSourceCode();
    }else {
      jsonList = dataViewEntityRepository.findByDataSourceCode(dataSourceCode);
    }
    JSONArray jsonViewArr = new JSONArray();
    if(!CollectionUtils.isEmpty(jsonList)) {
      jsonList.forEach(e -> {
        JSONObject jsonObject = new JSONObject();
        //这里的下标值，是按照查询语句的列的顺序来的
        jsonObject.put("id", e[0].toString());
        jsonObject.put("code", e[1].toString());
        jsonObject.put("title", e[2].toString());
        jsonObject.put("datasourcegroup", e[3].toString());
        jsonObject.put("outCount", e[4] == null ? 0 : ((BigInteger)e[4]).intValue());
        jsonObject.put("displayCount", e[5] == null ? 0 : ((BigInteger)e[5]).intValue());
        jsonViewArr.add(jsonObject);
      });
    }

    //2 查询所有分组信息
    JSONArray groupArr = new JSONArray();
    List<DataViewGroupEntity> groupList = dataViewGroupService.findByDataSourceCode(dataSourceCode);
    Validate.notNull(groupList, "根据数据源code值未能获取到视图分组信息，请检查!!");
    groupList.forEach(e -> {
      JSONObject jsonObject = new JSONObject();
      JSONArray viewArr = new JSONArray();
      jsonObject.put("id", e.getId());
      jsonObject.put("title", e.getGroupName());
      jsonObject.put("targetTable", e.getTargetTable());
      if(!jsonViewArr.isEmpty()) {
        jsonViewArr.stream().filter(view -> StringUtils.equals(e.getId(), ((JSONObject)view).getString("datasourcegroup"))).forEach(viewArr::add);
      }
      jsonObject.put("children", viewArr);
      groupArr.add(jsonObject);
    });
    return groupArr;
  }

  /**
   * 根据设置的视图唯一键标识，和前端传入的唯一键标识的值，重新获取视图的数据信息
   * 其中前端传入的primaryKey必须有值，但是primaryValue可能会没有值，如果primaryValue没有值，则直接返回null
   * primaryKey有多个的情况，则直接逗号分隔，同时primaryValue就是数组
   * 注：如果当前视图存在预制值，那么该方法可能会有问题，后期待完善
   */
  @Override
  public JSONArray findByViewAndDataKey(DataViewEntity dataViewEntity, InvokeParams invokeParams,Principal principal,String primaryKey,JSONArray primaryValue) {
    //1 简单的基础验证
    Validate.notBlank(primaryKey, "传入的视图primaryKey唯一性键标识不能为空!!");
    if(primaryValue == null) {
      return null;
    }
    Validate.notNull(dataViewEntity, "传入的视图参数不能为空!!");
    Validate.notBlank(dataViewEntity.getCode(), "根据传入的视图参数，未能获取视图CODE信息，请检查!!");
    // 确定存在性，并获取已经存在的sql信息
    DataViewEntity currrentDataView = dataViewEntityRepository.findByCode(dataViewEntity.getCode());
    Validate.notNull(currrentDataView, "根据视图CODE，未能获取到视图信息，请检查!!");
    String sql = currrentDataView.getSourceSql();
    Validate.notBlank(sql , "未能获取到视图SQL信息，请检查!!");
    
    Map<String,Object> sysPresets = null;
    Set<DataViewSystemEntity> systemEntities = currrentDataView.getSystemFilters();
    if(CollectionUtils.isEmpty(systemEntities)){
      sysPresets = sqlPresetValueAnalysis.systemPresetValues(systemEntities,principal);
    }
    return dataViewEntityRepository.executeSQLEcho(currrentDataView, invokeParams, sysPresets, primaryKey, primaryValue);
  }

  @Override
  public List<DataViewEntity> findByDataSource(String dataSourceId) {
    if(StringUtils.isBlank(dataSourceId)) {
      return Collections.emptyList();
    }
    return this.dataViewEntityRepository.findByDataSource(dataSourceId);
  }

  @Override
  public List<DataViewEntity> findByDataViewGroup(String dataViewGroupId) {
    if(StringUtils.isBlank(dataViewGroupId)) {
      return Collections.emptyList();
    }
    return this.dataViewEntityRepository.findByDataViewGroup(dataViewGroupId);
  }

  @Override
  public DataViewEntity findByFields(String fieldsId) {
    if(StringUtils.isBlank(fieldsId)) {
      return null;
    }
    return this.dataViewEntityRepository.findByFields(fieldsId);
  }

  @Override
  public DataViewEntity findByFilters(String filtersId) {
    if(StringUtils.isBlank(filtersId)) {
      return null;
    }
    return this.dataViewEntityRepository.findByFilters(filtersId);
  }

  @Override
  public DataViewEntity findBySystemFilters(String systemFiltersId) {
    if(StringUtils.isBlank(systemFiltersId)) {
      return null;
    }
    return this.dataViewEntityRepository.findBySystemFilters(systemFiltersId);
  }

  @Override
  public DataViewEntity findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.dataViewEntityRepository.findDetailsByCode(code);
  }

  @Override
  public DataViewEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.dataViewEntityRepository.findByCode(code);
  }

  @Override
  public Set<DataViewEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return Sets.newHashSet();
    }
    
    return this.dataViewEntityRepository.findDetailsByIds(ids);
  }

  @Override
  public DataViewEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    
    Optional<DataViewEntity> op = this.dataViewEntityRepository.findById(id);
    return op.orElse(null);
  }
  
  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.dataViewEntityRepository.countByIds(ids);
  }
}