package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dataview.DataViewSystemRepository;
import com.bizunited.platform.core.service.dataview.DataViewSystemService;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * DataViewSystemEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewSystemEntityServiceImpl")
public class DataViewSystemServiceImpl implements DataViewSystemService {
  @Autowired
  private DataViewSystemRepository dataViewSystemEntityRepository;
  /**
   * 在创建一个新的DataViewSystemEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewSystemEntity dataViewSystemEntity) {
    Validate.notNull(dataViewSystemEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewSystemEntity.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    dataViewSystemEntity.setId(null);
    Validate.notBlank(dataViewSystemEntity.getParamName(), "显示中文信息不能为空！");
    Validate.notNull(dataViewSystemEntity.getParamType(), "参数来源类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewSystemEntity.getParamName() == null || dataViewSystemEntity.getParamName().length() < 64 , "显示中文信息,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewSystemEntity.getParamKey() == null || dataViewSystemEntity.getParamKey().length() < 64 , "预制值标识,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewSystemEntity.getParamValue() == null || dataViewSystemEntity.getParamValue().length() < 256 , "具体的固定值,填入值超过了限定长度(256)，请检查!");
  }
  @Transactional
  @Override
  public DataViewSystemEntity create(DataViewSystemEntity dataViewSystemEntity , boolean ignoreValidate) { 
    if(!ignoreValidate) {
      this.createValidation(dataViewSystemEntity);
    }
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.dataViewSystemEntityRepository.save(dataViewSystemEntity);
    
    return dataViewSystemEntity;
  }
  @Transactional
  @Override
  public DataViewSystemEntity update(DataViewSystemEntity dataViewSystemEntity) { 
    this.updateValidation(dataViewSystemEntity);
    // ===================基本信息
    String currentId = dataViewSystemEntity.getId();
    Optional<DataViewSystemEntity> op = this.dataViewSystemEntityRepository.findById(currentId);
    DataViewSystemEntity currentDataViewSystemEntity = op.orElse(null);
    Validate.notNull(currentDataViewSystemEntity ,"未发现指定的原始模型对象信");

    // 开始重新赋值，首先是一般对象，updateable = true形式的
    currentDataViewSystemEntity.setParamName(dataViewSystemEntity.getParamName());
    currentDataViewSystemEntity.setParamType(dataViewSystemEntity.getParamType());
    currentDataViewSystemEntity.setParamKey(dataViewSystemEntity.getParamKey());
    currentDataViewSystemEntity.setParamValue(dataViewSystemEntity.getParamValue());
    currentDataViewSystemEntity.setParamSourceType(dataViewSystemEntity.getParamSourceType());
    
    this.dataViewSystemEntityRepository.save(currentDataViewSystemEntity);
    // =============
    // ManyToMany 结构的关联关系，需要开发人员自行处理
    // =============
    return currentDataViewSystemEntity;
  }
  /**
   * 在更新一个已有的DataViewSystemEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewSystemEntity dataViewSystemEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dataViewSystemEntity.getId()), "修改信息时，当前信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewSystemEntity.getParamName(), "显示中文信息不能为空！");
    Validate.notNull(dataViewSystemEntity.getParamType(), "参数来源类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewSystemEntity.getParamName() == null || dataViewSystemEntity.getParamName().length() < 64 , "显示中文信息,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewSystemEntity.getParamKey() == null || dataViewSystemEntity.getParamKey().length() < 64 , "预制值标识,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewSystemEntity.getParamValue() == null || dataViewSystemEntity.getParamValue().length() < 256 , "具体的固定值,填入值超过了限定长度(256)，请检查!");
  } 
  @Override
  public List<DataViewSystemEntity> findByDataView(String dataViewId) { 
    if(StringUtils.isBlank(dataViewId)) { 
      return Lists.newArrayList();
    }
    return this.dataViewSystemEntityRepository.findByDataView(dataViewId);
  }
  @Override
  public DataViewSystemEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewSystemEntityRepository.findDetailsById(id);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewSystemEntity> op = this.dataViewSystemEntityRepository.findById(id);
    op.ifPresent(dataViewSystemEntity -> this.dataViewSystemEntityRepository.deleteById(id));
  }

  @Override
  @Transactional
  public void create(Set<DataViewSystemEntity> systemFilters, DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity, "数据视图为空，请检查！");
    if(!CollectionUtils.isEmpty(systemFilters)) {
      systemFilters.forEach( e -> {
        this.createValidation(e);
        e.setDataView(dataViewEntity);
      });
    }
    if (!CollectionUtils.isEmpty(systemFilters)){
      this.dataViewSystemEntityRepository.saveAll(systemFilters);
    }
  }

  @Override
  @Transactional
  public Set<DataViewSystemEntity> update(DataViewEntity dataView, Set<DataViewSystemEntity> filters) {
    Validate.notNull(dataView, "数据视图不能为空！");
    Validate.notBlank(dataView.getId(), "数据视图ID不能为空！");
    this.updateValidation(filters);
    List<DataViewSystemEntity> dbFilters = dataViewSystemEntityRepository.findByDataView(dataView.getId());
    dataViewSystemEntityRepository.deleteAll(dbFilters);
    for (DataViewSystemEntity filter : filters) {
      filter.setDataView(dataView);
    }
    dataViewSystemEntityRepository.saveAll(filters);
    return filters;
  }

  private void updateValidation(Set<DataViewSystemEntity> filters) {
    if(CollectionUtils.isEmpty(filters)) {
      return;
    }
    for (DataViewSystemEntity filter : filters) {
      // 基础信息判断，基本属性，需要满足not null 且 updateable == true
      Validate.notBlank(filter.getParamName(), "显示中文信息不能为空！");
      Validate.notNull(filter.getParamType(), "参数来源类型不能为空！");
      // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
      Validate.isTrue(filter.getParamName() == null || filter.getParamName().length() < 64 , "显示中文信息,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(filter.getParamKey() == null || filter.getParamKey().length() < 64 , "预制值标识,填入值超过了限定长度(64)，请检查!");
      Validate.isTrue(filter.getParamValue() == null || filter.getParamValue().length() < 256 , "具体的固定值,填入值超过了限定长度(256)，请检查!");
    }
  }
} 
