package com.bizunited.platform.core.service.file;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Set;

/**
 * kuiperFileService服务接口的默认实现
 * @author yinwenjie
 */
public class NebulaFileServiceSimpleImpl implements NebulaFileService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(NebulaFileServiceSimpleImpl.class);
  
  /**
   * 保存文件的根路径描述（包括和图片设置的根目录重用）
   */
  @Value("${nebula.file.fileRoot}")
  private String fileRoot;
  /**
   * 最大允许的单个文件上传大小（单位MB）
   */
  @Value("${nebula.file.maxFileSize}")
  private Integer maxFileSize;
  /**
   * 允许上传的文件
   */
  @Value("${nebula.file.whitePrefixs}")
  private String[] whitePrefixs;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.file.NebulaFileService#saveFile(java.lang.String, java.lang.String, java.lang.String, byte[])
   */
  @Override
  public void saveFile(String relativePath, String fileName, String fileRename, byte[] fileContext) {
    /*
     * 文件保存过程为：
     * 1、验证文件内容的基本面
     * 2、默认取得根路径和可能存在的相对路径，构成文件存储的完成目录路径，并在没有该目录的情况下，建立目录
     * 3、然后进行文件存储，使用重命名后的文件名进行
     * */
    // 1、=====验证文件内容的基本面
    long fileSize = fileContext.length;
    LOGGER.debug("====fiel Upload====originalFilename:{},fileSize:{}" , fileName , fileSize);
    // 如果条件成立，说明大于最大上传大小了
    Validate.isTrue(fileSize < maxFileSize * 1024 * 1024 , "image file should be less than %dMB!" , maxFileSize);
    if(!StringUtils.isBlank(relativePath)) {
      Validate.matchesPattern(relativePath, "/{1}[a-z|A-Z|0-9|_|/|\\.]+", "relativePath must begin with chartset '/' or relativePath noncompliance !!");
    }
    // 文件后缀，必须存在后缀，其中txt后缀的文件都是允许的
    int nodeIndex = StringUtils.lastIndexOf(fileName, ".");
    String filePrefix = StringUtils.substring(fileName, nodeIndex + 1, fileName.length());
    Set<String> whitePrefixSet = Sets.newHashSet(whitePrefixs);
    whitePrefixSet.add("txt");
    Validate.isTrue(whitePrefixSet.stream().anyMatch(item -> StringUtils.equalsAnyIgnoreCase(item, filePrefix)) , "file prefix[%s] is not allow" , fileName);
    
    // 2、=====确定使用的文件夹，没有即创建
    String absoluteDirPath = StringUtils.join(fileRoot , relativePath);
    File absoluteDirPathFile = new File(absoluteDirPath);
    if(!absoluteDirPathFile.exists()) {
      Validate.isTrue(absoluteDirPathFile.mkdirs() , "无法创建指定的文件夹，因为没有操作权限[%s]，请检查文件存储的操作目录的权限!" , absoluteDirPath);
    }
    
    // 3、=====开始保存文件
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    try (FileOutputStream fileOut = new FileOutputStream(new File(fullPath));
        ByteArrayInputStream byteIn = new ByteArrayInputStream(fileContext);) {
      int maxLen = 204800;
      byte[] readContext = new byte[maxLen];
      int realen;
      while((realen = byteIn.read(readContext, 0, maxLen)) != -1) {
        fileOut.write(readContext, 0, realen);
      }
    } catch (IllegalStateException | IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.file.kuiperFileService#deleteFile(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  public void deleteFile(String relativePath, String fileName, String fileRename) {
    if(!StringUtils.isBlank(relativePath)) {
      Validate.matchesPattern(relativePath, "/{1}[a-z|A-Z|0-9|_|/|\\.]+", "relativePath must begin with chartset '/' or relativePath noncompliance !!");
    }
    String absoluteDirPath = StringUtils.join(fileRoot , relativePath);
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    File fullPathFile = new File(fullPath);
    
    // 如果存在则进行删除
    if(fullPathFile.exists()) {
      Validate.isTrue(fullPathFile.delete() , "无法删除文件，因为没有操作权限[%s]，请检查文件存储的操作目录的权限",fullPath);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.file.kuiperFileService#readFileContent(java.lang.String, java.lang.String)
   */
  @Override 
  public byte[] readFileContent(String relativePath, String fileRename) {
    /*
     * 读取文件内容的过程包括：
     * 1、检查文件路径、入参的基本面
     * 2、如果指定的完整路径下，文件存在，则进行读取，否则返回null
     * */
    String absoluteDirPath = StringUtils.join(fileRoot , relativePath);
    String fullPath = StringUtils.join(absoluteDirPath , "/" , fileRename);
    File fullPathFile = new File(fullPath);
    if (!fullPathFile.exists()) {
      return new byte[0];
    }
    // 如果不是文件，也不行
    if(!fullPathFile.isFile()) {
      return new byte[0];
    }
    
    byte[] fileContents = null;
    try (FileInputStream ois = new FileInputStream(fullPathFile);
        ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      int maxLen = 204800;
      byte[] readContext = new byte[maxLen];
      int realen;
      while((realen = ois.read(readContext, 0, maxLen)) != -1) {
        out.write(readContext, 0, realen);
      }
      fileContents = out.toByteArray();
    } catch (Exception e) {
      throw new IllegalArgumentException(e);
    }
    return fileContents;
  }
}
