package com.bizunited.platform.core.service.image;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.bizunited.platform.core.entity.OrdinaryFileEntity;
import com.bizunited.platform.core.repository.file.OrdinaryFileRepository;
import com.bizunited.platform.core.service.file.NebulaFileService;

@Service("imageViewServiceImpl")
public class FileViewServiceImpl implements FileViewService {
  /**
   * 日志
   * */
  private static final Log LOGGER = LogFactory.getLog(FileViewServiceImpl.class);
  private static final String MESS_CUT_IMAGE = "cutimage";
  private static final String MESS_ZOOM_IMAGE = "zoomimage";
  @Autowired
  private OrdinaryFileRepository ordinaryFileRepository;
  @Autowired
  private NebulaFileService kuiperFileService;

  @Override
  public byte[] fileQuery(String relativeLocal, String fileName) throws  IOException {
    if(relativeLocal == null || StringUtils.isBlank(fileName)) {
      return new byte[] {};
    }
    OrdinaryFileEntity ordinaryFile = this.ordinaryFileRepository.findByFileNameAndRelativeLocal(fileName, relativeLocal);
    if(ordinaryFile == null) {
      return new byte[] {};
    }
    String prefix = ordinaryFile.getPrefix();
    String originalFileName = ordinaryFile.getOriginalFileName();
    return this.fileContentsQuery(relativeLocal, originalFileName, fileName, "", prefix);
  }

  private byte[] fileContentsQuery(String relativeLocal, String fileName , String fileRename , String special ,String prefix) throws  IOException {
    /*
     * TODO 后续再做ecache缓存
     * 磁盘操作 ============= 
     * 1、根据imageRoot和imageFile的信息在文件系统上提取原始信息，如果没有提取到则不再处理
     * 2、根据既定的特效命令格式，分析请求者要求的特效情况 
     * 3、更具这个特效情况，使用特效构造者构造特效
     * 4、显示到页面上
     */
    // 1、========
    byte[] fileBytes = this.kuiperFileService.readFileContent(relativeLocal, fileRename);
    if (fileBytes == null) {
      return new byte[] {}; 
    }
  
    // 2、=======
    // 如果没有特效要求，就不进行特效处理，直接显示原图（原图不保存）,要进行图片特效，当前文件的扩展名必须是图片的扩展名
    if (StringUtils.isBlank(special) || (!StringUtils.equalsIgnoreCase(prefix, "jpeg")
        && !StringUtils.equalsIgnoreCase(prefix, "jpg") && !StringUtils.equalsIgnoreCase(prefix, "gif")
        && !StringUtils.equalsIgnoreCase(prefix, "bmp") && !StringUtils.equalsIgnoreCase(prefix, "png"))) {
      return fileBytes;
    }
    String[] specialSteps = special.split("\\-\\>");
    if (specialSteps.length == 0) {
      return fileBytes;
    }
    BufferedImage bufferedImage = null;
    try {
      bufferedImage = javax.imageio.ImageIO.read(new ByteArrayInputStream(fileBytes));
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
  
    // 3、=======
    // 图片处理器创建者
    ImageHandlerBuilder.Builder builder = new ImageHandlerBuilder.Builder();
    for (String specialStep : specialSteps) {
      String[] params = specialStep.split("\\|");
      if (StringUtils.equals(params[0], MESS_CUT_IMAGE)) {
        this.builderCutAndZoomImageHandle(params, builder, MESS_CUT_IMAGE);
      } else if (StringUtils.equals(params[0], MESS_ZOOM_IMAGE)) {
        this.builderCutAndZoomImageHandle(params, builder, MESS_ZOOM_IMAGE);
      } else if (StringUtils.equals(params[0], "markimage")) {
        this.builderMarkImageHandle(params, builder);
      }
    }
  
    // 4、=======
    ImageHandler imageHandle = builder.build();
    BufferedImage imageResults = imageHandle.dispose(bufferedImage);
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    javax.imageio.ImageIO.write(imageResults, prefix, out);
    return out.toByteArray();
  }

  /* (non-Javadoc)
   * @see com.cheshangma.platform.imageServer.service.iface.ImageViewService#imageQuery(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  public byte[] imageQuery(String folder, String imageFile, String prefix, String special) throws IOException {
    String relativePath = folder;
    String fileName =  imageFile + "." + prefix;
    return this.fileContentsQuery(relativePath, fileName, fileName, special, prefix);
  }

  /**
   * 这个私有方法用于创建水印图片的特效
   * @param params 水印图片的特效参数（实际上只有一个markValue）
   * @param builder 特效创建器
   * */
  private void builderMarkImageHandle(String[] params, ImageHandlerBuilder.Builder builder) {
    String markValue = null;
    Integer fontSize = 20;
    for (String param : params) {
      String[] paramVars = param.split("=");
      if (StringUtils.equals(paramVars[0], "markValue")) {
        markValue = paramVars[1];
      }
      if (StringUtils.equals(paramVars[0], "fontSize")) {
        fontSize = Integer.parseInt(paramVars[1]);
      }
    }

    // 对参数的正确选定进行判断
    if (StringUtils.isEmpty(markValue)) {
      throw new IllegalArgumentException("markValue must be set value!");
    }
    // 设定到创建器
    builder.createMarkHandler(markValue , fontSize);
  }

  /**
   * 这个私有方法用于创建裁剪图片/缩放图片的特效
   * @param params 裁剪图片/缩放图片的特效参数
   * @param builder 特效创建器
   * */
  private void builderCutAndZoomImageHandle(String[] params, ImageHandlerBuilder.Builder builder, String opType) {
    Integer width = null;
    Integer height = null;
    Float ratio = null;
    for (String param : params) {
      String[] paramVars = param.split("=");
      if (StringUtils.equals(paramVars[0], "width")) {
        width = Integer.parseInt(paramVars[1]);
      } else if (StringUtils.equals(paramVars[0], "height")) {
        height = Integer.parseInt(paramVars[1]);
      } else if (StringUtils.equals(paramVars[0], "ratio")) {
        ratio = Float.parseFloat(paramVars[1]);
      }
    }

    // 对参数的正确选定进行判断
    if (ratio != null && (ratio <= 0f || ratio >= 1f)) {
      throw new IllegalArgumentException("ratio must between from 0 to 1!");
    } else if (ratio == null && (width == null || height == null)) {
      throw new IllegalArgumentException("width and height must be set value!");
    }

    // 设定到创建器
    if (ratio != null && StringUtils.equals(opType, MESS_CUT_IMAGE)) {
      builder.createCutHandler(ratio);
    } else if (ratio != null && StringUtils.equals(opType, MESS_ZOOM_IMAGE)) {
      builder.createZoomHandler(ratio);
    } else if (ratio == null && StringUtils.equals(opType, MESS_CUT_IMAGE)) {
      builder.createCutHandler(width, height);
    } else if (ratio == null && StringUtils.equals(opType, MESS_ZOOM_IMAGE)) {
      builder.createZoomHandler(width, height);
    }
  }
  
  @Override
  public OrdinaryFileEntity findByFileNameAndRelativeLocal(String fileName, String relativeLocal) {
    if(relativeLocal == null || StringUtils.isBlank(fileName)) {
      return null;
    }
    return ordinaryFileRepository.findByFileNameAndRelativeLocal(fileName, relativeLocal);
  }
}
