package com.bizunited.platform.core.service.init;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.ButtonEntity;
import com.bizunited.platform.core.entity.CompetenceEntity;
import com.bizunited.platform.core.repository.ButtonRepository;
import com.bizunited.platform.core.repository.CompetenceRepository;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;


/**
 * 初始化菜单抽象类
 * @Author: Paul Chan
 * @Date: 2019-10-15 16:00
 */
public abstract class AbstractCompetenceInitProcess implements InitProcessService {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractCompetenceInitProcess.class);

  @Autowired(required = false)
  private ButtonRepository buttonRepository;
  @Autowired(required = false)
  private CompetenceRepository competenceRepository;

  @Override
  public int sort() {
    return 0;
  }

  /**
   * 初始化核心菜单,inputStream是一个菜单数组的文件输入流内容的格式见当前包下的nebula-competences.json资源文件
   * @param inputStream
   */
  protected void initCompetences(InputStream inputStream) throws IOException {
    if(inputStream == null) {
      return;
    }
    if(competenceRepository == null) {
      return;
    }
    StringBuilder sb = new StringBuilder();
    try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))) {
      String line = bufferedReader.readLine();
      while (line != null){
        sb.append(line);
        line = bufferedReader.readLine();
      }
    }
    String json = sb.toString();
    if(StringUtils.isBlank(json)) {
      LOGGER.warn("初始化菜单json为空！！");
      return;
    }
    JSONArray competences = JSON.parseArray(json);
    this.initCompetences(null, competences);
  }

  /**
   * 初始化核心菜单,parent是菜单的上级，如果菜单没有上级，则parent为null。
   * competences是CompetenceEntity对应的json数组，具体内容格式见nebula-competences.json资源文件，
   * 菜单地址根据菜单名称更新，若菜单不存在，则初始化菜单，且菜单默认启用。
   * 请保证json对象的字段跟nebula-competences.json资源文件一样
   * @param parent
   * @param competences
   */
  protected void initCompetences(CompetenceEntity parent, JSONArray competences) {
    if(CollectionUtils.isEmpty(competences)) {
      return;
    }
    Iterator<Object> iterator = competences.iterator();
    while (iterator.hasNext()) {
      Object object = iterator.next();
      JSONObject competenceJson = JSON.parseObject(object.toString());
      CompetenceEntity competence = JSON.parseObject(competenceJson.toJSONString(), CompetenceEntity.class);
      CompetenceEntity oldCompetence;
      if(parent != null) {
        oldCompetence = competenceRepository.findByCommentAndViewItemAndParent(competence.getComment(), 1, parent.getId());
      } else {
        oldCompetence = competenceRepository.findByCommentAndViewItemAndNullParent(competence.getComment(), 1);
      }
      if(oldCompetence == null) {
        competence.setCreateDate(new Date());
        competence.setModifyDate(new Date());
        competence.setTstatus(1);
        competence.setParent(parent);
        competenceRepository.save(competence);
        oldCompetence = competence;
      }
      this.initButtons(oldCompetence, competence.getButtons());
      if(competenceJson.containsKey("competences")) {
        this.initCompetences(oldCompetence, competenceJson.getJSONArray("competences"));
      }
    }
  }

  /**
   * 初始化菜单下的按钮
   * 该初始化只做按钮初始化新增的处理，不做更新处理
   * @param competence
   * @param buttons
   */
  private void initButtons(CompetenceEntity competence, Set<ButtonEntity> buttons) {
    if(buttonRepository == null || CollectionUtils.isEmpty(buttons)) {
      return;
    }
    for (ButtonEntity button : buttons) {
      Set<CompetenceEntity> competencesJson = button.getCompetences();
      ButtonEntity oldButton = buttonRepository.findByCode(button.getCode());
      if(oldButton == null) {
        button.setCompetence(competence);
        button.setSystem(true);
        button.setCreateTime(new Date());
        button.setModifyTime(new Date());
        button.setCompetences(null);
        buttonRepository.save(button);
      } else {
        Validate.isTrue(competence.getId().equals(oldButton.getCompetence().getId()), "重复的按钮比编码【%s】,请检查", button.getCode());
        button = oldButton;
      }
      this.initButtonCompetences(button, competencesJson);
    }
  }

  /**
   * 初始化按钮下的接口
   * @param button
   * @param competencesJson
   */
  private void initButtonCompetences(ButtonEntity button, Set<CompetenceEntity> competencesJson){
    if(CollectionUtils.isEmpty(competencesJson)) {
      return;
    }
    if(button == null){
      return;
    }
    Map<String, CompetenceEntity> competencesMap = new HashMap<>();
    Set<CompetenceEntity> buttonCompetences = button.getCompetences();
    if(buttonCompetences == null){
      buttonCompetences = new HashSet<>();
      button.setCompetences(buttonCompetences);
    }else {
      for (CompetenceEntity entity : buttonCompetences) {
        competencesMap.put(entity.getId(), entity);
      }
    }
    for(CompetenceEntity competence : competencesJson){
      CompetenceEntity oldCompetence = competenceRepository.findByResourceAndMethods(competence.getResource(), competence.getMethods());
      if(oldCompetence == null) {
        LOGGER.warn("未找到接口路径【{}】,请求方法【{}】，请检查!", competence.getResource(), competence.getMethods());
        continue;
      }
      if(competencesMap.containsKey(oldCompetence.getId())){
        continue;
      }
      buttonCompetences.add(oldCompetence);
    }
    buttonRepository.save(button);
  }
}
