package com.bizunited.platform.core.service.init;

import com.bizunited.platform.core.annotations.NebulaService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.common.constant.ParamClassTypeConst;
import com.bizunited.platform.core.entity.ServicableMethodEntity;
import com.bizunited.platform.core.entity.ServicableMethodPropertyEntity;
import com.bizunited.platform.core.service.NebulaStaticPersistentService;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

@Component("ServicableMethodInitProcess")
public class ServicableMethodInitProcess implements InitProcessService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ServicableMethodInitProcess.class);
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private NebulaStaticPersistentService nebulaStaticPersistentService;
  @Autowired
  private ServicableMethodService servicableMethodService;
  private ClassPool pool = ClassPool.getDefault();
  /**
   * 该属性表示该应用程序是否忽略ServicableMethod注解的扫描
   */
  @Value("${nebula.ignoreServicableMethod:false}")
  private boolean ignoreServicableMethod;
  
  @Override
  public boolean doProcess() {
    // 每次进程启动都要进行初始化动作
    return !this.ignoreServicableMethod;
  }
  
  @Override
  public int sort() {
    // 初始化动作靠前
    return 5;
  }

  @Override
  public boolean stopOnException() {
    // 一旦出现异常，则终止进程启动过程
    return true;
  }

  @Override
  public void init() {
    try {
      this.initServicableMethod();
    } catch(NotFoundException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    } 
  }
  /**
   * 对已经标注了serviceMethod的注解进行分析、保存处理——在spring启动成功后
   * @throws Exception
   */
  private void initServicableMethod() throws NotFoundException , ClassNotFoundException {
    LOGGER.info("启动器正在进行初始化服务方法扫描.......");
    Map<String, Object> beanMapping =  applicationContext.getBeansWithAnnotation(NebulaService.class);
    Collection<Object> mappingAnnotations = beanMapping.values();
    // 如果条件成立，说明当前没有扫描到任何Service服务层暴露定义，直接删除数据库中的设定信息
    servicableMethodService.deleteServiceMethodAndProperties();
    if(mappingAnnotations == null || mappingAnnotations.isEmpty()) {
      return;
    }

    // 寻找指定的实现类下，其直接实现的接口中的所有ServiceMethod标注
    ArrayList<CtMethod> serviceMethods = new ArrayList<>();
    for (Object mappingAnnotationItem : mappingAnnotations) {
      Class<?> targetClass = mappingAnnotationItem.getClass();
      Set<Class<?>> currentInterfaces = new LinkedHashSet<>();
      this.recursionFindInterface(targetClass, currentInterfaces);
      if(currentInterfaces.isEmpty()) {
        continue;
      }
      List<CtClass> implInterfaceClasses = new LinkedList<>();
      for (Class<?> currentInterfaceItem : currentInterfaces) {
        implInterfaceClasses.add(pool.getCtClass(currentInterfaceItem.getName()));
      }
      implInterfaceClasses.stream().map(CtClass::getDeclaredMethods).forEach(arrayItems ->
        Arrays.stream(arrayItems).filter(item -> item.hasAnnotation(NebulaServiceMethod.class)).forEach(item -> serviceMethods.add(item))
      );
    }
     
    /*
     * 处理过程为：
     * 1、首先判断当前ServiceMethod的基本信息，包括name、desc和returnPropertiesFilter、包名、类名、方法名等，检验name的重复性
     * 以及判断recordQuery和recordUpdate的定义
     * 2、然后分析和处理其返回值（详见具体的私有方法）
     * 3、接着分析和处理其形参（最重要的步骤，详见私有方法）
     * 4、将已经完成的各个ServiceMethod方法，重新进行数据库初始化——具体过程为先删除再依据name重新新建
     * */
    List<ServicableMethodEntity> serviceMethodReops = new ArrayList<>();
    for (CtMethod ctMethod : serviceMethods) {
      ServicableMethodEntity serviceMethodReop = new ServicableMethodEntity();
      // 1、==============
      NebulaServiceMethod serviceMethodAnno = (NebulaServiceMethod)ctMethod.getAnnotation(NebulaServiceMethod.class);
      String serviceMethodAnnoName = serviceMethodAnno.name();
      if(StringUtils.isBlank(serviceMethodAnnoName)) {
        LOGGER.warn("指定的kuiperServiceMethodAnno[{}]没有设定name，请检查!!", serviceMethodAnnoName);
        continue;
      }
      String serviceMethodAnnoDesc = serviceMethodAnno.desc();
      if(StringUtils.isBlank(serviceMethodAnnoDesc)) {
        LOGGER.warn("指定的kuiperServiceMethodAnno[{}]没有设定desc，请检查!!", serviceMethodAnnoName);
        continue;
      }
      String returnPropertiesFilter = serviceMethodAnno.returnPropertiesFilter();
      // 方法名
      String methodSimpleName = ctMethod.getName();
      // 完整的类名
      String className = ctMethod.getDeclaringClass().getName();
      // 和已经分析完成的暴露方法相比，不能有重复的暴露方法名
      Validate.isTrue(!serviceMethodReops.stream().map(ServicableMethodEntity::getName).filter(item -> StringUtils.equals(item, serviceMethodAnnoName)).findAny().isPresent(), "在进行服务暴露定义分析时，发现有重复的name[" + serviceMethodAnnoName + "]，请检查!!");
      //分析WRITE性质的方法注解定义
      analysisWriteMethodAnno(serviceMethodAnno , serviceMethodReop , ctMethod);
      serviceMethodReop.setUsedScope(serviceMethodAnno.scope().name());
      serviceMethodReop.setDescription(serviceMethodAnnoDesc);
      serviceMethodReop.setInterfaceName(className);
      String methodName = StringUtils.join(className , "." , methodSimpleName);
      serviceMethodReop.setMethodName(methodName);
      serviceMethodReop.setName(serviceMethodAnnoName);
      serviceMethodReop.setSimpleMethodName(methodSimpleName);
      serviceMethodReop.setReturnPropertiesFilter(returnPropertiesFilter);
      // 2、==============
      analysisMethodParameters(serviceMethodReop, ctMethod);
      
      // 3、==============
      analysisMethodReturnParameter(serviceMethodReop, ctMethod);
      
      // 4、==============
      serviceMethodReops.add(serviceMethodReop);
      servicableMethodService.create(serviceMethodReop);
    }
  }
  
  /**
   * 辨别当前注解的scope属性是否是WRITE性质的。如果是，需要进一步判断是否设置recordQuery和recordUpdate
   * 如果没有设置，这里约定recordQuery默认就是使用我们表单代码生成器生成的findDetailsByFormInstanceId方法，约定recordUpdate默认为空
   * 如果有设置，这里约定recordQuery和recordUpdate就是其设置的方法
   * 注：1.一般情况下，recordQuery就是我们的findDetailsByFormInstanceId，特殊情况才会主动去指定。特殊情况就是引入外界的数据填充
   * recordUpdate是除表单数据之外的业务级更新，当指定了这个属性，我们会再更新自己的表单数据时，调用该指定方法
   * 2.我们约定：不管recordQuery是默认的还是指定的，方法参数都是formInstanceId指定的recordUpdate方法参数只能是当前表单的主模型类
   * 该分析方法不会再次对指定的方法的参数进行分析，都按照上面的约定来约束
   * @param serviceMethodAnno
   * @param serviceMethodReop
   * @param ctMethod
   * @throws ClassNotFoundException 
   * @throws NotFoundException 
   */
  private void analysisWriteMethodAnno(NebulaServiceMethod serviceMethodAnno , ServicableMethodEntity serviceMethodReop , CtMethod ctMethod) throws ClassNotFoundException, NotFoundException {
    //判断是否写操作
    if(serviceMethodAnno.scope() != NebulaServiceMethod.ScopeType.WRITE ) {
      Validate.isTrue(StringUtils.isBlank(serviceMethodAnno.recordQuery()),"%s类%s方法是READ性质的操作，不能含有recordQuery注释，请检查!!",ctMethod.getDeclaringClass().getSimpleName(),ctMethod.getName());
      Validate.isTrue(StringUtils.isBlank(serviceMethodAnno.recoveryService()),"%s类%s方法是READ性质的操作，不能含有recordUpdate注释，请检查!!",ctMethod.getDeclaringClass().getSimpleName(),ctMethod.getName());
      return;
    }
    
    //判断是否指定详情的查询方法
    if(StringUtils.isBlank(serviceMethodAnno.recordQuery())) {
      serviceMethodReop.setRecordQuery(StringUtils.join(StringUtils.substringAfterLast(ctMethod.getDeclaringClass().getName(), "."),".findDetailsByFormInstanceId"));
    }else {
      //这里需要辨别指定的方法是否存在且合法
      this.checkMethodExist(serviceMethodAnno.recordQuery(),ctMethod,serviceMethodAnno.scope());
      serviceMethodReop.setRecordQuery(serviceMethodAnno.recordQuery());
    }
    
    //判断是否指定业务场景的更新方法
    if(StringUtils.isNotBlank(serviceMethodAnno.recoveryService())) {
    //这里需要辨别指定的方法是否存在且合法
      this.checkMethodExist(serviceMethodAnno.recoveryService(),ctMethod,serviceMethodAnno.scope());
      serviceMethodReop.setRecordUpdate(serviceMethodAnno.recoveryService());
    }else {
      serviceMethodReop.setRecordUpdate(serviceMethodAnno.name());
    }
  }
  
  /**
   * 用于检查指定方法是否存在
   * @param specityMethodName
   * @param ctMethod
   * @throws ClassNotFoundException
   * @throws NotFoundException 
   */
  private void checkMethodExist(String specityMethodName , CtMethod ctMethod, NebulaServiceMethod.ScopeType type) throws ClassNotFoundException, NotFoundException {
    String className = ctMethod.getDeclaringClass().getName();
    String[] specityMethodNames = StringUtils.split(specityMethodName, ".");
    Validate.isTrue(specityMethodNames.length == 2,"%s类的自定义日志服务方法名称不符合规定，请检查!!",className);
    String simpleClassName = StringUtils.substringAfterLast(className, ".");
    Validate.isTrue(StringUtils.equals(specityMethodNames[0], simpleClassName),"%s类的自定义日志服务方法名称不符合规定，请检查!!",className);
    Class<?> clazz = Class.forName(className);
    Method[] methods = clazz.getMethods();
    Validate.notEmpty(methods,"根据%s类未能获取到任何的方法定义，请检查!!",className);
    //记录方法是否存在
    boolean methodExist = false;
    //记录同名方法出现的次数
    int count = 0;
    //记录方法参数个数
    int paramCount = 0;
    for(Method method : methods) {
      if(StringUtils.equals(simpleClassName + "."+method.getName(), specityMethodName)) {
        methodExist = true;
        count++;
        paramCount = method.getParameterCount();
      }
    }
    Validate.isTrue(methodExist , "%s类未能获取到%s方法定义，请检查!!",simpleClassName,specityMethodName);
    Validate.isTrue(count == 1 , "%s类%s方法，获取到多个同名方法定义，请变更方法名!!",simpleClassName,specityMethodName);
    
    if(type == NebulaServiceMethod.ScopeType.READ) {
      Validate.isTrue(paramCount == 1 , "%s类%s方法参数只能为一个，请变更方法参数!!",simpleClassName,specityMethodName);
    }
  }
  
  
  /**
   * 递归寻找“真实”接口/类的所有父级接口
   * @param targetClass
   * @param currentInterfaces
   */
  private void recursionFindInterface(Class<?> targetClass , Set<Class<?>> currentInterfaces) {
    // 如果当前类已经被cglib代理了，则需要解析字符串以后，再进行处理
    if(org.springframework.aop.SpringProxy.class.isAssignableFrom(targetClass)) {
      String cglibClassName = targetClass.getName();
      String realClassName =  cglibClassName.split("\\$\\$")[0];
      Class<?> realClass = null;
      try {
        realClass = Class.forName(realClassName);
      } catch (ClassNotFoundException e) {
        LOGGER.warn(e.getMessage());
      }
      if(realClass != null) {
        this.recursionFindInterface(realClass, currentInterfaces);
      }
      return;
    }
    
    if(targetClass.isInterface()) {
      currentInterfaces.add(targetClass);
    }
    
    Class<?>[] interfaces = targetClass.getInterfaces();
    if(interfaces != null && interfaces.length > 0) {
      for (Class<?> interfaceItem : interfaces) {
        recursionFindInterface(interfaceItem , currentInterfaces);
      }
    }
  }
  /**
   * 该私有方法用于分析指定方法的参数值信息
   * @param serviceMethodReop
   * @param ctMethod
   */
  private void analysisMethodParameters(ServicableMethodEntity serviceMethodReop , CtMethod ctMethod) throws NotFoundException , ClassNotFoundException {
    String className = ctMethod.getDeclaringClass().getName();
    String methodName = ctMethod.getName();
    Set<ServicableMethodPropertyEntity> serviceMethodProperties = new LinkedHashSet<>();
    /*
     * ===========================================================================================
     * 1、进行参数分析
     * 1.1、规定的支持的基本类型，这部分的参数必须有@ServiceMethodParam注解标识，即使参数只有一个，也要注解标识 --->  params
     * 1.2、判断参数是否用户信息Principal，对于Principal参数的处理，这个版本允许多个，且不需要注解指定，并且只能是操作者
     * 1.3、判断参数是否Pageable类型，对于Pageable参数的处理，这个版本不允许多个
     * 1.4、基于SaturnContext上下文对象，分析是否是静态模型类型的参数，如果存在多个则报错 ---> formData
     *          如果没有任何静态模型类型的参数，则忽略本步骤的处理
     * 1.5、判断参数是否KuiperInvokeParams类型，KuiperInvokeParams类型只允许一个(其实就是k-v结构的数据集)
     *  
     *  注：如果参数类型以上都不是，则抛出异常信息，不能识别的参数类型
     * */
    CtClass[] parameterTypes = ctMethod.getParameterTypes();
    if(parameterTypes == null || parameterTypes.length == 0) {
      return;
    } 
    
    Map<String , Map<Integer,CtClass>> baseParameters = new HashMap<>();
    Map<Integer , CtClass> objectParameters = new HashMap<>();
    Map<Integer , CtClass> pageParameters = new HashMap<>();
    Map<Integer , CtClass> principalParameters = new HashMap<>();
    Map<Integer , CtClass> mapParameters = new HashMap<>();
    
    Object[][] paramAnno = ctMethod.getParameterAnnotations();
    
    //for循环这里只是把参数信息分门别类的记录下来，详细的判断在后面的逻辑里
    //判断参数，并记录参数的信息及索引index位置信息。并且保存有ServiceMethodParam注解的限定名信息
    for(int index = 0 ; index < parameterTypes.length ; index++) {
      CtClass parameterTypeItem = parameterTypes[index];
      
      boolean flag = false;//当前元素是否已验证成功的开关值
      
      
      //基础的规定的类型判断
      Map<Integer,CtClass> maps = baseParameters.get(parameterTypeItem.getName()) == null ? new HashMap<>() : baseParameters.get(parameterTypeItem.getName());
      
      //先对参数的名称去除 [] 数组标识
      String baseParamName = parameterTypeItem.getName().replaceAll("\\[\\]", "");
      
      //先按名称过滤
      if(ParamClassTypeConst.getBaseParamClassTypeList().contains(baseParamName) || ParamClassTypeConst.getBaseParamSpecialTypeList().contains(baseParamName) ) {
        maps.put(index, parameterTypeItem);
        baseParameters.put(parameterTypeItem.getName(), maps);
        continue;
      }
      
      //过滤是否继承或实现关系
      for(String specialType : ParamClassTypeConst.getBaseParamSpecialTypeList()) {
        if(pool.getCtClass(baseParamName).subtypeOf(pool.getCtClass(specialType))) {
          maps.put(index, parameterTypeItem);
          baseParameters.put(parameterTypeItem.getName(), maps);
          flag = true;
          break;
        }
      }
      if(flag) continue;
      
      
      //一些允许的指定的类型判断(ConditionParams，Principal，Pageable)
      if(parameterTypeItem.subtypeOf(pool.getCtClass(InvokeParams.class.getName())) ) {
        mapParameters.put(index, parameterTypeItem);
        continue;
      } else if(parameterTypeItem.subtypeOf(pool.getCtClass("java.security.Principal"))) {
        principalParameters.put(index, parameterTypeItem);
        continue;
      } else if(parameterTypeItem.subtypeOf(pool.getCtClass("org.springframework.data.domain.Pageable"))) {
        pageParameters.put(index, parameterTypeItem);
        continue;
      }
      
      //判断参数是否符合Kuiper生成的类信息
      String parameterClaaName = parameterTypeItem.getName();
      Validate.isTrue(nebulaStaticPersistentService.findByPersistentClass(parameterClaaName) != null,"在扫描服务暴露[" + className + ":" + methodName + "]时，发现有不能识别的参数对象类型，请检查!!");
      objectParameters.put(index, parameterTypeItem);
    }
    //开始判断参数的详细信息
    CtClass parameterTypeItem = null;
    Integer parameterAnnoIndex = null;

    //1.首先判断规定的基本类型
    //注：规定的基本类型，统一存储为jvm的class类型
    if(baseParameters.size() == 1) {
      Map<Integer,CtClass> map = baseParameters.values().iterator().next();
      for(Entry<Integer, CtClass> entry : map.entrySet()) {
        parameterTypeItem = entry.getValue();
        parameterAnnoIndex = entry.getKey();
        Validate.isTrue(Arrays.stream(paramAnno[parameterAnnoIndex.intValue()]).findAny().filter(e -> e instanceof ServiceMethodParam).isPresent(),
          "在扫描服务暴露[" + className + ":" + methodName + "]时，规定的基本类型参数上并没有发现ServiceMethodParam注解标注，请检查!!");
        ServicableMethodPropertyEntity methodPropertyEntity  = buildMethodPropertyEntity(parameterTypeItem.getName(),1, parameterAnnoIndex, null, serviceMethodReop, paramAnno);
        serviceMethodProperties.add(methodPropertyEntity);
      }
    }else if(baseParameters.size() > 1){
      Set<String> paramClassTypes = baseParameters.keySet();
      for(String v : paramClassTypes) {
        Map<Integer,CtClass> sameBaseParamsMap = baseParameters.get(v);
        if(sameBaseParamsMap.size() == 1) {
          Validate.isTrue(Arrays.stream(paramAnno[sameBaseParamsMap.keySet().iterator().next().intValue()]).findAny().filter(e -> e instanceof ServiceMethodParam).isPresent(),
            "在扫描服务暴露[" + className + ":" + methodName + "]时，规定的基本类型参数上并没有发现ServiceMethodParam注解标注，请检查!!");
          ServicableMethodPropertyEntity methodPropertyEntity  = buildMethodPropertyEntity(sameBaseParamsMap.values().iterator().next().getName(),1, sameBaseParamsMap.keySet().iterator().next(), null, serviceMethodReop, paramAnno);
          serviceMethodProperties.add(methodPropertyEntity);
          continue;
        }
        Set<Integer> indexSet = sameBaseParamsMap.keySet();
        for(Integer index : indexSet) {
          //判断是否存在注解信息
          Validate.isTrue(Arrays.stream(paramAnno[index]).findAny().filter(item -> item instanceof ServiceMethodParam).isPresent(),"在扫描服务暴露[" + className + ":" + methodName + "]时，规定的基本类型参数上并没有发现ServiceMethodParam注解标注，请检查!!");
          //规定的基本类型通过了验证判断，则开始构造ServicableMethodPropertyEntity
          ServicableMethodPropertyEntity methodPropertyEntity  = buildMethodPropertyEntity(sameBaseParamsMap.get(index).getName(),1, index, null, serviceMethodReop, paramAnno);
          serviceMethodProperties.add(methodPropertyEntity);
        }
      }
    }
    
    //1.2 开始判断参数是否用户信息Principal，如果是多个，不需要注解指定。但Principal只能是操作者信息
    if(principalParameters.size() > 0) {
      Set<Integer> parameterIndexs = principalParameters.keySet();
      for (Integer parameterIndex : parameterIndexs) {
        parameterTypeItem = principalParameters.get(parameterIndex);
        parameterAnnoIndex = parameterIndex;
        ServicableMethodPropertyEntity methodPropertyEntity  = buildMethodPropertyEntity(parameterTypeItem.getName(),2, parameterAnnoIndex, null, serviceMethodReop, paramAnno);
        serviceMethodProperties.add(methodPropertyEntity);
      }
    }
    
    //1.3 开始判断参数是否Pageable类型
    Validate.isTrue(pageParameters.size() <= 1,"在扫描服务暴露[" + className + ":" + methodName + "]时，发现多个Pageable类型参数，至多只允许一个，请检查!!");
    if(pageParameters.size() == 1) {
      parameterTypeItem = pageParameters.get(pageParameters.keySet().iterator().next());
      parameterAnnoIndex = pageParameters.keySet().iterator().next();
      serviceMethodProperties.add(this.buildMethodPropertyEntity(parameterTypeItem.getName(),3, parameterAnnoIndex, null, serviceMethodReop, paramAnno));
    }
    
    //1.4 开始判断参数是否符合Kuiper静态模型的对象类型
    Validate.isTrue(objectParameters.size() <= 1,"在扫描服务暴露[" + className + ":" + methodName + "]时，发现多个符合Kuiper静态模型的对象类型，请检查!!");
    if(objectParameters.size() == 1) {
      parameterTypeItem = objectParameters.values().iterator().next();
      parameterAnnoIndex = objectParameters.keySet().iterator().next();
      String objectClassName = parameterTypeItem.getName();
      serviceMethodProperties.add(this.buildMethodPropertyEntity(parameterTypeItem.getName(),4, parameterAnnoIndex, objectClassName, serviceMethodReop, paramAnno));
    }
    
    //1.5 开始判断参数是否KuiperInvokeParams类型，只允许一个KuiperInvokeParams
    Validate.isTrue(mapParameters.size() <= 1,"在扫描服务暴露[" + className + ":" + methodName + "]时，发现多个KuiperInvokeParams类型参数，至多只允许一个，请检查!!");
    if(mapParameters.size() == 1) {
      parameterTypeItem = mapParameters.get(mapParameters.keySet().iterator().next());
      parameterAnnoIndex = mapParameters.keySet().iterator().next();
      serviceMethodProperties.add(this.buildMethodPropertyEntity(parameterTypeItem.getName(),5, parameterAnnoIndex, null, serviceMethodReop, paramAnno));
    }
    
    
    
    //判断已有@ServiceMthodParam标注的注解方法参数，其限定名是否相同。如果相同，则抛出异常
    Set<String> annonIsRepeat = new HashSet<>();
    for(ServicableMethodPropertyEntity e : serviceMethodProperties) {
      Validate.isTrue(!(StringUtils.isNotBlank(e.getAnnonQualifiedName()) && !annonIsRepeat.add(e.getAnnonQualifiedName())),
        "在扫描服务暴露[" + className + ":" + methodName + "]时，发现多个有注解的参数限定名相同，请检查!!");
    }
    
    //设置方法参数实体数据
    serviceMethodReop.setProperties(serviceMethodProperties);
  }
  
  /**
   * 该私有方法用于分析指定方法的返回值信息
   * @param serviceMethodReop
   * @param ctMethod
   */
  private void analysisMethodReturnParameter(ServicableMethodEntity serviceMethodReop , CtMethod ctMethod) throws NotFoundException , ClassNotFoundException {
    CtClass returnClass = null;
    CtClass collectionClass = null;
    boolean isCollectionInterface = false;
    String className = ctMethod.getDeclaringClass().getName();
    String methodName = ctMethod.getName();
    Class<?> javaClass = Class.forName(className);
    // 通过java原生类工具，得到方法定义，以便后续取得集合返回信息中的泛型
    List<Class<?>> parametersClasses = new LinkedList<>();
    TreeSet<ServicableMethodPropertyEntity> properties = new TreeSet<>((e1, e2) -> e1.getPropertyIndex().intValue() - e2.getPropertyIndex().intValue())  ;
    if(serviceMethodReop.getProperties() != null && !serviceMethodReop.getProperties().isEmpty()) {
      properties.addAll(serviceMethodReop.getProperties());
    }
    for(ServicableMethodPropertyEntity parametersCtClassItem : properties) {
      String parametersClassName = parametersCtClassItem.getParamClass();
      if(parametersCtClassItem.getParamType() == 1 && ParamClassTypeConst.getBaseClass(parametersClassName) != null) {
        parametersClasses.add(ParamClassTypeConst.getBaseClass(parametersClassName));
      } else {
        parametersClasses.add(Class.forName(parametersClassName));
      }
    }
    Method javaMethod = null;
    try {
      javaMethod = javaClass.getDeclaredMethod(methodName, parametersClasses.toArray(new Class[]{}));
    } catch(NoSuchMethodException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("错误的方法" + className + ":" + methodName + "，请检查!!");
    }
    
    // 分析返回值是否是集合性质，即是说实现了java.util.Collection接口
    try {
      returnClass = ctMethod.getReturnType();
      collectionClass = pool.get("java.util.Collection");
      if(returnClass == null || StringUtils.equals(returnClass.getName(), "void")) {
        returnClass = pool.get("java.lang.Void");
      }
      isCollectionInterface = returnClass.subtypeOf(collectionClass) || returnClass.subtypeOf(pool.get(Page.class.getName()));
    } catch (NotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e);
    }
    
    // 如果当前返回值信息是集合性质，则分析其中的泛型，作为返回值类型
    if(isCollectionInterface) {
      String fieldTypeName = null;
      // 接着试图取得泛型
      Type genericType = javaMethod.getGenericReturnType();
      // 如果条件成立才说明在反射类型描述中存在泛型信息
      if(genericType instanceof ParameterizedType) {
        ParameterizedType pt = (ParameterizedType) genericType;
        Class<?> genericClazz = (Class<?>)pt.getActualTypeArguments()[0];
        fieldTypeName = genericClazz.getName();
      }
      
      serviceMethodReop.setReturnCollection(true);
      serviceMethodReop.setReturnClassName(fieldTypeName);
    } else if(StringUtils.equals(returnClass.getName(), "void")) {
      Validate.isTrue(!NebulaServiceMethod.ScopeType.READ.name().equalsIgnoreCase(serviceMethodReop.getUsedScope()),"错误的方法标注" + className + ":" + methodName + "是查询方法，但返回值类型为void，请检查!!");
      serviceMethodReop.setReturnCollection(false);
      serviceMethodReop.setReturnClassName("java.lang.Void");
    }else {
      serviceMethodReop.setReturnCollection(false);
      serviceMethodReop.setReturnClassName(returnClass.getName());
    }
  }
  
  /**
   * 根据参数上的注解和参数下标，获取指定@ServiceMethodParam注解的限定名
   * @param index
   * @param paramAnno
   * @return
   */
  private String getAnnonQualifiedName(int index  , Object[][] paramAnno) {
    if(paramAnno == null || paramAnno.length == 0) {
      return null;
    }
    for(Object annon : paramAnno[index]) {
      if(annon instanceof ServiceMethodParam) {
        ServiceMethodParam smp = (ServiceMethodParam) annon;
        return smp.name();
      }
    }
    return null;
  }
  /**
   * 根据条件构造ServicableMethodPropertyEntity
   * @param paramType
   * @param index
   * @param modelType
   * @param serviceMethodEntity
   * @param paramAnno
   * @return
   */
  private ServicableMethodPropertyEntity buildMethodPropertyEntity(String paramClass , int paramType , int index , String modelType , ServicableMethodEntity serviceMethodEntity , Object[][] paramAnno) {
    ServicableMethodPropertyEntity serviceMethodProperty = new ServicableMethodPropertyEntity();
    //这一部分是用于匹配基本数组的类型，保证把javassist的类类型命名，变更为jvm的类类型
    //ServicableMethodPropertyEntity存储的参数类型，一定是jvm的命名格式
    if(paramType == 1) {
      if(paramClass.contains(".") && paramClass.contains("[]")) {
        int num = count(paramClass,"[]");
        Validate.isTrue(num <= 2,"在扫描服务暴露[" + serviceMethodEntity.getInterfaceName() + ":" + serviceMethodEntity.getSimpleMethodName() + "]时，发现错误的数组类型，目前只支持到二维，请检查!!");
        String str = "";
        do {
          str += "[";
          num--;
        }while(num > 0);
        paramClass = paramClass.replaceAll("\\[\\]", "");
        paramClass = str + "L" + paramClass + ";";
      }else if(!paramClass.contains(".") && paramClass.contains("[]")) {
        paramClass = ParamClassTypeConst.getJavassistBaseClass(paramClass);
      }
    }
    
    serviceMethodProperty.setParamClass(paramClass);
    serviceMethodProperty.setAnnonQualifiedName(getAnnonQualifiedName(index,paramAnno));
    serviceMethodProperty.setParamType(paramType);
    serviceMethodProperty.setPropertyIndex(index);
    serviceMethodProperty.setModelType(modelType);
    serviceMethodProperty.setServiceMethod(serviceMethodEntity);
    return serviceMethodProperty;
  }
  /**
   * 查找指定字符串重复次数
   * @param source
   * @param reg
   * @return
   */
  private int count(String source,String reg) {
    int n = 0;//计数器
    int index = 0;//指定字符的长度
    index = source.indexOf(reg);
    while(index!=-1) {
      n++;
      index = source.indexOf(reg,index+1);
    }
    return n;
  }
}
