package com.bizunited.platform.core.service.invoke;

import java.util.List;

import org.apache.commons.lang3.Validate;

/**
 * 该默认的链式调用过程控制器，在默认环境下使用。
 * @author yinwenjie
 */
public class DefaultHandleChain extends AbstractHandleChain implements HandleChain {
  /**
   * request请求链，主要处理权限、解析参数以及请求过程执行
   */
  private List<InvokeRequestHandle> currentInvokeRequestHandles;
  /**
   * 对调用处理链的结果进行再次的处理
   */
  private List<InvokeResponseHandle> currentInvokeResponseHandles;
  /**
   * 本次调用过程中，在整个调用过程中都起作用的上下文对象。
   */
  private InvokeProxyContext invokeProxyContext;
  /**
   * 当前处理链管理器正在处理的request节点的索引号
   */
  private int currentRequestIndex = 0;
  /**
   * 当前处理链管理器正在处理的response节点的索引号
   */
  private int currentResponseIndex = 0;
  
  @Override
  public void doHandle(InvokeProxyContext context, ChainLogic chainLogic) throws InvokeProxyException {
    Validate.notNull(invokeProxyContext , "调用过程上下文对象不能为空!!");
    Validate.notNull(currentInvokeRequestHandles , "未发现任何request请求连处理节点，请检查!!");
    Validate.isTrue(!currentInvokeRequestHandles.isEmpty() , "未发现任何request请求连处理节点，请检查!!");
    Validate.notNull(currentInvokeResponseHandles , "未发现任何response请求连处理节点，请检查!!");
    Validate.isTrue(!currentInvokeResponseHandles.isEmpty() , "未发现任何response请求连处理节点，请检查!!");
    /*
     * 工作原理为：首先递归处理invokeRequestHandles链，
     * 然后处理invokeResponseHandles
     * 
     * 处理过程为：
     * 1、依次对invokeRequestHandles中的处理节点进行处理，如果出现以下情况，则终止对invokeRequestHandles的处理：
     *   a、如果有异常则终止对invokeRequestHandles的处理，并在上下文中进行记录
     *   b、如果处理节点返回的是BREAK标记，也终止对invokeRequestHandles链的处理
     * 
     * 2、接着对invokeResponseHandles链进行处理，处理规则如下：
     *   a、如果在处理invokeRequestHandles链时发现异常，那么在处理invokeResponseHandles链时，
     *   只有那些标记了handleException为true的RequestHandle节点才 被处理。如果此时又出现异常，则以并抛出
     *   b、如果在处理invokeRequestHandles链时正常完成（遇到END标记，也算是正常完成），
     *   那么则依次处理invokeRequestHandles链上的节点，如果此时出现异常，则直接抛出记录并抛出异常
     * */
    int maxRequestHandlesSize = currentInvokeRequestHandles.size();
    int maxResponseHandlesSize = currentInvokeResponseHandles.size();
    // 如果一下条件成立，则说明整个处理过程已经完成
    if(currentRequestIndex >= maxRequestHandlesSize && currentResponseIndex >= maxResponseHandlesSize) {
      this.throwException(context);
      return;
    }
    
    // 1、=========
    // 1.a、如果条件成立，则说明需要继续处理request链，且之前的处理都正常完成
    if(currentRequestIndex < maxRequestHandlesSize && chainLogic == ChainLogic.CONTINUE) {
      InvokeRequestHandle currentHandle = currentInvokeRequestHandles.get(currentRequestIndex++);
      try {
        currentHandle.doHandle(invokeProxyContext, this);
      } catch(Exception e) {
        this.buildException(invokeProxyContext, e);
        currentRequestIndex = maxRequestHandlesSize;
        this.doHandle(invokeProxyContext, ChainLogic.CONTINUE);
      }
      this.throwException(context);
      return;
    }
    // 1.b、如果条件成立，说明处理request链的处理过程中遇到了Break标记，那么直接处理response链
    else if(currentRequestIndex < maxRequestHandlesSize && chainLogic == ChainLogic.BREAK) {
      currentRequestIndex = maxRequestHandlesSize;
      this.doHandle(invokeProxyContext, ChainLogic.CONTINUE);
      this.throwException(context);
      return;
    }
    
    // 2、=========
    // 如果条件成立，则说明需要接着/继续处理response链
    if(currentResponseIndex < maxResponseHandlesSize && currentRequestIndex >= maxRequestHandlesSize && chainLogic == ChainLogic.CONTINUE) {
      InvokeResponseHandle currentHandle = currentInvokeResponseHandles.get(currentResponseIndex++);
      // 如果条件成立，说明之前处理request链出现了异常，且当前response节点不能在有异常的情况下处理，则跳过
      if(invokeProxyContext.isException() && currentHandle.handleException(invokeProxyContext)) {
        currentHandle.doHandle(invokeProxyContext, this);
        this.throwException(context);
        return;
      } else if(invokeProxyContext.isException() && !currentHandle.handleException(invokeProxyContext)) {
        this.doHandle(context, chainLogic);
        this.throwException(context);
        return;
      }
      
      // 执行response的处理节点
      try {
        currentHandle.doHandle(invokeProxyContext, this);
      } catch(Exception e) {
        // 如果有异常，那么在这里进行记录，之前request链的异常暂时可以丢弃，帮助开发者先行修复response链上的异常
        this.buildException(invokeProxyContext, e);
        throw new InvokeProxyException(invokeProxyContext, e);
      }
    }
    this.throwException(context);
    return;
  }
  
  void setInvokeRequestHandles(List<InvokeRequestHandle> currentInvokeRequestHandles) {
    this.currentInvokeRequestHandles = currentInvokeRequestHandles;
  }
  
  void setInvokeResponseHandles(List<InvokeResponseHandle> currentInvokeResponseHandles) {
    this.currentInvokeResponseHandles = currentInvokeResponseHandles;
  }
  
  void setInvokeProxyContext(InvokeProxyContext invokeProxyContext) {
    this.invokeProxyContext = invokeProxyContext;
  }
}