package com.bizunited.platform.core.service.invoke;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

/**
 * 服务源调用代理器。服务源调用代理器不允许直接通过new完成初始化，只能通过ServicableProxy.build进行初始化。
 * @author yinwenjie
 */
public class InvokeProxy implements HandleChain {
  
  /**
   * 被代理的真实执行器对象
   */
  private HandleChain targetHandleChain;
  /**
   * 本次调用的上下文对象
   */
  private InvokeProxyContext invokeProxyContext;
  
  /**
   * 代理调用方法，其中的params参数信息将在正式调用前被事先写入到调用器的上下文中
   * @param params K-V结构中的调用入参
   * @throws InvokeProxyException
   */
  public Object doHandle(Map<String , Object> params) throws InvokeProxyException {
    Validate.notNull(this.targetHandleChain , "错误的代理目标，请检查!!");
    Validate.notNull(this.invokeProxyContext , "错误的上下文信息，请检查!!");
    invokeProxyContext.setChainParams(params);
    this.targetHandleChain.doHandle(this.invokeProxyContext, HandleChain.ChainLogic.CONTINUE);
    return this.invokeProxyContext.getResult();
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.invoke2.HandleChain#doHandle(com.bizunited.platform.core.service.invoke2.invokeProxyContext, com.bizunited.platform.core.service.invoke2.HandleChain.ChainLogic)
   */
  @Override
  public void doHandle(InvokeProxyContext context, ChainLogic chainLogic) throws InvokeProxyException {
    throw new IllegalArgumentException("不允许在服务过程代理器中直接调用doHandle(invokeProxyContext, ChainLogic)方法;");
  }

  /**
   * 典型的创建者模式，用来保证ServicableProxy唯一实例创建时的内聚性
   * @author yinwenjie
   */
  public static class Build {
    private List<InvokeRequestHandle> invokeRequestHandles = new LinkedList<>();
    private List<InvokeResponseHandle> invokeResponseHandles = new LinkedList<>();
    /**
     * 被代理的真实执行器的类型，目前只有两种：DefaultHandleChain和GroovyScriptHandleChain
     */
    private Class<? extends HandleChain> targetChainType;
    
    /**
     * 该线程中使用的类加载器
     */
    private ClassLoader classLoader;
    /**
     * 这里的k-v参数信息，将在执行build方法时，写入代理器上下文的
     */
    private Map<String, Object> params = new HashMap<>();
    /**
     * 通过Builder构建context上下文以及proxy
     */
    public InvokeProxy build() {
      /*
       * 构造过程如下：
       * 1、首先监测各个需要设定的参数是否都已经完成最小化设定
       * 2、如果当前具体的处理链管理器为DefaultHandleChain则进行以下初始化动作
       *   2.1、新建一个DefaultHandleChain实例
       *   2.2、对DefaultHandleChain中的各个属性进行赋值
       * 3、TODO 目前看来只需要一个执行链管理器，后续视情况在重新使用GroovyScriptHandleChain
       * */
      // 1、=======
      Validate.isTrue(invokeRequestHandles != null && !invokeRequestHandles.isEmpty() , "调用器构建时，至少要有一个请求处理节点（InvokeRequestFilter）" );
      Validate.isTrue(invokeResponseHandles != null && !invokeResponseHandles.isEmpty() , "调用器构建时，至少要有一个响应处理节点（InvokeResponseFilter）");
      Validate.notNull(classLoader , "调用器构建时，需要指定处理器内部加载class时，使用的类加载器");
      if(this.targetChainType == null) {
        targetChainType = DefaultHandleChain.class;
      }
      Validate.isTrue(this.targetChainType != InvokeProxy.class , "具体处理链管理器不能指定为InvokeProxy.class");
      
      // 2、========
      HandleChain handleChain = null;
      InvokeProxyContext invokeProxyContext = null;
      if(this.targetChainType == DefaultHandleChain.class) {
        DefaultHandleChain defaultHandleChain = new DefaultHandleChain();
        defaultHandleChain.setInvokeRequestHandles(this.invokeRequestHandles);
        defaultHandleChain.setInvokeResponseHandles(this.invokeResponseHandles);
        // 设定上下文
        invokeProxyContext = new InvokeProxyContext();
        invokeProxyContext.setClassLoader(classLoader);
        if(this.params != null && !this.params.isEmpty()) {
          invokeProxyContext.setChainParams(this.params);
        }
        defaultHandleChain.setInvokeProxyContext(invokeProxyContext);
        handleChain = defaultHandleChain;
      }
      Validate.notNull(handleChain , "未匹配的HandleChain实现类");
      
      // 建立代理器
      InvokeProxy invokeProxy = new InvokeProxy();
      invokeProxy.targetHandleChain = handleChain;
      invokeProxy.invokeProxyContext = invokeProxyContext;
      return invokeProxy;
    }
    
    /**
     * 设置类加载器
     * @param classLoader 
     * @return 
     */
    public Build addClassLoader(ClassLoader classLoader) {
      this.classLoader = classLoader;
      return this;
    }
    
    /**
     * 该方法在代理器构建设置阶段，将执行链上下文需要的自定义变量信息预先写入
     * @param key 上下文变量的key（如果重复将被覆盖）
     * @param value 上下文变量的内容，不能为null，否则不能写入
     * @return 
     */
    public Build addChainParam(String key , Object value) {
      if(StringUtils.isBlank(key) || value == null) {
        return this;
      }
      this.params.put(key, value);
      return this;
    }
    
    /**
     * 添加response结果处理链
     * @param responses 
     * @return
     */
    public Build addInvokeResponseFilter(InvokeResponseHandle ...responses) {
      Arrays.stream(responses).forEach(filter -> invokeResponseHandles.add(filter));
      return this;
    }
    
    /**
     * 添加request请求处理链
     * @param requests 
     * @return
     */
    public Build addInvokeRequestFilter(InvokeRequestHandle ...requests) {
      Arrays.stream(requests).forEach(filter -> invokeRequestHandles.add(filter));
      return this;
    }
    
    /**
     * 设置被代理的真实执行器的类型，目前只有两种：DefaultHandleChain和GroovyScriptHandleChain
     * @param targetChainType  
     * @return 
     */
    public Build setTargetHandleChain(Class<? extends HandleChain> targetChainType) {
      this.targetChainType = targetChainType;
      return this;
    }
  }
}