package com.bizunited.platform.core.service.scheduler;

import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.DynamicTaskSchedulerService;
import com.bizunited.platform.rbac.server.service.redis.RedisMutexService;

/**
 * 动态调度任务的执行权抢占工作者，这是为了保证整个业务节点的服务集群只会有一个节点运行工作器
 * @author yinwenjie
 */
@Component("_DynamicTaskNotifyMaster")
public class DynamicTaskNotifyMaster {
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private DynamicTaskSchedulerService dynamicTaskSchedulerService;
  /**
   * 动态任务执行器，为了避免多节点执行所使用的锁名字
   */
  @Value("${scheduler.lockKey:_REDIS_LOCK_KEY_DYNAMICTASK}")
  private String dynamicTaskLockKey;
  /**
   * 该变量描述当前进程是否有线程正在执行动态调度任务执行权限的抢占工作
   */
  private static boolean notifyingMaster = false;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicTaskNotifyMaster.class);
  /**
   * 动态定时任务初始化加载器：只有负责任务运行的master节点，才能完成加载。
   * 其它slave节点如果调用该方法，那么初始化动作将进入阻塞状态。
   */
  @Async
  public void notifyMaster() {
    if(notifyingMaster) {
      return;
    }
    // 该方法只能被调用一次。
    int j = 0;
    while(true) {
      if(j++ == Integer.MAX_VALUE) {
        break;
      }
      synchronized (DynamicTaskSchedulerService.class) {
        if(notifyingMaster) {
          return;
        }
        notifyingMaster = true;
        break;
      }
    }
    
    /*
     *  - 该方法是一个不会退出的方法，在该方法中根据是否获得了特定的分布式锁的情况，决定动态任务在本进程中的加载情况：</br>
     * 1、如果当前取得了特定的分布式锁，则进行动态任务在本进程的初始化工作，如何进行初始化，请参见init()方法中的描述
     *  - 这里注意一点：install()方法是周期性执行且不会退出的，这是为了将那些手动变更运行状态的动态任务，增加或者剔除出本进程；</p>
     * 
     * 2、如果当前线程未取得特定的分布式锁，那么将调用uninstall方法，将可能存在于本进程中的动态任务全部剔除出内存（但是无需变更其数据库状态）
     *  - 完成后继续以无限循环的方式，尝试获得分布式锁。
     */
    int modCount = 0;
    while(true) {
      if(modCount++ == Integer.MAX_VALUE) {
        break;
      }
      
      boolean succLock = false;
      try {
        // 1、==============
        // 试图获取锁的时间为10秒。如果条件成立，说明已经获取到锁
        // 轮询时间为30秒（暂时先不放给开发人员）
        succLock = this.redisMutexService.tryLock(dynamicTaskLockKey, TimeUnit.SECONDS, 10);
        if(succLock) {
          synchronized (this) {
            while(true) {
              if(modCount++ == Integer.MAX_VALUE) {
                return;
              }
              
              this.dynamicTaskSchedulerService.loading();
              this.wait(30000l);
            } 
          } 
        } 
        // 2、==============
        // 如果条件成立，说明在规定时间内，没有获取到锁
        else {
          this.dynamicTaskSchedulerService.unloading();
        }
      } catch (Exception e) {
        LOGGER.error(e.getMessage() , e);
      } finally {
        if(succLock) {
          this.redisMutexService.unlock(dynamicTaskLockKey);
        }
      }
    } 
  } 
}