package com.bizunited.platform.core.service.scheduler.handle;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.HandleChain;
import com.bizunited.platform.core.service.invoke.InvokeProxyContext;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.invoke.InvokeRequestHandle;
import com.bizunited.platform.core.service.invoke.HandleChain.ChainLogic;

/**
 * 该定义的处理节点，用于在动态任务功能中，调用指定的运行脚本
 * @author yinwenjie
 */
@Component("DynamicTaskInvokeHandle")
public class DynamicTaskInvokeHandle implements InvokeRequestHandle {
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private ApplicationContext applicationContext;
  
  @Override
  public void doHandle(InvokeProxyContext context, HandleChain chain) throws InvokeProxyException {
    /*
     * 由于2019年11月28日，对业务构建平台的定时器功能进行了需求调整，所以该类的处理过程调整如下：
     * 1、首先判定任务信息本身的合规性
     * 2、如果当前任务是一个Groovy脚本性质的执行任务，则在加载个参数后，使用scriptService.invoke进行执行
     * 如果当前任务是一个托管到Spring的使用DynamicTaskService注解定义的指定方法，
     * 则采用spring上下文加载后，通过反射调用进行执行（请注意执行时方法本身的入参说明）
     */
    
    // 1、=======
    DynamicTaskSchedulerEntity currentTask = (DynamicTaskSchedulerEntity)context.getChainParam("currentTask");
    Validate.notNull(currentTask , "在进行动态任务执行时，未发现指定的动态调用任务!");
    Map<String, Object> parmas = new HashMap<>();
    parmas.put("currentTask", currentTask);
    
    // 2、======
    // 如果条件成立，说明是执行一个groovy脚本性质的定时任务
    if(currentTask.getInvokeType() == 1) {
      this.invokeGroovyScript(currentTask, context, parmas);
    } 
    // 如果条件成立，说明是执行一个基于注解的java method定时任务
    else if(currentTask.getInvokeType() == 2) {
      this.invokeMethod(currentTask, context, parmas);
    } else {
      // 其它情况抛出异常
      throw new InvokeProxyException(context, "错误的执行脚本类型");
    }
    
    chain.doHandle(context, ChainLogic.CONTINUE);
  }
  
  /**
   * 该死有方法用于执行基于DynamicTaskService注解的java method方法
   */
  private void invokeMethod(DynamicTaskSchedulerEntity currentTask , InvokeProxyContext context , Map<String, Object> parmas) throws InvokeProxyException {
    /*
     * 处理过程为：
     * 1、首先拿到currentTask之中的基本信息，并判定有效性
     * 2、然后根据其中的得到的bean的名字、方法的名字，开始进行调用
     * 注意其中的传参要求：如果其中要求了InvokeProxyContext参数，还要拼凑参数进行传入
     * */
    // 1、=======
    String taskCode = currentTask.getTaskCode();
    String invokeBeanName = currentTask.getInvokeBeanName();
    Validate.notBlank(invokeBeanName , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeBeanName信息非法" , taskCode);
    String invokeMethod = currentTask.getInvokeMethod();
    Validate.notBlank(invokeMethod , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeMethod信息非法" , taskCode);
    boolean proxyContextParam = currentTask.getProxyContextParam();
    Object currentBean = null;
    try {
      currentBean = applicationContext.getBean(invokeBeanName);
    } catch(NoSuchBeanDefinitionException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean，请检查" , invokeBeanName));
    }
    
    // 2、=======
    Class<?> beanClass = applicationContext.getType(invokeBeanName);
    Method beanClassMethod = null;
    // 开始进行调用
    try {
      if(proxyContextParam) {
        beanClassMethod = beanClass.getMethod(invokeMethod, new Class[] {InvokeProxyContext.class});
        beanClassMethod.invoke(currentBean, new Object[] {context});
      } else {
        beanClassMethod = beanClass.getMethod(invokeMethod, new Class[] {});
        beanClassMethod.invoke(currentBean, new Object[] {});
      }
    } catch(NoSuchMethodException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean中指定的方法%s，请检查" , invokeBeanName , invokeMethod));
    } catch(InvocationTargetException | IllegalAccessException e) {
      throw new IllegalArgumentException("执行动态调用任务时,方法执行错误，请检查" , e);
    }
  }
  
  /**
   * 该私有方法用于执行Groovy脚本性质的定时器任务
   */
  private void invokeGroovyScript(DynamicTaskSchedulerEntity currentTask , InvokeProxyContext context , Map<String, Object> parmas) throws InvokeProxyException {
    // 脚本信息
    String scriptName = currentTask.getScriptName();
    ScriptEntity scriptEntity = this.scriptService.findByName(scriptName);
    Validate.notNull(scriptEntity , "在进行动态任务执行时，未发现指定的groovy脚本信息，请检查%s!" , currentTask.getId());
    String scriptId = scriptEntity.getId();
    // 可传入的信息包括：
    // context上下文中的chainParams信息
    Set<String> chainParamsKeys = context.getParamKeys();
    for (String chainParamsKey : chainParamsKeys) {
      parmas.put(chainParamsKey, context.getChainParam(chainParamsKey));
    }
    // spring上下文对象
    Map<String, Object> returnResults = null;
    returnResults = this.scriptService.invoke(scriptId, parmas);
    if(returnResults != null && !returnResults.isEmpty()) {
      context.setChainParams(returnResults);
    }
  }
}