package com.bizunited.platform.core.service.security;

import com.bizunited.platform.core.common.enums.CompetenceTypeEnum;
import com.bizunited.platform.core.entity.ButtonEntity;
import com.bizunited.platform.core.repository.ButtonRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.ButtonService;
import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.ButtonVo;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 按钮功能在服务中提供的基本操作，主要为了和角色，接口进行绑定
 * @author: zengxingwang
 * @date: 2020/1/13 14:44
 */
public class ButtonServiceImpl implements ButtonService {

  /**
   * 可忽略方法级判定的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles;

  @Autowired
  private ButtonRepository buttonRepository;
  @Autowired
  @Lazy
  private CompetenceService competenceService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RoleService roleService;
  @Autowired
  private UserService userService;

  /**
   * 建立指定按钮和指定接口的绑定关系
   * @param buttonId
   * @param competenceIds
   */
  @Override
  @Transactional
  public void bindCompetences(String buttonId, String[] competenceIds) {
    Validate.notBlank(buttonId, "进行按钮和接口绑定时，必须指定当前按钮编号!!");
    Optional<ButtonEntity> op = this.buttonRepository.findById(buttonId);
    Validate.isTrue(op.isPresent(), "没有发现指定的按钮信息");
    Validate.notEmpty(competenceIds, "进行按钮和接口绑定时，必须至少指定一个按钮编号!!");
    for (String competenceId : competenceIds) {
      CompetenceVo opCompetence = this.competenceService.findById(competenceId);
      Validate.notNull(opCompetence, "没有发现指定的接口信息[%s]，请检查!!", competenceId);
      Validate.isTrue(CompetenceTypeEnum.COMPETENCE_VIEW_ITEM_ZERO.getType().equals(opCompetence.getViewItem()), "只能绑定功能接口");
      long count = this.buttonRepository.countByCompetenceIdAndButtonId(buttonId, competenceId);
      if(count == 0L) {
        this.buttonRepository.bindCompetence(buttonId, competenceId);
      }
    }
  }

  /**
   * 取消指定按钮和指定接口的绑定关系
   * @param buttonId
   * @param competenceIds
   */
  @Override
  @Transactional
  public void unbindCompetences(String buttonId, String[] competenceIds) {
    Validate.notBlank(buttonId, "进行按钮和接口解绑时，必须指定当前按钮编号!!");
    Optional<ButtonEntity> op = this.buttonRepository.findById(buttonId);
    Validate.isTrue(op.isPresent(), "没有发现指定的按钮信息");
    Validate.notEmpty(competenceIds, "进行按钮和接口解绑时，必须至少指定一个按钮编号!!");
    for (String competenceId : competenceIds) {
      CompetenceVo opCompetence = this.competenceService.findById(competenceId);
      Validate.notNull(opCompetence, "没有发现指定的接口信息[%s]，请检查!!", competenceId);
      Validate.isTrue(CompetenceTypeEnum.COMPETENCE_VIEW_ITEM_ZERO.getType().equals(opCompetence.getViewItem()), "只能解绑功能接口");
      this.buttonRepository.unbindCompetence(buttonId, competenceId);
    }
  }

  /**
   * 根据按钮id查询按钮信息
   * @param buttonId
   * @return
   */
  @Override
  public ButtonVo findDateilsById(String buttonId){
    if(StringUtils.isBlank(buttonId)){
      return null;
    }
    ButtonEntity button = this.buttonRepository.findDateilsById(buttonId);
    if(button != null){
      return this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, HashSet.class, ArrayList.class, "competences", "competence", "roles");
    }
    return null;
  }

  /**
   * 根据用户ID查询用户的按钮权限
   * @param userId
   * @return
   */
  @Override
  public Set<ButtonVo> findByUserId(String userId) {
    if(StringUtils.isBlank(userId)){
      return Sets.newHashSet();
    }
    Set<ButtonVo> buttonVos = new HashSet<>();
    List<RoleVo> roleVos = roleService.findAllByUserId(userId, 0);
    for(RoleVo role : roleVos){
      Set<ButtonEntity> buttons = this.buttonRepository.findByRoleId(role.getId());
      if(!CollectionUtils.isEmpty(buttons)){
        Collection<ButtonVo> buttonVo = this.nebulaToolkitService.copyCollectionByWhiteList(buttons, ButtonEntity.class, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
        buttonVos.addAll(buttonVo);
      }
    }
    return this.distinctByCode(buttonVos);
  }

  @Override
  @Transactional
  public ButtonVo create(ButtonVo button) {
    //入参校验
    this.createValidation(button);
    ButtonEntity currentButton = nebulaToolkitService.copyObjectByWhiteList(button, ButtonEntity.class, HashSet.class, ArrayList.class, "competence");
    Date nowDate = new Date();
    currentButton.setCreateTime(nowDate);
    currentButton.setModifyTime(nowDate);
    currentButton.setSystem(false);
    buttonRepository.save(currentButton);
    button.setId(currentButton.getId());
    return button;
  }

  /**
   * 批量创建按钮
   * @param buttons
   * @return
   */
  @Override
  @Transactional
  public void batchCreate(Set<ButtonVo> buttons){
    Validate.notEmpty(buttons, "批量创建按钮时，必须至少有一个按钮!!");
    Set<ButtonEntity> buttonSet = new HashSet<>();
    Set<String> codes = new HashSet<>();
    for(ButtonVo buttonVo : buttons){
      //入参校验
      this.createValidation(buttonVo);
      //判断是否重复
      Validate.isTrue(!codes.contains(buttonVo.getCode()), "传入按钮数据有重复，请检查！");
      codes.add(buttonVo.getCode());

      ButtonEntity currentButton = nebulaToolkitService.copyObjectByWhiteList(buttonVo, ButtonEntity.class, HashSet.class, ArrayList.class, "competence");
      currentButton.setCreateTime(new Date());
      currentButton.setModifyTime(new Date());
      currentButton.setSystem(false);
      buttonSet.add(currentButton);
    }
    buttonRepository.saveAll(buttonSet);
  }

  @Override
  @Transactional
  public ButtonVo update(ButtonVo button) {
    //入参校验
    this.updateValidation(button);
    ButtonEntity updateButton = buttonRepository.findById(button.getId()).orElse(null);
    Validate.notNull(updateButton, "未在数据层找到对应的按钮信息!!");
    Validate.isTrue(!updateButton.getSystem(), "系统按钮不能更新");
    if (button.getButtonDesc() != null){
      updateButton.setButtonDesc(button.getButtonDesc());
    }
    updateButton.setName(button.getName());
    updateButton.setCode(button.getCode());
    updateButton.setModifyTime(new Date());
    buttonRepository.save(updateButton);
    return this.nebulaToolkitService.copyObjectByWhiteList(updateButton, ButtonVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notNull(id, "删除必须要给定主键信息!");
    ButtonEntity button = buttonRepository.findById(id).orElse(null);
    if(button == null) {
      return;
    }
    Validate.isTrue(!button.getSystem(), "系统按钮不能删除");
    buttonRepository.unbindRoleById(id);
    buttonRepository.unbindCompetenceById(id);
    buttonRepository.delete(button);
  }

  /**
   * 根据id批量删除按钮
   * @param ids
   */
  @Override
  @Transactional
  public void batchDelete(String[] ids){
    Validate.notEmpty(ids, "批量删除按钮时，必须至少指定一个按钮id!!");
    for(String id : ids){
      this.deleteById(id);
    }
  }

  @Override
  public Set<ButtonVo> findByCompetenceId(String competenceId) {
    if (StringUtils.isBlank(competenceId)){
      return Sets.newHashSet();
    }
    Set<ButtonEntity> buttons = buttonRepository.findByCompetenceId(competenceId);
    if (CollectionUtils.isEmpty(buttons)){
      return Sets.newHashSet();
    }
    Collection<ButtonVo> buttonVos = nebulaToolkitService.copyCollectionByWhiteList(buttons, ButtonEntity.class, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newLinkedHashSet(buttonVos);
  }

  @Override
  public Set<ButtonVo> findByRoleCode(String roleCode) {
    if (StringUtils.isBlank(roleCode)){
      return Sets.newHashSet();
    }
    Set<ButtonEntity> buttons = buttonRepository.findByRoleCode(roleCode);
    if (CollectionUtils.isEmpty(buttons)){
      return Sets.newHashSet();
    }
    Collection<ButtonVo> buttonVos = nebulaToolkitService.copyCollectionByWhiteList(buttons, ButtonEntity.class, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(buttonVos);
  }

  @Override
  public Set<ButtonVo> findByCurrentUser(Principal principal) {
    String account = principal.getName();
    Validate.notBlank(account, "未获取到登录账号信息！");
    UserVo user = userService.findByAccount(account);
    Validate.notNull(user, "未获取到用户信息：%s", account);
    List<RoleVo> roles = roleService.findAllByUserId(user.getId(), 0);
    Validate.notEmpty(roles, "当前登录用户未配置角色！");
    List<String> roleNames = roles.stream().map(RoleVo::getRoleName).collect(Collectors.toList());
    Sets.SetView<String> intersections = Sets.intersection(Sets.newHashSet(ignoreMethodCheckRoles), Sets.newHashSet(roleNames));
    Set<ButtonVo> buttonVos = new HashSet<>();
    if(!CollectionUtils.isEmpty(intersections)) {
      // 包含忽略权限的角色
      List<ButtonEntity> buttons = buttonRepository.findAll();
      Collection<ButtonVo> vos = nebulaToolkitService.copyCollectionByWhiteList(buttons, ButtonEntity.class, ButtonVo.class, HashSet.class, ArrayList.class);
      buttonVos.addAll(vos);
      return buttonVos;
    }
    // 查询用户所有角色的按钮权限
    for (RoleVo role : roles) {
      Set<ButtonEntity> buttons = buttonRepository.findByRoleId(role.getId());
      Collection<ButtonVo> vos = nebulaToolkitService.copyCollectionByWhiteList(buttons, ButtonEntity.class, ButtonVo.class, HashSet.class, ArrayList.class);
      buttonVos.addAll(vos);
    }
    return this.distinctByCode(buttonVos);
  }

  /**
   * 对按钮按照编码进行去重
   * @param buttons
   * @return
   */
  private Set<ButtonVo> distinctByCode(Set<ButtonVo> buttons) {
    if(CollectionUtils.isEmpty(buttons)) {
      return buttons;
    }
    Set<String> buttonCodes = new HashSet<>();
    Iterator<ButtonVo> iterator = buttons.iterator();
    while (iterator.hasNext()) {
      ButtonVo button = iterator.next();
      if(!buttonCodes.add(button.getCode())) {
        iterator.remove();
      }
    }
    return buttons;
  }

  /**
   * 修改-参数校验
   * @param button
   *
   */
  private void updateValidation(ButtonVo button) {
    Validate.notBlank(button.getCode(), "按钮编码不能为空，请检查!");
    Validate.notBlank(button.getName(), "按钮名称不能为空，请检查!");
    Validate.notBlank(button.getId(), "修改数据必须要有ID");
    //验证按钮编码是否重复
    long countByCode = buttonRepository.countByCodeWithoutId(button.getCode(), button.getId());
    Validate.isTrue(countByCode == 0L, "按钮的编码重复，请检查!");
  }

  /**
   * 新增-参数校验
   * @param button
   *
   */
  private void createValidation(ButtonVo button) {
    Validate.notNull(button, "保存的按钮对象不能为空!");
    Validate.notNull(button.getCompetence(), "菜单对象为空，请检查!");
    Validate.notBlank(button.getCompetence().getId(), "菜单ID不能为空");
    Validate.notBlank(button.getCode(), "按钮编码不能为空，请检查!");
    Validate.notBlank(button.getName(), "按钮名称不能为空，请检查!");
    Validate.isTrue(button.getId() == null, "创建数据不能有ID");
    //验证是否是菜单
    CompetenceVo competenceVo = competenceService.findById(button.getCompetence().getId());
    Validate.notNull(competenceVo, "未查询到该菜单!");
    Validate.isTrue(competenceVo.getViewItem() == 1, "当前功能必须是菜单!");
    //验证按钮编码是否重复
    long countByCode = buttonRepository.countByCode(button.getCode());
    Validate.isTrue(countByCode == 0L, "按钮的编码重复，请检查!");
  }
}
