package com.bizunited.platform.core.service.security;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.enums.NormalStatusEnum;
import com.bizunited.platform.core.common.enums.RbacRelationEnum;
import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.RbacSettingEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.OrganizationRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * @author yinwenjie ycx
 */
public class OrganizationServiceImpl implements OrganizationService {

  private static final String ERROR_MESS_OGS = "没有找到指定的组织机构";
  private static final String MESS_ORG_NAME = "orgName";
  private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationServiceImpl.class);

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private OrganizationRepository organizationRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private RbacSettingService rbacSettingService;


  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#create(com.bizunited.platform.rbac.server.vo.OrganizationVo)
   */
  @Override
  @Transactional
  public OrganizationVo create(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(organization.getId()), "添加组织机构时，不能传入组织机构id,请重新添加!!");
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    Validate.notBlank(organization.getType(), "组织机构类型不能为空，请检查!!");
    String code = organization.getCode();
    Validate.notNull(code, "组织机构编号不能为空");
    OrganizationEntity oldOrg = organizationRepository.findByCode(code);
    Validate.isTrue(null == oldOrg, "存在重复的code,请重新输入组织机构编码!!");

    // 转换后保存
    OrganizationEntity currentOrg = this.nebulaToolkitService.copyObjectByWhiteList(organization, OrganizationEntity.class, HashSet.class, ArrayList.class);
    //新增如果存在父级组织机构
    OrganizationVo organizationParent = organization.getParent();
    if (organizationParent != null ){
      Validate.notBlank(organizationParent.getId(), "当前组织机构父级组织机构的id不能为空，请检查!!");
      OrganizationEntity organizationEntity = this.nebulaToolkitService.copyObjectByWhiteList(organizationParent, OrganizationEntity.class, HashSet.class, ArrayList.class);
      currentOrg.setParent(organizationEntity);
    }
    organizationRepository.save(currentOrg);
    organization.setId(currentOrg.getId());
    return organization;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#update(com.bizunited.platform.rbac.server.vo.OrganizationVo)
   */
  @Override
  @Transactional
  public OrganizationVo update(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.notBlank(organization.getId(), "修改组织机构时，id不能为空,请重新添加!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(organization.getId());
    OrganizationEntity currentOrg = op.orElse(null);
    Validate.notNull(currentOrg, "未在数据层找到对应的组织结构信息!!");


    // 修改前验证和赋值
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    if (StringUtils.isNotBlank(organization.getType())) {
      currentOrg.setType(organization.getType());
    }
    currentOrg.setOrgName(orgName);
    currentOrg.setDescription(organization.getDescription());
    organizationRepository.save(currentOrg);

    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public OrganizationVo updateStatus(String orgId) {
    Validate.notBlank(orgId, "组织机构id不能为空，请检查");
    Optional<OrganizationEntity> op = organizationRepository.findById(orgId);
    OrganizationEntity organization = op.orElse(null);
    Validate.notNull(organization, "没有该组织机构，请检查!!");

    Integer status = organization.getTstatus();
    // 0:禁用,1:启用
    status = (status == NormalStatusEnum.ENABLE.getStatus() ?
            NormalStatusEnum.DISABLE.getStatus() : NormalStatusEnum.ENABLE.getStatus());
    organization.setTstatus(status);
    organizationRepository.save(organization);
    this.updateChildrenStatus(organization, status);
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(organization, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 启用或禁用时，将子级也同时启用或禁用
   * @param organization
   * @param status
   */
  private void updateChildrenStatus(OrganizationEntity organization, Integer status) {
    Validate.notNull(organization, "未在数据层找到对应的组织结构信息!!");
    List<OrganizationEntity> organizations = this.organizationRepository.findByParent(organization.getId());
    if (CollectionUtils.isEmpty(organizations)){
      return;
    }
    for (OrganizationEntity child:organizations) {
      child.setTstatus(status);
      this.organizationRepository.save(child);
      this.updateChildrenStatus(child, status);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#bindParent(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindParent(String currentOrgId, String parentOrgId) {
    Validate.notBlank(currentOrgId, "指定的组织机构必须传入!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.notBlank(parentOrgId, "指定的父级组织机构必须传入!!");
    Validate.isTrue(!StringUtils.equals(currentOrgId, parentOrgId), "当前组织机构不能与父级组织机构重复");
    Optional<OrganizationEntity> optional = this.organizationRepository.findById(parentOrgId);
    Validate.isTrue(optional.isPresent(), "没有找到指定的父级组织机构!!");
    Validate.isTrue(!StringUtils.equals(currentOrgId, parentOrgId), "绑定组织机构是，不能自己绑定自己!!");
    this.organizationRepository.bindParent(currentOrgId, parentOrgId);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindParent(java.lang.String)
   */
  @Override
  @Transactional
  public void unbindParent(String currentOrgId) {
    Validate.notBlank(currentOrgId, "进行绑定结构消除时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    this.organizationRepository.unbindParent(currentOrgId);
  }
  /**
   * 验证绑定用户和组织机构
   * @param currentOrgId
   * @param userId
   * @param rbacSettingEntity
   */
  private void validateBindUser(RbacSettingEntity rbacSettingEntity, String currentOrgId, String userId){
    Optional<UserEntity> optional = userRepository.findById(userId);
    Validate.isTrue(optional.isPresent(), "没有该用户[%s],请检查!!", userId);
    if (rbacSettingEntity == null) {
      long count = organizationRepository.countByOrgIdAndUserId(currentOrgId, userId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = rbacSettingEntity.getUserOrgRelation();
      //如果用户和组织绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = organizationRepository.countByOrgIdAndUserId(currentOrgId, userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = organizationRepository.countOrganizationByUserId(userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织多对一，指定用户[%s]已经绑定了组织，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = organizationRepository.countUserByOrgId(currentOrgId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织一对多，指定组织[%s]已经绑定了用户，不能继续绑定，请检查!!", currentOrgId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countOrganization = organizationRepository.countOrganizationByUserId(userId);
        Validate.isTrue(countOrganization == 0, "在批量绑定时,用户和组织一对一，指定用户[%s]已经绑定了组织，不能继续绑定，请检查!!", userId);
        long countUser = organizationRepository.countUserByOrgId(currentOrgId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和组织一对一，指定组织[%s]已经绑定了用户，不能继续绑定，请检查!!", currentOrgId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#bindUsers(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员绑定时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.isTrue(userIds != null && userIds.length > 0, "进行组织机构-人员绑定时,至少传入一个人员编号信息!!");
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    for (String userId : userIds) {
      this.validateBindUser(rbacSettingEntity, currentOrgId, userId);
      this.organizationRepository.bindUsers(currentOrgId, userId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindUser(java.lang.String)
   */
  @Override
  @Transactional
  public void unbindUser(String userId) {
    Validate.notBlank(userId, "进行组织机构-人员解绑绑定时，必须指定当前的人员信息!!");
    Optional<UserEntity> op = this.userRepository.findById(userId);
    Validate.isTrue(op.isPresent(), "进行组织机构-人员解绑绑定时，未找到指定的人员信息【id=%s】" , userId);
    this.organizationRepository.unbindUser(userId);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unbindUser(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员解绑时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.notNull(userIds, "进行组织机构-人员解绑时，必须指定当前的人员信息!!");
    for (String userId : userIds) {
      Optional<UserEntity> optional = this.userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "进行组织机构-人员解绑绑定时，未找到指定的人员信息【id=%s】" , userId);
      this.organizationRepository.unbindUser(currentOrgId, userId);
    }
  }

  /**
   * 验证绑定角色和组织机构
   * @param currentOrgId
   * @param roleId
   * @param rbacSettingEntity
   */
  private void validateBindRole(RbacSettingEntity rbacSettingEntity, String currentOrgId, String roleId){
    Optional<RoleEntity> optional = roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "没有该角色[%s],请检查!!", roleId);
    if (rbacSettingEntity == null) {
      long count = organizationRepository.countByOrgAndRoleId(currentOrgId, roleId);
      Validate.isTrue(count == 0, "在批量绑定时,指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
    } else {
      int bindRelation = rbacSettingEntity.getRoleOrgRelation();
      //如果用户和组织绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = organizationRepository.countByOrgAndRoleId(currentOrgId, roleId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和组织多对多，指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = organizationRepository.countOrganizationByRoleId(roleId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和组织多对一，指定角色[%s]已经绑定了组织，不能继续绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = organizationRepository.countRoleByOrgId(currentOrgId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和组织一对多，指定组织[%s]角色，不能继续绑定，请检查!!", currentOrgId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countOrganization = organizationRepository.countOrganizationByRoleId(roleId);
        Validate.isTrue(countOrganization == 0, "在批量绑定时,角色和组织一对一，指定角色[%s]已经绑定了组织，不能继续绑定，请检查!!", roleId);
        long countRole = organizationRepository.countRoleByOrgId(currentOrgId);
        Validate.isTrue(countRole == 0, "在批量绑定时,角色和组织一对一，指定组织[%s]角色，不能继续绑定，请检查!!", currentOrgId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#bindRoles(java.lang.String, java.lang.String[])
   */
  @Transactional
  @Override
  public void bindRoles(String currentOrgId, String[] roleIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-角色绑定时，必须指定当前的组织机构信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行组织机构-角色绑定时,至少传入一个角色编号信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    for (String roleId : roleIds) {
      this.validateBindRole(rbacSettingEntity, currentOrgId, roleId);
      this.organizationRepository.bindRole(currentOrgId, roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindRole(java.lang.String[])
   */
  @Transactional
  @Override
  public void unbindRole(String[] roleIds) {
    Validate.notNull(roleIds, "进行组织机构-角色解绑绑定时，必须指定当前的角色信息!!");
    for (String roleId : roleIds) {
      Optional<RoleEntity> optional = this.roleRepository.findById(roleId);
      Validate.isTrue(optional.isPresent(), "进行组织机构-角色解绑绑定时，未找到指定的角色信息【id=%s】" , roleId);
      this.organizationRepository.unbindRole(roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByNullParent()
   */
  @Override
  public List<OrganizationVo> findByNullParent() {
    List<OrganizationEntity> organizationEntitys = this.organizationRepository.findByNullParent();
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByParent(java.lang.String)
   */
  @Override
  public List<OrganizationVo> findByParent(String parentId) {
    if (StringUtils.isBlank(parentId)) {
      return Lists.newArrayList();
    }
    List<OrganizationEntity> organizationEntitys = this.organizationRepository.findByParent(parentId);
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByNullParentAndStatus(java.lang.Integer)
   */
  @Override
  public List<OrganizationVo> findByNullParentAndStatus(Integer status) {
    if (status == null) {
      return Collections.emptyList();
    }
    List<OrganizationEntity> organizationEntitys = this.organizationRepository.findByNullParentAndStatus(status);
    if(CollectionUtils.isEmpty(organizationEntitys)) {
      return Collections.emptyList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByParentAndStatus(java.lang.String, java.lang.Integer)
   */
  @Override
  public List<OrganizationVo> findByParentAndStatus(String parentId, Integer status) {
    if (StringUtils.isBlank(parentId) || status == null) {
      return Lists.newArrayList();
    }
    List<OrganizationEntity> organizationEntitys = this.organizationRepository.findByParentAndStatus(parentId, status);
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByOrgName(java.lang.String)
   */
  @Override
  public JSONArray findByOrgName(String orgName) {
    if (StringUtils.isBlank(orgName)) {
      return null;
    }
    /*
     * 这个方法需要递归进行，过程如下
     * 1、首先根据一般的查询语句，查询满足条件的一个或者多个组织机构
     * 2、最后进行这个组织机构所有父级节点的遍历
     * 在遍历过程中，将连续查询当前组织结构的父级信息，并将其遍历节点的所有子级节点置为null
     * */
    // 1、======
    List<OrganizationEntity> currentOrgs = this.organizationRepository.findByOrgName(orgName);
    if (currentOrgs == null || currentOrgs.isEmpty()) {
      return null;
    }

    // 2、======
    JSONArray jsonObjects = new JSONArray();
    for (OrganizationEntity orgItem : currentOrgs) {
      JSONObject jsonObject = this.recursiveFindParent(orgItem);
      if (jsonObject != null) {
        jsonObjects.add(jsonObject);
      }
    }
    return jsonObjects;
  }

  /**
   * 对一个指定的组织机构节点，递归查询其下所有的父级节点信息
   * @param currentOrg
   */
  private JSONObject recursiveFindParent(OrganizationEntity currentOrg) {
    if (currentOrg == null || currentOrg.getTstatus() != 1) {
      return null;
    }
    JSONObject orgObject = new JSONObject();
    orgObject.put("id", currentOrg.getId());
    orgObject.put(MESS_ORG_NAME, currentOrg.getOrgName());
    orgObject.put("code", currentOrg.getCode());
    orgObject.put("createTime", currentOrg.getCreateTime());
    orgObject.put("sortIndex", currentOrg.getSortIndex());
    orgObject.put("status", currentOrg.getTstatus());
    orgObject.put("type", currentOrg.getType());
    // 构造角色信息
    Set<RoleEntity> roles = currentOrg.getRoles();
    JSONArray roleArray = new JSONArray();
    if (roles != null && !roles.isEmpty()) {
      for (RoleEntity role : roles) {
        JSONObject roleObject = new JSONObject();
        roleObject.put("id", role.getId());
        roleObject.put("comment", role.getComment());
        roleObject.put("createDate", role.getCreateDate());
        roleObject.put("isDeny", role.getIsDeny());
        roleObject.put("roleCode", role.getRoleCode());
        roleObject.put("roleName", role.getRoleName());
        roleObject.put("status", role.getTstatus());
        roleObject.put("roleType", role.getRoleType());
        roleArray.add(roleObject);
      }
    }
    orgObject.put("roles", new JSONArray());

    // 查询父级结构
    OrganizationEntity parent = currentOrg.getParent();
    if (parent != null) {
      orgObject.put("parent", this.recursiveFindParent(parent));
    }
    return orgObject;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findById(java.lang.String)
   */
  @Override
  public OrganizationVo findDetailsById(String id) {
    Validate.notBlank(id, "组织机构id不能为空,请检查!!");
    OrganizationEntity currentOrg = organizationRepository.findDetailsById(id);

    return this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, LinkedHashSet.class, ArrayList.class, "parent","child","roles","users","positions");
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByUserId(java.lang.String)
   */
  @Override
  public JSONArray findByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return null;
    }

    Set<OrganizationEntity> currentOrgs = this.organizationRepository.findByUserId(userId);
    if (currentOrgs == null || currentOrgs.isEmpty()) {
      return null;
    }

    // 递归构造父级（包括父级的Roles信息）
    JSONArray jsonObjects = new JSONArray();
    for (OrganizationEntity orgItem : currentOrgs) {
      JSONObject jsonObject = this.recursiveFindParent(orgItem);
      if (jsonObject != null) {
        jsonObjects.add(jsonObject);
      }
    }
    return jsonObjects;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findUnbindOrgById(java.lang.String)
   */
  @Override
  public Object[] findUnbindOrgById(String currentOrgId) {
    /*
     * 1:查询出所有状态为启用的组织机构id与组织机构名称，并存入集合jsonArray中。
     * 2：根据当前组织机构id递归查询出所有子级组织机构的id与组织机构名称，并存入另外一个集合array中。
     * 3：检索出当前组织机构的父级组织机构id,并查询出父级组织机构名称，保存到array中。
     * 4：根据jsonArray移除掉所有array集合中的元素，就是可以绑定的组织机构节点
     * 5：返回可以绑定的组织机构id与组织机构名称
     */
    if (StringUtils.isBlank(currentOrgId)) {
      return new Object[0];
    }
    List<OrganizationEntity> allOrganizations = this.organizationRepository.findAllByStatus();
    JSONArray jsonArray = new JSONArray();
    if (null != allOrganizations && !allOrganizations.isEmpty()) {
      for (OrganizationEntity org : allOrganizations) {
        JSONObject orgAllObject = new JSONObject();
        orgAllObject.put("id", org.getId());
        orgAllObject.put(MESS_ORG_NAME, org.getOrgName());
        jsonArray.add(orgAllObject);
      }
    }
    LOGGER.debug("所有组织机构节点: {}", JSON.toJSONString(jsonArray));
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    OrganizationEntity childOrg = op.orElse(null);
    if(childOrg == null){
      throw new IllegalArgumentException("未找到任何组织机构信息，请检查!!");
    }

    JSONArray array = this.findChildOrgs(childOrg, new JSONArray());
    //查询父级组织机构信息,并放入要移除的集合中
    if (null != childOrg.getParent() && StringUtils.isNotBlank(childOrg.getParent().getId())) {
      String parentOrgId = childOrg.getParent().getId();
      Optional<OrganizationEntity> optional = this.organizationRepository.findById(parentOrgId);
      if (optional.isPresent()) {
        JSONObject jsonObject = new JSONObject();
        OrganizationEntity parentOrg = optional.get();
        jsonObject.put("id", parentOrg.getId());
        jsonObject.put(MESS_ORG_NAME, parentOrg.getOrgName());
        array.add(jsonObject);
      }
    }
    LOGGER.debug("不可以绑定的组织机构节点(当前组织机构以及所有子级组织机构): {}", JSON.toJSONString(array));
    if (!array.isEmpty()) {
      jsonArray.removeAll(array);
    }
    LOGGER.debug("可以绑定的组织机构节点: {}", JSON.toJSONString(jsonArray));
    return jsonArray.toArray();
  }

  /**
   * 对一个指定的组织机构节点，递归查询其下所有的子级节点信息
   * @param currentOrg
   */
  private JSONArray findChildOrgs(OrganizationEntity currentOrg, JSONArray jsonArray) {
    if (currentOrg != null) {
      Optional<OrganizationEntity> op = organizationRepository.findById(currentOrg.getId());
      OrganizationEntity organization = op.orElse(null);
      Validate.notNull(organization, "没有可用的组织机构,请检查!!");

      JSONObject orgObject = new JSONObject();
      orgObject.put("id", organization.getId());
      orgObject.put(MESS_ORG_NAME, organization.getOrgName());
      jsonArray.add(orgObject);
      // 查询子级结构
      List<OrganizationEntity> childs = organizationRepository.findByParent(organization.getId());
      if (childs != null && !childs.isEmpty()) {
        for (OrganizationEntity child : childs) {
          this.findChildOrgs(child, jsonArray);
        }
      }
    }
    return jsonArray;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findAll()
   */
  @Override
  public List<OrganizationVo> findAll() {
    List<OrganizationEntity> orgs = organizationRepository.findAll(Sort.by(Sort.Order.desc("createTime")));
    if(orgs == null) {
      return Collections.emptyList();
    }
    // 用非持久化对象返回，避免parent和child两个属性的过滤冲突
    Collection<OrganizationVo> result = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class, "parent");
    return new ArrayList<>(result);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findOrgByUserId(java.lang.String)
   */
  @Override
  public Set<OrganizationVo> findOrgByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user = userRepository.findDetailsById(userId);
    if(user==null){
      return Sets.newHashSet();
    }
    Set<OrganizationEntity> orgs = user.getOrgs();
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> vos = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(vos);
  }

  @Override
  public Set<OrganizationVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<OrganizationEntity> orgs = organizationRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> ovos = this.nebulaToolkitService.copyCollectionByWhiteList(orgs,OrganizationEntity.class,OrganizationVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(ovos);
  }

  /**
   * 组织机构查询-查询所有根节点信息
   * @return
   */
  @Override
  public List<OrganizationVo> findRootOrg() {
    List<OrganizationEntity> entityList = this.organizationRepository.findRootOrg();
    if (CollectionUtils.isEmpty(entityList)){
      return Lists.newArrayList();
    }
    Collection<OrganizationVo> result = this.nebulaToolkitService.copyCollectionByWhiteList(entityList, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class);
    return new ArrayList<>(result);
  }


  /**
   * 树结构查询-根据组织机构ID查询根节点以及所有子节点以树形结构返回
   * @param id
   * @return
   */
  @Override
  public OrganizationVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    OrganizationEntity organizationEntity = this.organizationRepository.findById(id).orElse(null);
    if(organizationEntity == null){
      return null;
    }
    return this.findOrgTree(organizationEntity);
  }

  /**
   * 根据组织机构名称实现模糊查询
   * @param orgName
   * @return
   */
  @Override
  public Set<OrganizationVo> findByOrgNameLike(String orgName) {
    Set<OrganizationEntity> entitySet = this.organizationRepository.findByOrgNameLike(orgName);
    if (CollectionUtils.isEmpty(entitySet)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> organizationVos = this.nebulaToolkitService.copyCollectionByWhiteList(entitySet, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class);

    return Sets.newHashSet(organizationVos);
  }

  /**
   * 一级查询
   * @param currentOrg
   * @return
   */
  private OrganizationVo findOrgTree(OrganizationEntity currentOrg) {
    OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, HashSet.class, ArrayList.class );
    organizationVo.setChild(this.findChildren(organizationVo));
    return organizationVo;
  }

  /**
   * 子集查询-子集查找遍历，递归查询其下所有的子级节点信息
   * @param currentOrg
   * @return
   */
  private Set<OrganizationVo> findChildren(OrganizationVo currentOrg) {
    if (Objects.isNull(currentOrg)){
      return Sets.newHashSet();
    }
    Set<OrganizationVo> organizationVoSet = new HashSet<>();
    // 查询子级结构
    List<OrganizationEntity> childs = organizationRepository.findByParent(currentOrg.getId());
    if (CollectionUtils.isEmpty(childs)) {
      return Sets.newHashSet();
    }
    for (OrganizationEntity child : childs) {
      OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(child, OrganizationVo.class, HashSet.class, ArrayList.class);
      organizationVoSet.add(organizationVo);
      organizationVo.setChild(this.findChildren(organizationVo));
    }
    return organizationVoSet;
  }

  @Override
  public OrganizationVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    OrganizationEntity org = organizationRepository.findByCode(code);
    if(org == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public OrganizationVo findByChild(String childId) {
    Validate.notBlank(childId, "子级组织ID不能为空");
    OrganizationEntity org = organizationRepository.findByChildId(childId);
    if(org == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class);
  }
}
