package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.nebular.sfa.worksign.req.SfaApplyTimeInfoReqVo;
import org.apache.commons.lang.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * @author ren.gang
 * @ClassName SfaSignUtils.java
 * @Description Sfa系统-考勤管理相关工具类
 * @createTime 2020年12月02日 14:08:00
 */
public class SfaSignUtils {

    /**
     * 验证开始时间和结束时间
     * 1、空校验
     * 2、日期顺序校验
     * @param startDate
     * @param endDate
     * @return 两个日期的间隔
     */
    public static int verifyStartAndEndDate(String startDate, String endDate) {
        // 校验开始时间格式是否正确
        try {
            LocalDate.parse(startDate);
        } catch (Exception e) {
            throw new BusinessException("开始时间错误。日期格式为：yyyy-MM-dd");
        }
        // 校验结束时间格式
        try {
            LocalDate.parse(endDate);
        } catch (Exception e) {
            throw new BusinessException("结束时间错误。日期格式为：yyyy-MM-dd");
        }
        // 开始时间到结束时间的间隔数。开始和结束一致为0
        int intervalNum = LocalDate.parse(startDate).until(LocalDate.parse(endDate)).getDays();
        // 校验开始时间不能超过结束时间
        if(intervalNum < 0) {
            throw new BusinessException("开始时间超过结束时间");
        }
        return intervalNum;
    }

    // 详情日期 日期类型全字段
    public static final String TIME_TYPE_ALL_STR = "1,2,3";

    /**
     * 验证考勤相关申请功能，数据信息
     * 1、当前方法参数不能为空，或者和正确参数不匹配
     * 2、开始-结束时间 和规则
     * 3、日期明细列表
     * 4、申请天数
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @param applyList 日期明细列表
     * @param applyDays 申请时长
     * @return
     */
    public static List<SfaApplyTimeInfoReqVo> verifyApply(String startDate, String endDate, List<SfaApplyTimeInfoReqVo> applyList, String applyDays) {

        // 校验开始结束日期
        int intervalNum = verifyStartAndEndDate(startDate, endDate);

        // 日期明细列表空检验
        if(applyList == null || applyList.size() == 0) {
            throw new BusinessException("日期明细不能为空");
        }
        // 间隔=0时，包含一条数据;间隔=1是，包含两条数据;间隔>1时，包含2条或者间隔数+1条数据
        if(applyList.size() == 1) {
            if(intervalNum != 0) {
                throw new BusinessException("日期明细列表和 申请开始结束时间不一致。至少包含开始和结束日期的明细");
            }
        } else {
            if(intervalNum == 0) {
                throw new BusinessException("日期明细列表和 申请开始结束时间不一致。至少包含开始和结束日期的明细");
            }
        }

        // 遍历校验日期明细
        for(int r =0; r < applyList.size(); r ++) {
            SfaApplyTimeInfoReqVo vo = applyList.get(r);
            // 日期校验
            try {
                LocalDate.parse(vo.getTimeStr());
            }catch (Exception e) {
                throw new BusinessException("日期明细列表中 日期 错误。日期格式为：yyyy-MM-dd");
            }
            // 日期类型校验
            if(StringUtils.isEmpty(vo.getTimeType())
                    || TIME_TYPE_ALL_STR.indexOf(vo.getTimeType()) == -1) {
                throw new BusinessException("日期明细列表中 日期类型 错误。日期类型(1全天，2上午，3下午)");
            }
            if(r == 0) { // 第一条数据是开始日期
                if(!vo.getTimeStr().equals(startDate)){
                    throw new BusinessException("日期明细列表和 申请开始时间不一致");
                }
            } else if(r == applyList.size()-1) { //最后一条数据是结束日期
                if(!vo.getTimeStr().equals(endDate)) {
                    throw new BusinessException("日期明细列表和 申请结束时间不一致");
                }
            } else { // 其他日期要按照顺序排列
                if(LocalDate.parse(startDate).plusDays(r).until(LocalDate.parse(vo.getTimeStr())).getDays() != 0) {
                    throw new BusinessException("日期明细列表和 申请开始结束时间不一致");
                }
            }
        }

        // 补全数据
        List<SfaApplyTimeInfoReqVo> returnList = new ArrayList<>();
        if(intervalNum <= 1 || applyList.size() == intervalNum+1) {
            returnList = applyList;
        } else {
            returnList.add(applyList.get(0));
            for(int r =1; r < intervalNum; r ++) {
                SfaApplyTimeInfoReqVo vo = new SfaApplyTimeInfoReqVo();
                vo.setTimeStr(LocalDate.parse(startDate).plusDays(r).toString());
                vo.setTimeType(SfaCommonEnum.dataTimeType.ALL_DAY.getValue());
                returnList.add(vo);
            }
            returnList.add(applyList.get(1));
        }

        // 记录申请时间(天)统计
        BigDecimal countApplyDays = new BigDecimal(0);
        for(SfaApplyTimeInfoReqVo vo : returnList){
            // 统计申请时长
            if(SfaCommonEnum.dataTimeType.ALL_DAY.getValue().equals(vo.getTimeType())) {
                countApplyDays = countApplyDays.add(new BigDecimal(1));
            } else {
                countApplyDays = countApplyDays.add(new BigDecimal(0.5));
            }
        };

        // 校验申请时间是否正确
        try {
            new BigDecimal(applyDays);
        }catch (Exception e) {
            throw new BusinessException("申请时长错误");
        }
        if(new BigDecimal(applyDays).compareTo(countApplyDays) != 0) {
            throw new BusinessException("申请时长和明细时间不一致");
        }

        return returnList;
    }

}
