package com.biz.crm.util;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mdm.cusorg.MdmCustomerOrgRedisFeign;
import com.biz.crm.nebular.mdm.cusorg.CusOrgVo;
import com.biz.crm.nebular.mdm.cusorg.MdmCustomerOrgParentChildrenRedisVo;
import com.biz.crm.nebular.mdm.cusorg.MdmCustomerOrgRedisVo;
import com.biz.crm.nebular.mdm.cusorg.MdmCustomerOrgReloadRedisVo;
import com.biz.crm.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 客户组织查询工具类
 */
@Slf4j
public class CustomerOrgUtil {

    private static RedisService redisService;

    private static MdmCustomerOrgRedisFeign mdmCustomerOrgRedisFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmCustomerOrgRedisFeign(MdmCustomerOrgRedisFeign customerOrgRedisFeign) {
        mdmCustomerOrgRedisFeign = customerOrgRedisFeign;
    }

    /**
     * 根据客户组织编码获取客户组织详情
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static CusOrgVo getCustomerOrgByCode(String customerOrgCode) {
        if (customerOrgCode != null && !"".equals(customerOrgCode)) {
            Map<String, MdmCustomerOrgRedisVo> customerOrgRedisMap = getCustomerOrgSelfRedisMapFromLocalFirst(Collections.singletonList(customerOrgCode));
            if (customerOrgRedisMap.containsKey(customerOrgCode)) {
                MdmCustomerOrgRedisVo mdmCustomerOrgRedisVo = customerOrgRedisMap.get(customerOrgCode);
                return CrmBeanUtil.copy(mdmCustomerOrgRedisVo, CusOrgVo.class);
            }
        }
        return null;
    }

    /**
     * 根据客户组织编码集合获取客户组织详情
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getCustomerOrgByCodeList(List<String> customerOrgCodeList) {
        List<CusOrgVo> list = new ArrayList<>();
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            Map<String, MdmCustomerOrgRedisVo> customerOrgRedisMap = getCustomerOrgSelfRedisMapFromLocalFirst(customerOrgCodeList);
            if (!customerOrgRedisMap.isEmpty()) {
                list.addAll(customerOrgRedisMap.values().stream().map(item -> CrmBeanUtil.copy(item, CusOrgVo.class)).collect(Collectors.toList()));
            }
        }
        return list;
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部上级
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListIncludeSelf(String customerOrgCode) {
        if (customerOrgCode != null && !"".equals(customerOrgCode)) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgParentRedisMapFromLocalFirst(Collections.singletonList(customerOrgCode));
            if (!map.isEmpty()) {
                MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                if (redisVo != null) {
                    List<MdmCustomerOrgRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, CusOrgVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部上级
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListIncludeSelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgParentRedisMapFromLocalFirst(customerOrgCodeList);
            Set<MdmCustomerOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (!map.containsKey(customerOrgCode)) {
                        continue;
                    }
                    MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), CusOrgVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部上级
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeSelf(String customerOrgCode) {
        if (customerOrgCode != null && !"".equals(customerOrgCode)) {
            List<CusOrgVo> parentCustomerOrgListIncludeSelf = getParentCustomerOrgListIncludeSelf(customerOrgCode);
            if (parentCustomerOrgListIncludeSelf != null && parentCustomerOrgListIncludeSelf.size() > 0) {
                return parentCustomerOrgListIncludeSelf.stream().filter(item -> !customerOrgCode.equals(item.getCustomerOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的客户组织
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeSelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgParentRedisMapFromLocalFirst(customerOrgCodeList);
            Set<MdmCustomerOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (!map.containsKey(customerOrgCode)) {
                        continue;
                    }
                    MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !customerOrgCode.equals(item.getCustomerOrgCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), CusOrgVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中的任何一个）全部上级
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeAnySelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            List<CusOrgVo> parentCustomerOrgListExcludeSelf = getParentCustomerOrgListExcludeSelf(customerOrgCodeList);
            if (parentCustomerOrgListExcludeSelf != null && parentCustomerOrgListExcludeSelf.size() > 0) {
                final Set<String> customerOrgCodeSet = new HashSet<>(customerOrgCodeList);
                return parentCustomerOrgListExcludeSelf.stream().filter(item -> !customerOrgCodeSet.contains(item.getCustomerOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部下级
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListIncludeSelf(String customerOrgCode) {
        if (customerOrgCode != null && !"".equals(customerOrgCode)) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgChildrenRedisMapFromLocalFirst(Collections.singletonList(customerOrgCode));
            if (!map.isEmpty()) {
                MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                if (redisVo != null) {
                    List<MdmCustomerOrgRedisVo> list = redisVo.getList();
                    if (list != null && list.size() > 0) {
                        return CrmBeanUtil.copyList(list, CusOrgVo.class);
                    }
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部下级
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListIncludeSelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgChildrenRedisMapFromLocalFirst(customerOrgCodeList);
            Set<MdmCustomerOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (!map.containsKey(customerOrgCode)) {
                        continue;
                    }
                    MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList());
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), CusOrgVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部下级
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeSelf(String customerOrgCode) {
        if (customerOrgCode != null && !"".equals(customerOrgCode)) {
            List<CusOrgVo> parentCustomerOrgListIncludeSelf = getChildrenCustomerOrgListIncludeSelf(customerOrgCode);
            if (parentCustomerOrgListIncludeSelf != null && parentCustomerOrgListIncludeSelf.size() > 0) {
                return parentCustomerOrgListIncludeSelf.stream().filter(item -> !customerOrgCode.equals(item.getCustomerOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的客户组织
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeSelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            Map<String, MdmCustomerOrgParentChildrenRedisVo> map = getCustomerOrgChildrenRedisMapFromLocalFirst(customerOrgCodeList);
            Set<MdmCustomerOrgRedisVo> set = new HashSet<>(16);
            if (!map.isEmpty()) {
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (!map.containsKey(customerOrgCode)) {
                        continue;
                    }
                    MdmCustomerOrgParentChildrenRedisVo redisVo = map.get(customerOrgCode);
                    if (redisVo == null || redisVo.getList() == null || redisVo.getList().size() <= 0) {
                        continue;
                    }
                    set.addAll(redisVo.getList().stream().filter(item -> !customerOrgCode.equals(item.getCustomerOrgCode())).collect(Collectors.toSet()));
                }
                if (!set.isEmpty()) {
                    return CrmBeanUtil.copyList(new ArrayList<>(set), CusOrgVo.class);
                }
            }
        }
        return Collections.emptyList();
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中任何一个）全部下级
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeAnySelf(List<String> customerOrgCodeList) {
        if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
            List<CusOrgVo> parentCustomerOrgListExcludeSelf = getChildrenCustomerOrgListExcludeSelf(customerOrgCodeList);
            if (parentCustomerOrgListExcludeSelf != null && parentCustomerOrgListExcludeSelf.size() > 0) {
                final Set<String> customerOrgCodeSet = new HashSet<>(customerOrgCodeList);
                return parentCustomerOrgListExcludeSelf.stream().filter(item -> !customerOrgCodeSet.contains(item.getCustomerOrgCode())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部上级（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListIncludeSelfEnable(String customerOrgCode) {
        return getParentCustomerOrgListIncludeSelf(customerOrgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部上级（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListIncludeSelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListIncludeSelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部上级（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeSelfEnable(String customerOrgCode) {
        return getParentCustomerOrgListExcludeSelf(customerOrgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部上级，如果本身有上下级关系，则会返回处于上级的客户组织（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeSelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeSelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中的任何一个）全部上级（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getParentCustomerOrgListExcludeAnySelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeAnySelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部下级（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListIncludeSelfEnable(String customerOrgCode) {
        return getChildrenCustomerOrgListIncludeSelf(customerOrgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部下级（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListIncludeSelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListIncludeSelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部下级（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeSelfEnable(String customerOrgCode) {
        return getChildrenCustomerOrgListExcludeSelf(customerOrgCode).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部下级，如果参数本身有上下级关系，则会返回处于下级的客户组织（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeSelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeSelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中任何一个）全部下级（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<CusOrgVo> getChildrenCustomerOrgListExcludeAnySelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeAnySelf(customerOrgCodeList).stream().filter(item -> CrmEnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus())).collect(Collectors.toList());
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部上级客户组织编码
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListIncludeSelf(String customerOrgCode) {
        return getParentCustomerOrgListIncludeSelf(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部上级客户组织编码
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListIncludeSelf(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListIncludeSelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部上级客户组织编码
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeSelf(String customerOrgCode) {
        return getParentCustomerOrgListExcludeSelf(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部上级客户组织编码，如果本身有上下级关系，则会返回处于上级的客户组织
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeSelf(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeSelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中的任何一个）全部上级客户组织编码
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeAnySelf(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeAnySelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部下级客户组织编码
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListIncludeSelf(String customerOrgCode) {
        return getChildrenCustomerOrgListIncludeSelf(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部下级客户组织编码
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListIncludeSelf(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListIncludeSelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部下级客户组织编码
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeSelf(String customerOrgCode) {
        return getChildrenCustomerOrgListExcludeSelf(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部下级客户组织编码，如果参数本身有上下级关系，则会返回处于下级的客户组织
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeSelf(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeSelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中任何一个）全部下级客户组织编码
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeAnySelf(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeAnySelf(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部上级客户组织编码（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListIncludeSelfEnable(String customerOrgCode) {
        return getParentCustomerOrgListIncludeSelfEnable(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部上级客户组织编码（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListIncludeSelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListIncludeSelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部上级客户组织编码（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeSelfEnable(String customerOrgCode) {
        return getParentCustomerOrgListExcludeSelfEnable(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部上级客户组织编码，如果本身有上下级关系，则会返回处于上级的客户组织（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeSelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeSelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中的任何一个）全部上级客户组织编码（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getParentCustomerOrgCodeListExcludeAnySelfEnable(List<String> customerOrgCodeList) {
        return getParentCustomerOrgListExcludeAnySelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }


    /**
     * 根据客户组织编码获取客户组织及客户组织全部下级客户组织编码（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListIncludeSelfEnable(String customerOrgCode) {
        return getChildrenCustomerOrgListIncludeSelfEnable(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织及客户组织全部下级客户组织编码（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListIncludeSelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListIncludeSelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码获取客户组织（不含自己）全部下级客户组织编码（只查启用）
     *
     * @param customerOrgCode 客户组织编码
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeSelfEnable(String customerOrgCode) {
        return getChildrenCustomerOrgListExcludeSelfEnable(customerOrgCode).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含自己）全部下级客户组织编码，如果参数本身有上下级关系，则会返回处于下级的客户组织（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeSelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeSelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }

    /**
     * 根据客户组织编码集合获取客户组织（不含参数中任何一个）全部下级客户组织编码（只查启用）
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    public static List<String> getChildrenCustomerOrgCodeListExcludeAnySelfEnable(List<String> customerOrgCodeList) {
        return getChildrenCustomerOrgListExcludeAnySelfEnable(customerOrgCodeList).stream().map(CusOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    }


    /**
     * 清除全部客户组织Redis缓存和本地缓存
     */
    public static void deleteAllCache() {
        redisService.del(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH);
        redisService.del(RedisParam.MDM_CUSTOMER_ORG_SELF_HASH);
        redisService.del(RedisParam.MDM_CUSTOMER_ORG_PARENT_HASH);
        redisService.del(RedisParam.MDM_CUSTOMER_ORG_CHILDREN_HASH);
        ThreadLocalUtil.delCustomerOrg();
    }

    /**
     * 重新加载redis缓存
     */
    public static void reloadAllCache() {
        deleteAllCache();
        Result<List<MdmCustomerOrgReloadRedisVo>> result = mdmCustomerOrgRedisFeign.findRedisReloadList();
        if (result.isSuccess()) {
            List<MdmCustomerOrgReloadRedisVo> list = result.getResult();
            if (list != null && list.size() > 0) {
                List<List<MdmCustomerOrgReloadRedisVo>> partition = Lists.partition(list, 500);
                for (List<MdmCustomerOrgReloadRedisVo> item :
                        partition) {
                    Map<String, MdmCustomerOrgRedisVo> selfMap = item.stream().collect(Collectors.toMap(MdmCustomerOrgReloadRedisVo::getCustomerOrgCode, MdmCustomerOrgReloadRedisVo::getSelf));
                    Map<String, List<MdmCustomerOrgRedisVo>> parentMap = item.stream().collect(Collectors.toMap(MdmCustomerOrgReloadRedisVo::getCustomerOrgCode, MdmCustomerOrgReloadRedisVo::getParentList));
                    Map<String, List<MdmCustomerOrgRedisVo>> childrenMap = item.stream().collect(Collectors.toMap(MdmCustomerOrgReloadRedisVo::getCustomerOrgCode, MdmCustomerOrgReloadRedisVo::getChildrenList));
                    redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_SELF_HASH, new HashMap<Object, Object>(selfMap), RedisParam.MONTH3);
                    redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_PARENT_HASH, new HashMap<Object, Object>(parentMap), RedisParam.MONTH3);
                    redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_CHILDREN_HASH, new HashMap<Object, Object>(childrenMap), RedisParam.MONTH3);
                }
            }
        }
    }

    /*--------------------------------------------------以下是私有方法--------------------------------------------------*/

    /**
     * 获取客户组织缓存数据，优先从本地缓存取，再从redis取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgRedisVo> getCustomerOrgSelfRedisMapFromLocalFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (ThreadLocalUtil.customerOrgSelfContains(customerOrgCode)) {
                        MdmCustomerOrgRedisVo fromCustomerOrgSelf = ThreadLocalUtil.getFromCustomerOrgSelf(customerOrgCode);
                        map.put(customerOrgCode, fromCustomerOrgSelf);
                    } else {
                        getFromRedisCodeSet.add(customerOrgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmCustomerOrgRedisVo> mapFromRedisFirst = getCustomerOrgSelfRedisMapFromRedisFirst(new ArrayList<>(customerOrgCodeList));

                    for (String customerOrgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(customerOrgCode)) {
                            map.put(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                        }
                        ThreadLocalUtil.addToCustomerOrgSelf(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取客户组织缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgRedisVo> getCustomerOrgSelfRedisMapFromRedisFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
                List<MdmCustomerOrgRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashSet<Object>(customerOrgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = customerOrgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> customerOrgListFromRedis = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_SELF_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (customerOrgListFromRedis != null && customerOrgListFromRedis.size() > 0) {
                        List<MdmCustomerOrgRedisVo> collect = customerOrgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmCustomerOrgRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmCustomerOrgRedisVo::getCustomerOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }

                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmCustomerOrgRedisVo>> result = mdmCustomerOrgRedisFeign.findRedisListByCustomerOrgCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmCustomerOrgRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmCustomerOrgRedisVo> collect = list.stream().collect(Collectors.toMap(MdmCustomerOrgRedisVo::getCustomerOrgCode, v -> v));

                            Map<String, MdmCustomerOrgRedisVo> cacheMap = new HashMap<>(16);
                            for (String customerOrgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(customerOrgCode)) {
                                    lockCodeMap.put(customerOrgCode, customerOrgCode);
                                } else {
                                    redisVoList.add(collect.get(customerOrgCode));
                                    cacheMap.put(customerOrgCode, collect.get(customerOrgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_SELF_HASH, new HashMap<Object, Object>(cacheMap), RedisParam.MONTH3);
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmCustomerOrgRedisVo::getCustomerOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取客户组织上级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgParentChildrenRedisVo> getCustomerOrgParentRedisMapFromLocalFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (ThreadLocalUtil.customerOrgParentContains(customerOrgCode)) {
                        MdmCustomerOrgParentChildrenRedisVo fromCustomerOrgParent = ThreadLocalUtil.getFromCustomerOrgParent(customerOrgCode);
                        map.put(customerOrgCode, fromCustomerOrgParent);
                    } else {
                        getFromRedisCodeSet.add(customerOrgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmCustomerOrgParentChildrenRedisVo> mapFromRedisFirst = getCustomerOrgParentRedisMapFromRedisFirst(new ArrayList<>(customerOrgCodeList));

                    for (String customerOrgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(customerOrgCode)) {
                            map.put(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                        }
                        ThreadLocalUtil.addToCustomerOrgParent(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取客户组织上级缓存数据，优先从redis缓存存取，再从数据库取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgParentChildrenRedisVo> getCustomerOrgParentRedisMapFromRedisFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
                List<MdmCustomerOrgParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashSet<Object>(customerOrgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = customerOrgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> customerOrgListFromRedis = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_PARENT_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (customerOrgListFromRedis != null && customerOrgListFromRedis.size() > 0) {
                        List<MdmCustomerOrgParentChildrenRedisVo> collect = customerOrgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmCustomerOrgParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmCustomerOrgParentChildrenRedisVo>> result = mdmCustomerOrgRedisFeign.findRedisCustomerOrgParentListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmCustomerOrgParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmCustomerOrgParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode, v -> v));

                            Map<String, MdmCustomerOrgParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String customerOrgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(customerOrgCode)) {
                                    lockCodeMap.put(customerOrgCode, customerOrgCode);
                                } else {
                                    redisVoList.add(collect.get(customerOrgCode));
                                    cacheMap.put(customerOrgCode, collect.get(customerOrgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_PARENT_HASH, new HashMap<Object, Object>(cacheMap), RedisParam.MONTH3);
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取客户组织下级缓存数据，优先从本地缓存缓存取，再从redis缓存取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgParentChildrenRedisVo> getCustomerOrgChildrenRedisMapFromLocalFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {

                Set<String> getFromRedisCodeSet = new HashSet<>(16);
                for (String customerOrgCode :
                        customerOrgCodeList) {
                    if (ThreadLocalUtil.customerOrgChildrenContains(customerOrgCode)) {
                        MdmCustomerOrgParentChildrenRedisVo fromCustomerOrgChildren = ThreadLocalUtil.getFromCustomerOrgChildren(customerOrgCode);
                        map.put(customerOrgCode, fromCustomerOrgChildren);
                    } else {
                        getFromRedisCodeSet.add(customerOrgCode);
                    }
                }

                if (!getFromRedisCodeSet.isEmpty()) {
                    Map<String, MdmCustomerOrgParentChildrenRedisVo> mapFromRedisFirst = getCustomerOrgChildrenRedisMapFromRedisFirst(new ArrayList<>(customerOrgCodeList));

                    for (String customerOrgCode :
                            getFromRedisCodeSet) {
                        if (mapFromRedisFirst.containsKey(customerOrgCode)) {
                            map.put(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                        }
                        ThreadLocalUtil.addToCustomerOrgChildren(customerOrgCode, mapFromRedisFirst.get(customerOrgCode));
                    }
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 获取客户组织下级缓存数据，优先从redis缓存取，再从数据库取
     *
     * @param customerOrgCodeList 客户组织编码集合
     * @return
     */
    private static Map<String, MdmCustomerOrgParentChildrenRedisVo> getCustomerOrgChildrenRedisMapFromRedisFirst(List<String> customerOrgCodeList) {
        Map<String, MdmCustomerOrgParentChildrenRedisVo> map = new HashMap<>(16);
        try {
            if (customerOrgCodeList != null && customerOrgCodeList.size() > 0) {
                List<MdmCustomerOrgParentChildrenRedisVo> redisVoList = new ArrayList<>();
                Set<String> lockCodeList = new HashSet<>(16);
                List<Object> lockListFromHash = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashSet<Object>(customerOrgCodeList));
                if (lockListFromHash != null && lockListFromHash.size() > 0) {
                    lockCodeList.addAll(lockListFromHash.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.toSet()));
                }
                Set<String> notLockGetFromRedisCodeSet = customerOrgCodeList.stream().filter(item -> !lockCodeList.contains(item)).collect(Collectors.toSet());
                Set<String> redisHasCodeSet = new HashSet<>(16);
                if (!notLockGetFromRedisCodeSet.isEmpty()) {
                    List<Object> customerOrgListFromRedis = redisService.hmget(RedisParam.MDM_CUSTOMER_ORG_CHILDREN_HASH, new HashSet<Object>(notLockGetFromRedisCodeSet));
                    if (customerOrgListFromRedis != null && customerOrgListFromRedis.size() > 0) {
                        List<MdmCustomerOrgParentChildrenRedisVo> collect = customerOrgListFromRedis.stream().filter(Objects::nonNull).map(item -> CrmBeanUtil.copy(item, MdmCustomerOrgParentChildrenRedisVo.class)).collect(Collectors.toList());
                        if (collect.size() > 0) {
                            redisVoList.addAll(collect);
                            redisHasCodeSet.addAll(collect.stream().map(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode).collect(Collectors.toSet()));
                        }
                    }
                }
                Set<String> getByFeignCodeSet = notLockGetFromRedisCodeSet.stream().filter(item -> !redisHasCodeSet.contains(item)).collect(Collectors.toSet());
                if (!getByFeignCodeSet.isEmpty()) {
                    Result<List<MdmCustomerOrgParentChildrenRedisVo>> result = mdmCustomerOrgRedisFeign.findRedisCustomerOrgChildrenListByCodeList(new ArrayList<>(getByFeignCodeSet));
                    if (result.isSuccess()) {
                        Map<String, String> lockCodeMap = new HashMap<>(16);
                        List<MdmCustomerOrgParentChildrenRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, MdmCustomerOrgParentChildrenRedisVo> collect = list.stream().collect(Collectors.toMap(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode, v -> v));

                            Map<String, MdmCustomerOrgParentChildrenRedisVo> cacheMap = new HashMap<>(16);
                            for (String customerOrgCode :
                                    getByFeignCodeSet) {
                                if (!collect.containsKey(customerOrgCode)) {
                                    lockCodeMap.put(customerOrgCode, customerOrgCode);
                                } else {
                                    redisVoList.add(collect.get(customerOrgCode));
                                    cacheMap.put(customerOrgCode, collect.get(customerOrgCode));
                                }
                            }
                            if (!cacheMap.isEmpty()) {
                                redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_CHILDREN_HASH, new HashMap<Object, Object>(cacheMap), RedisParam.MONTH3);
                            }
                        } else {
                            lockCodeMap.putAll(getByFeignCodeSet.stream().collect(Collectors.toMap(k -> k, v -> v)));
                        }
                        if (!lockCodeMap.isEmpty()) {
                            redisService.hmset(RedisParam.MDM_CUSTOMER_ORG_LOCK_HASH, new HashMap<Object, Object>(lockCodeMap), RedisParam.TIME3);
                        }
                    }
                }
                if (redisVoList.size() > 0) {
                    map.putAll(redisVoList.stream().collect(Collectors.toMap(MdmCustomerOrgParentChildrenRedisVo::getCustomerOrgCode, v -> v)));
                }
            }
        } catch (Exception e) {
            log.error("客户组织查询工具类异常：{}", e);
        }
        return map;
    }

}
