package com.biz.crm.util;

import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.mdm.feign.MdmDictFeign;
import com.biz.crm.nebular.mdm.dict.resp.DictAttrConfRedisVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataRedisVo;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.dict.resp.DictGroupRedisVo;
import com.biz.crm.service.RedisService;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据字典工具类
 **/
@Slf4j
public class DictUtil {

    public static final String PATH_SPLIT = "/";

    private static RedisService redisService;

    private static MdmDictFeign mdmDictFeign;

    public static void setRedisService(RedisService redis) {
        redisService = redis;
    }

    public static void setMdmDictFeign(MdmDictFeign feign) {
        mdmDictFeign = feign;
    }

    /**
     * 根据数据字典类型编码（或者路径编码）查询字典类型（或者路径下（不含当前））的全部字典值
     *
     * @param dictTypeCode 字典类型编码（或者路径编码）
     * @return map
     */
    public static Map<String, String> dictMap(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return fullMapByPath(dictTypeCode);
        }
        return new HashMap<>(16);
    }

    /**
     * 根据数据字典类型编码（或者路径编码）查询字典类型（或者路径下（不含当前））的全部字典值，dictCode和dictValue反转
     *
     * @param dictTypeCode 字典类型编码（或者路径编码）
     * @return map
     */
    public static Map<String, String> dictRevertMap(String dictTypeCode) {
        Map<String, String> map = dictMap(dictTypeCode);
        if (!map.isEmpty()) {
            return map.keySet().stream().collect(Collectors.toMap(map::get, x -> x, (x1, x2) -> x1));
        }
        return new HashMap<>(16);
    }

    /**
     * 根据字典类型编码查询第一层字典值，如果是路径编码，则查询路径编码下一层的值
     *
     * @param dictTypeCode
     * @return
     */
    public static Map<String, String> dictSingleMap(String dictTypeCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            map = singleMapByPath(dictTypeCode);
        }
        return map;
    }

    /**
     * （批量）根据数据字典类型编码（或者路径编码）查询字典类型（或者路径下（不含当前））的全部字典值
     *
     * @param dictTypeCodes 分类编码集合
     * @return map
     */
    public static Map<String, Map<String, String>> getDictValueMapsByCodes(List<String> dictTypeCodes) {
        Map<String, Map<String, String>> maps = new HashMap<>(16);
        try {
            if (dictTypeCodes != null && dictTypeCodes.size() > 0) {
                for (String dictTypeCode :
                        dictTypeCodes) {
                    maps.put(dictTypeCode, dictMap(dictTypeCode));
                }
            }
        } catch (Exception e) {
            log.error("获取字典数据异常!!!!!!!!");
            e.printStackTrace();
        }
        return maps;
    }

    /**
     * 根据数据字典类型编码（或者路径编码）查询字典类型（或者路径下（不含当前））的全部字典值
     *
     * @param dictTypeCode 字典类型编码
     * @return map
     */
    public static Map<String, String> getDictValueMapsByCodes(String dictTypeCode) {
        return dictMap(dictTypeCode);
    }

    /**
     * 获取字典的直接下级Map
     *
     * @param dictTypeCode 数据字典类型编码或者路径
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getChildrenMap(String dictTypeCode, String dictCode) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            map = singleMapByPath(dictTypeCode + PATH_SPLIT + dictCode);
        }
        return map;
    }

    /**
     * 获取扩展字段Map
     *
     * @param dictTypeCode 数据字典类型编码或者路径编码
     * @param dictCode     字典编码
     * @return
     */
    public static Map<String, String> getExtendMap(String dictTypeCode, String dictCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            DictGroupRedisVo group = getDictGroup(realDictTypeCode);
            if (group != null) {
                List<DictAttrConfRedisVo> confList = group.getConfList();
                if (confList != null && confList.size() > 0) {
                    for (DictDataRedisVo data :
                            group.getDataList()) {
                        if (dictCode.equals(data.getDictCode())) {
                            return generateExtendMap(data, group.getConfList());
                        }
                    }
                }
            }
        }
        return new LinkedHashMap<>();
    }

    /**
     * 根据字典类型编码或者路径编码（如果是路径编码，结果不包含路径对应数据）查询树形结构
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> tree(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return treeByPath(dictTypeCode, true);
        }
        return new ArrayList<>();
    }

    /**
     * 根据字典类型编码或者路径编码（不含当前路径）获取全部下级字典值
     *
     * @param dictTypeCode
     * @return
     */
    public static List<DictDataVo> list(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return listByPath(dictTypeCode, true);
        }
        return new ArrayList<>();
    }

    /**
     * 获取字典的直接下级List
     *
     * @param dictTypeCode 数据字典类型编码或者路径编码
     * @param dictCode     字典编码
     * @return
     */
    public static List<DictDataVo> getChildrenList(String dictTypeCode, String dictCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            List<DictDataVo> list = treeByPath(dictTypeCode + PATH_SPLIT + dictCode, true);
            if (list != null && list.size() > 0) {
                return list.stream().peek(item -> item.setChildren(null)).collect(Collectors.toList());
            }
        }
        return new ArrayList<>();
    }

    /**
     * 根据字典类型编码和字典编码获取字典值
     *
     * @param dictTypeCode 字典类型编码或者路径编码
     * @param dictCode     字典编码
     * @return
     */
    public static String dictValue(String dictTypeCode, String dictCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictCode != null && !"".equals(dictCode)) {
            Map<String, String> map = fullMapByPath(dictTypeCode);
            if (map.containsKey(dictCode)) {
                return map.get(dictCode);
            }
        }
        return null;
    }

    /**
     * 根据数据字典值反向获取数据字典编码集合
     *
     * @param dictTypeCode 字典类型编码或者路径编码
     * @param dictValue    字典值
     * @return
     */
    public static List<String> getCodeListByValue(String dictTypeCode, String dictValue) {
        List<String> list = new ArrayList<>();
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictValue != null && !"".equals(dictValue)) {
            List<DictDataVo> listByPath = listByPath(dictTypeCode, false);
            if (listByPath != null && listByPath.size() > 0) {
                list.addAll(listByPath.stream().filter(item -> dictValue.equals(item.getDictValue())).map(DictDataVo::getDictCode).collect(Collectors.toList()));
            }
        }
        return list;
    }

    /**
     * 根据数据字典值反向获取数据字典编码（取第一个）
     *
     * @param dictTypeCode 字典类型编码或者路径编码
     * @param dictValue    字典值
     * @return
     */
    public static String getFirstCodeByValue(String dictTypeCode, String dictValue) {
        if (dictTypeCode != null && !"".equals(dictTypeCode) && dictValue != null && !"".equals(dictValue)) {
            List<DictDataVo> listByPath = listByPath(dictTypeCode, false);
            for (DictDataVo item :
                    listByPath) {
                if (item.getDictValue().equals(dictValue)) {
                    return item.getDictCode();
                }
            }
        }
        return null;
    }

    /**
     * 删除数据字典缓存
     *
     * @param dictTypeCode
     */
    public static void deleteRedisCache(String dictTypeCode) {
        if (StringUtils.isNotEmpty(dictTypeCode)) {
            String realDictTypeCode = getRealDictTypeCode(dictTypeCode);
            redisService.del(RedisParam.DICT_TYPE_GROUP + realDictTypeCode);
            redisService.del(RedisParam.DICT_TYPE_LOCK + realDictTypeCode);
        }
    }

    /*--------------------------------------------------以下为私有方法--------------------------------------------------*/

    /**
     * 获取字典组
     *
     * @param dictTypeCode 字典类型编码
     * @return
     */
    private static DictGroupRedisVo getDictGroup(String dictTypeCode) {
//        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
//            Map<String, DictGroupRedisVo> dictGroupMap = getDictGroupMap(Collections.singletonList(dictTypeCode));
//            if (!dictGroupMap.isEmpty()) {
//                return dictGroupMap.get(dictTypeCode);
//            }
//        }
//        return null;
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            if (ThreadLocalUtil.dictGroupContains(dictTypeCode)) {
                return ThreadLocalUtil.getFromDictGroup(dictTypeCode);
            }
            Map<String, DictGroupRedisVo> dictGroupMap = getDictGroupMap(Collections.singletonList(dictTypeCode));
            DictGroupRedisVo group = dictGroupMap.get(dictTypeCode);
            ThreadLocalUtil.addToDictGroup(dictTypeCode, group);
            return group;
        }
        return null;
    }

    /**
     * 获取多个字典组集合
     *
     * @param dictTypeCodeList 字典类型编码集合
     * @return
     */
    private static Map<String, DictGroupRedisVo> getDictGroupMap(List<String> dictTypeCodeList) {
        Map<String, DictGroupRedisVo> map = new HashMap<>(16);
        try {
            if (dictTypeCodeList != null && dictTypeCodeList.size() > 0) {
                List<String> unlockCodeList = dictTypeCodeList.stream().filter(
                        dictTypeCode -> !checkLock(dictTypeCode)
                ).collect(Collectors.toList());
                List<String> notCacheCodeList = new ArrayList<>();
                for (String dictTypeCode :
                        unlockCodeList) {
                    Object o = redisService.get(RedisParam.DICT_TYPE_GROUP + dictTypeCode);
                    if (o == null) {
                        notCacheCodeList.add(dictTypeCode);
                    } else {
                        map.put(dictTypeCode, (DictGroupRedisVo) o);
                    }
                }
                if (notCacheCodeList.size() > 0) {
                    Result<List<DictGroupRedisVo>> result = mdmDictFeign.getDictGroupRedisList(dictTypeCodeList);
                    if (result.isSuccess()) {
                        List<DictGroupRedisVo> list = result.getResult();
                        if (list != null && list.size() > 0) {
                            Map<String, DictGroupRedisVo> collect = list.stream().collect(Collectors.toMap(DictGroupRedisVo::getDictTypeCode, v -> v));
                            for (String dictTypeCode :
                                    notCacheCodeList) {
                                if (!collect.containsKey(dictTypeCode)) {
                                    lock(dictTypeCode);
                                } else {
                                    addRedisCache(collect.get(dictTypeCode));
                                    map.put(dictTypeCode, collect.get(dictTypeCode));
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.error("数据字典工具类异常：{}", e);
        }
        return map;
    }

    /**
     * 添加字典缓存到redis
     *
     * @param group
     */
    private static void addRedisCache(DictGroupRedisVo group) {
        if (group != null) {
            redisService.set(RedisParam.DICT_TYPE_GROUP + group.getDictTypeCode(), group);
        }
    }

    /**
     * 字典解锁
     *
     * @param dictTypeCode
     */
    private static void lock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            redisService.setSeconds(RedisParam.DICT_TYPE_LOCK + dictTypeCode, dictTypeCode, RedisParam.TIME5);
        }
    }

    /**
     * 检查字段是否加锁
     *
     * @param dictTypeCode
     * @return
     */
    private static boolean checkLock(String dictTypeCode) {
        if (dictTypeCode != null && !"".equals(dictTypeCode)) {
            return redisService.get(RedisParam.DICT_TYPE_LOCK + dictTypeCode) != null;
        }
        return false;
    }

    /**
     * 获取路径对应的真实字典类型编码
     *
     * @param path 路径
     * @return
     */
    private static String getRealDictTypeCode(String path) {
        if (StringUtils.isNotEmpty(path)) {
            if (path.contains(PATH_SPLIT)) {
                String[] split = path.split(PATH_SPLIT);
                return split[0];
            }
        }
        return path;
    }

    /**
     * 获取路径下（不含路径）的全部值的map
     *
     * @param path 路径
     * @return
     */
    private static Map<String, String> fullMapByPath(String path) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (path != null && !"".equals(path)) {
            List<DictDataVo> list = listByPath(path, false);
            if (list != null && list.size() > 0) {
                map = list.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
            }
        }
        return map;
    }

    /**
     * 获取路径的下一级的map
     *
     * @param path 路径
     * @return
     */
    private static Map<String, String> singleMapByPath(String path) {
        Map<String, String> map = new LinkedHashMap<>(16);
        if (path != null && !"".equals(path)) {
            List<DictDataVo> list = treeByPath(path, false);
            if (list != null && list.size() > 0) {
                map = list.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
            }
        }
        return map;
    }

    /**
     * 获取路径下（不含路径）的字典集合（树形）
     *
     * @param path       路径
     * @param withExtend 是否解析扩展字段
     * @return
     */
    private static List<DictDataVo> treeByPath(String path, boolean withExtend) {
        List<DictDataVo> tree = new ArrayList<>();
        if (path != null && !"".equals(path)) {
            List<DictDataVo> list = listByPath(path, withExtend);
            if (list != null && list.size() > 0) {
                tree.addAll(generateTree(list));
            }
        }
        return tree;
    }

    /**
     * 获取路径下（不含路径）的字典集合
     *
     * @param path       路径
     * @param withExtend 是否解析扩展字段
     * @return
     */
    private static List<DictDataVo> listByPath(String path, boolean withExtend) {
        if (!StringUtils.isEmpty(path)) {
            String realDictTypeCode = getRealDictTypeCode(path);
            DictGroupRedisVo dictGroup = getDictGroup(realDictTypeCode);
            if (dictGroup != null) {
                List<DictDataVo> list = new ArrayList<>();
                for (DictDataRedisVo item :
                        dictGroup.getDataList()) {
                    String path2 = path + PATH_SPLIT;
                    if (!item.getPath().startsWith(path2)) {
                        continue;
                    }
                    DictDataVo copy = CrmBeanUtil.copy(item, DictDataVo.class);
                    if (withExtend) {
                        if (CollectionUtil.listNotEmptyNotSizeZero(dictGroup.getConfList())) {
                            copy.setExtendMap(generateExtendMap(item, dictGroup.getConfList()));
                        }
                    }
                    list.add(copy);
                }
                return list;
            }
        }
        return new ArrayList<>();
    }

    /**
     * 解析扩展字段
     *
     * @param data     字典值
     * @param confList 扩展字段配置
     * @return
     */
    private static Map<String, String> generateExtendMap(DictDataRedisVo data, List<DictAttrConfRedisVo> confList) {
        if (data != null && confList != null && confList.size() > 0) {
            Map<String, String> map = new HashMap<>();
            for (DictAttrConfRedisVo conf :
                    confList) {
                Object obj = ReflectUtil.getFiledValueByName(data, conf.getExtField());
                map.put(conf.getFieldCode(), obj == null ? "" : (String) obj);
            }
            return map;
        }
        return Collections.emptyMap();
    }

    /**
     * 生成树形结构
     *
     * @param totalList 平铺集合
     * @return
     */
    private static List<DictDataVo> generateTree(List<DictDataVo> totalList) {

        if (CollectionUtil.listEmpty(totalList)) {
            return new ArrayList<>();
        }

        //构建树list
        List<DictDataVo> treeList = new ArrayList<>();
        //当前操作层级数据
        List<DictDataVo> curLevelList = new ArrayList<>();
        //未操作数据
        List<DictDataVo> restList = new ArrayList<>();

        //key:id
        Map<String, DictDataVo> totalMap = totalList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));

        //查找第一层
        for (DictDataVo item : totalList) {
            if (StringUtils.isEmpty(item.getParentDictCode()) || !totalMap.containsKey(item.getParentDictCode())) {
                treeList.add(item);
                curLevelList.add(item);
            } else {
                restList.add(item);
            }
        }

        //构建数据，从第二层开始
        while (curLevelList.size() > 0 && restList.size() > 0) {
            List<DictDataVo> restTempList = new ArrayList<>();
            List<DictDataVo> curLevelTempList = new ArrayList<>();
            Map<String, String> curLevelMap = curLevelList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictCode));
            Map<String, List<DictDataVo>> curLevelChildrenMap = new HashMap<>();

            for (DictDataVo item : restList) {
                if (curLevelMap.containsKey(item.getParentDictCode())) {
                    curLevelTempList.add(item);

                    List<DictDataVo> childrenList = new ArrayList<>();
                    if (curLevelChildrenMap.containsKey(item.getParentDictCode())) {
                        childrenList.addAll(curLevelChildrenMap.get(item.getParentDictCode()));
                    }
                    childrenList.add(item);
                    curLevelChildrenMap.put(item.getParentDictCode(), childrenList);
                } else {
                    restTempList.add(item);
                }
            }

            for (DictDataVo item : curLevelList) {
                if (curLevelChildrenMap.containsKey(item.getDictCode())) {
                    item.setChildren(curLevelChildrenMap.get(item.getDictCode()));
                }
            }

            curLevelList.clear();
            curLevelList.addAll(curLevelTempList);
            restList.clear();
            restList.addAll(restTempList);
        }

        return treeList;
    }
}
