package com.biz.crm.util;

import com.biz.crm.base.BusinessException;
import org.apache.commons.lang3.StringUtils;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;

public class CrmDateUtils {
    private CrmDateUtils(){}
    public static final DateTimeFormatter yyyyMMddHHmmssSSS = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss,SSS");
    public static final DateTimeFormatter yyyyMMddHHmmss = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    public static final DateTimeFormatter yyyyMMddHH_mm_ss = DateTimeFormatter.ofPattern("yyyy-MM-dd HH_mm_ss");
    public static final DateTimeFormatter yyyyMMdd = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter HHmmss = DateTimeFormatter.ofPattern("HH:mm:ss");
    /**
     * 12:00:00
     */
    public static final String TIME_STR_12 = "12:00:00";
    /**
     * 每天最早的时刻
     */
    public static final String TIME_STR_00 = "00:00:00";

    /**
     * 每天最晚的时刻
     */
    public static final String TIME_STR_235959 = "23:59:59";

    /**
     * 字符转日期时间
     * @param dateTime
     * @return
     */
    public static LocalDateTime parseyyyyMMddHHmmss(String dateTime){
        try {
            return LocalDateTime.parse(dateTime, yyyyMMddHHmmss);
        }catch (Exception e){
            throw new BusinessException("格式化日期时间失败！", e);
        }

    }

    /**
     * date + time 字符转日期时间
     *
     * @param date 日期
     * @param time 时间
     * @param maxTime 当时间为空或为 00:00:00 时, false:使用最小时间(00:00:00),true:使用最大时间(23:59:59)
     * @return
     */
    public static LocalDateTime parseCrmDateTime(String date, String time, boolean maxTime){
        if(StringUtils.isBlank(date)){
            return null;
        }
        String onEmptyTime = maxTime ? TIME_STR_235959 : TIME_STR_00;
        return parseyyyyMMddHHmmss(date + " " + (StringUtils.isBlank(time) || TIME_STR_00.equals(time) ? onEmptyTime : time));
    }

    /**
     * 验证日期格式
     * @param date
     * @return
     */
    public static boolean validFormatYYYY_MM_DD(String date) {
        try {
            LocalDate.parse(date);
            return true;
        } catch (Exception e){}
        return false;
    }

    /**
     * 日期比较。第一个在第二个之前
     * @param firstDate YYYY-MM-DD
     * @param secondDate YYYY-MM-DD
     * @return
     */
    public static boolean isBefore(String firstDate, String secondDate) {
        return LocalDate.parse(firstDate).isBefore(LocalDate.parse(secondDate));
    }

    /**
     * 日期比较。第一个在第二个之后
     * @param firstDate YYYY-MM-DD
     * @param secondDate YYYY-MM-DD
     * @return
     */
    public static boolean isAfter(String firstDate, String secondDate) {
        return LocalDate.parse(firstDate).isAfter(LocalDate.parse(secondDate));
    }

    /**
     * 传入时间查询此时间所属的周的周1 返回yyyyMMdd
     * @param localDate
     * @return
     */
    public static String gainMonday(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.previous(DayOfWeek.MONDAY)).toString();
    }

    /**
     * 传入时间查询此时间所属的周的周日 返回yyyyMMdd
     * @param localDate
     * @return
     */
    public static String gainSunday(LocalDate localDate) {
        return localDate.plusDays(7).with(TemporalAdjusters.previous(DayOfWeek.SUNDAY)).toString();
    }

    /**
     * 传入时间查询此时间所属的月的1号 返回yyyyMMdd
     * @param localDate
     * @return
     */
    public static String gainFirstMonth(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.firstDayOfMonth()).toString();
    }

    /**
     * 传入时间查询此时间所属的月的30/31号 返回yyyyMMdd
     * @param localDate
     * @return
     */
    public static String gainLastMonth(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.lastDayOfMonth()).toString();
    }


}
