package com.biz.crm.excel.component;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.upload.UploadEnum;
import com.biz.crm.excel.util.DefaultExportContext;
import com.biz.crm.mdm.tableconfig.MdmFunctionSubFeign;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.upload.excel.req.ExcelExportReqVo;
import com.biz.crm.nebular.upload.vo.UploadVo;
import com.biz.crm.upload.excel.ExcelExportFeign;
import com.biz.crm.util.CrmDateUtils;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.util.Result;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.Duration;
import java.time.LocalDateTime;


/**
 *  @author: luoqi
 *  @Date: 2021-1-5 14:58
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
@Component
public class ExportHelper {
    //MQ服务
    @Resource
    private RocketMQProducer rocketMQProducer;
    @Resource
    private ExcelExportFeign excelExportFeign;
    @Resource
    private MdmFunctionSubFeign mdmFunctionSubFeign;
    /**
     * excel保存至下载中心
     * @param exportContext
     */
    public void saveExcelFile(DefaultExportContext exportContext) {
        ExcelExportReqVo excelExportReqVo = this.buildExcelExportReqVo(exportContext);
        Result<ExcelExportReqVo> result = excelExportFeign.saveExcelFileBackId(excelExportReqVo);
        ExcelExportReqVo reqVo = ApiResultUtil.objResult(result, true);
        String id = reqVo.getId();
        if(StringUtils.isBlank(id)){
            this.sendWebsocketMsgMQ("保存excel到文件中心失败", exportContext);
            throw new BusinessException("保存excel到文件中心失败");
        }
        exportContext.setTaskId(reqVo.getId());
        exportContext.setTaskCode(reqVo.getFileCode());
    }

    /**
     * 处理异常情况
     * @param exportContext
     * @param msg
     */
    public void processException(DefaultExportContext exportContext, String msg){
        exportContext.setErrorInfo(msg);
        this.sendWebsocketMsgMQ(msg, exportContext);
        exportContext.setErrorInfo(msg);
        exportContext.setTaskEndTime(LocalDateTime.now());
        this.updateExcelFile(exportContext, false);
        throw new BusinessException(msg);
    }
    /**
     * 更新excel下载文件
     *
     * @param exportContext
     */
    public void updateExcelFile(DefaultExportContext exportContext, boolean success) {
        //校验并封装参数信息
        ExcelExportReqVo excelExportReqVo = this.checkUpdateParam(exportContext, success);
        Duration duration = Duration.between(exportContext.getTaskBeginTime(), exportContext.getTaskEndTime());
        //相差的分钟数
        String minutes = String.valueOf(duration.toMillis());
        excelExportReqVo.setTimeSpent(minutes);
        ApiResultUtil.checkResult(excelExportFeign.updateExcelFileById(excelExportReqVo));
    }
    /**
     * 校验更新参数信息
     *
     * @param exportContext
     * @return
     */
    private ExcelExportReqVo checkUpdateParam(DefaultExportContext exportContext, boolean success) {
        ExcelExportReqVo excelExportReqVo = new ExcelExportReqVo();
        excelExportReqVo.setId(exportContext.getTaskId());
        if(success) {
//            requestParam.put("fileName", excelJsonObject.get("fileName"));
//            requestParam.put("objectName", excelJsonObject.get("objectName"));
//            requestParam.put("filePath", excelJsonObject.get("url"));
//            exportContext.getUploadResultStr();
            excelExportReqVo.setFileName(exportContext.getExcelFileName());
            UploadVo uploadVo = exportContext.getUploadVo();
            excelExportReqVo.setObjectName(uploadVo.getObjectName());
            excelExportReqVo.setFilePath(uploadVo.getUrl());
            excelExportReqVo.setFileStatus(UploadEnum.fileStatus.EXPORT_SUCCESS.getVal());

        } else {
            excelExportReqVo.setRemarks(exportContext.getErrorInfo());
            excelExportReqVo.setFileStatus(UploadEnum.fileStatus.EXPORT_FAIL.getVal());
        }

        return excelExportReqVo;
    }


    /**
     * 组装下载中心任务对象信息
     *
     * @param exportContext
     */
    private ExcelExportReqVo buildExcelExportReqVo(DefaultExportContext exportContext) {
        String excelFileName = exportContext.getFunctionName() + "_" + exportContext.getTaskCreatedTime().format(CrmDateUtils.yyyyMMddHH_mm_ss);
        exportContext.setExcelFileName(excelFileName);

        ExcelExportReqVo excelExportReqVo = new ExcelExportReqVo();
        excelExportReqVo.setExcelFileName(excelFileName);
        excelExportReqVo.setFileParam(JsonPropertyUtil.toJsonString(exportContext.getExportParam()));
        excelExportReqVo.setFileSource(exportContext.getFunctionName());
        excelExportReqVo.setFileType(UploadEnum.fileType.EXPORT.getVal());
        excelExportReqVo.setWebSocketId(exportContext.getExportParam().getSid());
        excelExportReqVo.setFileStatus(UploadEnum.fileStatus.EXPORT_IN_EXECUTION.getVal());
        return excelExportReqVo;
    }


    /**
     * 发送 WebsocketMsg
     * @param msg
     * @param exportContext
     */
    public void sendWebsocketMsgMQ(String msg, DefaultExportContext exportContext){
        String webSocketClientId = exportContext.getExportParam().getSid();
        if(StringUtils.isBlank(webSocketClientId)){
            if(log.isInfoEnabled()){
                log.info("未提交 websocket id，忽略 websocket 消息发送！");
            }
            return;
        }
        JSONObject requestParam = new JSONObject();
        requestParam.put("sid",webSocketClientId);
        requestParam.put("websocketMessage",msg);
        RocketMQMessageBody body = new RocketMQMessageBody() {{
            this.setTag(RocketMQConstant.CRM_MQ_TAG.WEBSOCKET_SESSION);
            this.setMsgBody(JSONObject.toJSONString(requestParam));
        }};
        rocketMQProducer.convertAndSend(body);
    }

    /**
     * 发送 WebsocketMsg
     * @param msg
     */
    public void sendWebsocketMsgMQ(String msg, String sid){
        if(StringUtils.isBlank(sid)){
            if(log.isInfoEnabled()){
                log.info("未提交 websocket id，忽略 websocket 消息发送！");
            }
            return;
        }
        JSONObject requestParam = new JSONObject();
        requestParam.put("sid",sid);
        requestParam.put("websocketMessage",msg);
        RocketMQMessageBody body = new RocketMQMessageBody() {{
            this.setTag(RocketMQConstant.CRM_MQ_TAG.WEBSOCKET_SESSION);
            this.setMsgBody(JSONObject.toJSONString(requestParam));
        }};
        rocketMQProducer.convertAndSend(body);
    }

    /**
     * 删除任务
     * @param msg
     * @param exportContext
     */
    public void delTask(String msg, DefaultExportContext exportContext){
        this.sendWebsocketMsgMQ(msg, exportContext);
        ExcelExportReqVo reqVo = new ExcelExportReqVo() {{
            this.setIds(Lists.newArrayList(exportContext.getTaskId()));
        }};
        ApiResultUtil.checkResult(excelExportFeign.delete(reqVo));
    }
}
