package com.biz.crm.excel.component.saver.mdm.position;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.excel.component.saver.AbstractExcelImportSaver;
import com.biz.crm.excel.component.saver.ExcelImportSaver;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.position.MdmPositionImportVo;
import com.biz.crm.mdm.position.entity.MdmPositionBpmRoleEntity;
import com.biz.crm.mdm.position.entity.MdmPositionEntity;
import com.biz.crm.mdm.position.entity.MdmPositionRoleEntity;
import com.biz.crm.mdm.position.mapper.MdmPositionBpmRoleMapper;
import com.biz.crm.mdm.position.mapper.MdmPositionMapper;
import com.biz.crm.mdm.position.mapper.MdmPositionRoleMapper;
import com.biz.crm.mdm.positionlevel.entity.MdmPositionLevelEntity;
import com.biz.crm.mdm.positionlevel.mapper.MdmPositionLevelMapper;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionBpmRoleRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.util.*;
import com.biz.crm.util.websocket.WebsocketUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;


@Slf4j
@Component("mdmPositionImportSaver")
@Transactional("mdmTransactionManager")
public class MdmPositionImportSaver<M extends BaseMapper<T>, T> extends AbstractExcelImportSaver<MdmPositionMapper, MdmPositionEntity, MdmPositionImportVo> implements ExcelImportSaver<MdmPositionImportVo> {

    @Resource
    private MdmPositionMapper mdmPositionMapper;
    @Resource
    private MdmPositionLevelMapper mdmPositionLevelMapper;
    @Resource
    private MdmPositionRoleMapper mdmPositionRoleMapper;
    @Resource
    private MdmPositionBpmRoleMapper mdmPositionBpmRoleMapper;
    @Autowired
    private WebsocketUtil websocketUtil;
    @Resource
    private RocketMQProducer rocketMQProducer;

    @Override
    public void save(List<MdmPositionImportVo> data, DefaultImportContext context) {
        log.info("职位导入:{}" + data.size());
        if (!CollectionUtils.isEmpty(data)) {
            String sid = context.getImportParamVo().getWebSocketClientId();
            websocketUtil.sendMsg(sid, "开始组装保存数据");
            List<MdmPositionEntity> entityList = new ArrayList<>();
            List<MdmPositionBpmRoleEntity> positionBpmRoleEntityList = new ArrayList<>();
            List<MdmPositionRoleEntity> positionRoleEntityList = new ArrayList<>();

            List<MdmPositionImportVo> collect = data.stream().filter(x -> MdmPositionImportVo.ProcessTypeEnum.SUCCESS == x.getProcessType()).collect(Collectors.toList());
            generatePositionName(collect);
            //批量获取规则编码
//            List<String> codes = CodeUtil.generateCodeList(CodeRuleEnum.MDM_POSITION_CODE.getCode(), collect.size());
//            int i = 0;
            for (MdmPositionImportVo item :
                    collect) {
                MdmPositionEntity copy = CrmBeanUtil.copy(item, MdmPositionEntity.class);
//                copy.setPositionCode(codes.get(i));
//                i++;
                copy.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                copy.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
                copy.setRuleCode(UUID.randomUUID().toString().replace("-", ""));
                copy.setLevelNum(1);
                entityList.add(copy);

                if (StringUtils.isNotEmpty(item.getRoleCode())) {
                    positionRoleEntityList.addAll(Arrays.stream(item.getRoleCode().split(",")).map(x -> {
                        MdmPositionRoleEntity positionRoleEntity = new MdmPositionRoleEntity();
                        positionRoleEntity.setPositionCode(copy.getPositionCode());
                        positionRoleEntity.setRoleCode(x);
                        return positionRoleEntity;
                    }).collect(Collectors.toList()));
                }

                if (StringUtils.isNotEmpty(item.getBpmRoleCode())) {
                    positionBpmRoleEntityList.addAll(Arrays.stream(item.getBpmRoleCode().split(",")).map(x -> {
                        MdmPositionBpmRoleEntity positionRoleEntity = new MdmPositionBpmRoleEntity();
                        positionRoleEntity.setPositionCode(copy.getPositionCode());
                        positionRoleEntity.setBpmRoleCode(x);
                        return positionRoleEntity;
                    }).collect(Collectors.toList()));
                }
            }

            websocketUtil.sendMsg(sid, "结束组装保存数据");

            websocketUtil.sendMsg(sid, "开始保存数据");
            if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                List<List<MdmPositionEntity>> partition = Lists.partition(entityList, 50);
                websocketUtil.sendMsg(sid, "开始保存数据===>开始保存职位");
                for (List<MdmPositionEntity> item :
                        partition) {
                    this.saveBatch(item);
                    this.sendPositionAdd(CrmBeanUtil.copyList(item, MdmPositionRespVo.class));
                }
                websocketUtil.sendMsg(sid, "开始保存数据===>结束保存职位===>" + entityList.size());

                websocketUtil.sendMsg(sid, "开始保存数据===>开始保存职位关联角色");
                if (CollectionUtil.listNotEmptyNotSizeZero(positionRoleEntityList)) {
                    for (MdmPositionRoleEntity item :
                            positionRoleEntityList) {
                        mdmPositionRoleMapper.insert(item);
                    }
                }
                websocketUtil.sendMsg(sid, "开始保存数据===>结束保存职位关联角色===>" + positionRoleEntityList.size());

                websocketUtil.sendMsg(sid, "开始保存数据===>开始保存职位关联流程角色");
                if (CollectionUtil.listNotEmptyNotSizeZero(positionBpmRoleEntityList)) {
                    for (MdmPositionBpmRoleEntity item :
                            positionBpmRoleEntityList) {
                        mdmPositionBpmRoleMapper.insert(item);
                    }
                    this.sendPositionRelBpmRoleAdd(CrmBeanUtil.copyList(positionBpmRoleEntityList, MdmPositionBpmRoleRespVo.class));
                }
                websocketUtil.sendMsg(sid, "开始保存数据===>结束保存职位关联流程角色===>" + positionBpmRoleEntityList.size());

                websocketUtil.sendMsg(sid, "开始全表重置职位降维编码");
                resetRuleCode(sid);
                websocketUtil.sendMsg(sid, "结束全表重置职位降维编码");
                websocketUtil.sendMsg(sid, "结束保存数据");
            }
        }
    }

    /**
     * 全表重置降维编码，方法需要优化
     */
    private void resetRuleCode(String sid) {
        long l = System.currentTimeMillis();
        log.info("-----------重置职位降维编码（导入）_" + l + "_开始-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）开始");
        mdmPositionMapper.copyIdToRuleCode();
        log.info("-----------重置职位降维编码（导入）_" + l + "_编码重置成id成功-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）编码重置成id成功");
        mdmPositionMapper.setNullNotExistParentCode();
        log.info("-----------重置职位降维编码（导入）_" + l + "_清除无效上级编码成功-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）清除无效上级编码成功");
        QueryWrapper<MdmPositionEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("del_flag", CrmDelFlagEnum.NORMAL.getCode());
        wrapper.and(x -> x.eq("parent_code", "").or().isNull("parent_code"));
        List<MdmPositionEntity> list = mdmPositionMapper.selectList(wrapper);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            for (int i = 0; i < list.size(); i++) {
                updateCurAndChildren(list.get(i).getPositionCode(), TreeRuleCodeUtil.numToSingleCode(i + 1), 1);
            }
        }
        log.info("-----------重置职位降维编码（导入）_" + l + "_编码重新生成成功-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）编码重新生成成功");
        PositionUtil.deleteAllCache();
        log.info("-----------重置职位降维编码（导入）_" + l + "_缓存清除成功-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）缓存清除成功");
        log.info("-----------重置职位降维编码（导入）_" + l + "_结束-----------");
        websocketUtil.sendMsg(sid, "重置职位降维编码（导入）结束");
    }

    /**
     * 更新职位及职位下级降维编码和层级
     *
     * @param positionCode 当前职位编码
     * @param curCode      当前职位降维编码
     * @param levelNum     当前层级
     */
    private void updateCurAndChildren(String positionCode, String curCode, int levelNum) {

        //更新当前
        this.lambdaUpdate()
                .eq(MdmPositionEntity::getPositionCode, positionCode)
                .set(MdmPositionEntity::getRuleCode, curCode)
                .set(MdmPositionEntity::getLevelNum, levelNum)
                .update();

        //查询下一层
        List<MdmPositionEntity> list = this.lambdaQuery()
                .eq(MdmPositionEntity::getParentCode, positionCode)
                .eq(MdmPositionEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .select(MdmPositionEntity::getPositionCode)
                .list();

        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            //遍历下级
            for (int i = 0; i < list.size(); i++) {
                //递归调用
                updateCurAndChildren(list.get(i).getPositionCode(), curCode + TreeRuleCodeUtil.numToSingleCode(i + 1), (levelNum + 1));
            }
        }
    }

    /**
     * 生成职位名称
     *
     * @param collect
     */
    public void generatePositionName(List<MdmPositionImportVo> collect) {
        if (collect != null && collect.size() > 0) {
            List<MdmPositionImportVo> emptyPositionNameList = collect.stream().filter(x -> StringUtils.isEmpty(x.getPositionName())).collect(Collectors.toList());
            if (!emptyPositionNameList.isEmpty()) {
                Set<String> existNameSet = collect.stream().filter(x -> StringUtils.isNotEmpty(x.getPositionName())).map(MdmPositionImportVo::getPositionName).collect(Collectors.toSet());
                Map<String, List<MdmPositionImportVo>> groupByPositionLevelCode = emptyPositionNameList.stream().collect(Collectors.groupingBy(MdmPositionImportVo::getPositionLevelCode));
                for (Map.Entry<String, List<MdmPositionImportVo>> entry :
                        groupByPositionLevelCode.entrySet()) {
                    generatePositionNameGroupByPositionLevel(entry.getKey(), entry.getValue(), existNameSet);
                }
            }
        }
    }

    /**
     * 分组生成职位名称
     *
     * @param positionLevelCode
     * @param collect
     * @param generateNameSet
     */
    public void generatePositionNameGroupByPositionLevel(String positionLevelCode, List<MdmPositionImportVo> collect, Set<String> generateNameSet) {
        Assert.hasText(positionLevelCode, "缺失职位级别编码");

        QueryWrapper<MdmPositionLevelEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("position_level_code", positionLevelCode);
        MdmPositionLevelEntity positionLevel = mdmPositionLevelMapper.selectOne(wrapper);
        Assert.notNull(positionLevel, "无效的职位级别");
        Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(positionLevel.getEnableStatus()), "该职位级别已停用");

        Integer suffixSequence = positionLevel.getSuffixSequence();
        if (suffixSequence == null) {
            suffixSequence = 0;
        }
        Set<String> positionNameSet = new HashSet<>(16);
        List<MdmPositionEntity> positionList = this.lambdaQuery()
                .like(MdmPositionEntity::getPositionName, positionLevel.getPositionLevelName())
                .select(MdmPositionEntity::getPositionName)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(positionList)) {
            positionNameSet.addAll(positionList.stream().map(MdmPositionEntity::getPositionName).collect(Collectors.toSet()));
        }
        for (MdmPositionImportVo item :
                collect) {
            String positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
            while (positionNameSet.contains(positionName) || generateNameSet.contains(positionName)) {
                positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
            }
            item.setPositionName(positionName);
            generateNameSet.add(positionName);
        }
        positionLevel.setSuffixSequence(suffixSequence);
        mdmPositionLevelMapper.updateById(positionLevel);
    }

    protected void sendPositionAdd(List<MdmPositionRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位新增发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_ADD_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位新增发送MQ消息成功：{}", list);
    }

    protected void sendPositionRelBpmRoleAdd(List<MdmPositionBpmRoleRespVo> list) {
        if (list == null || list.size() == 0) {
            return;
        }
        log.info("职位关联流程角色发送MQ消息：{}", list);
        RocketMQMessageBody rocketMQMessageBody = new RocketMQMessageBody();
        rocketMQMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.POSITION_REL_BPM_ROLE_ADD_TAG);
        rocketMQMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(list));
        rocketMQProducer.convertAndSend(rocketMQMessageBody);
        log.info("职位关联流程角色发送MQ消息成功：{}", list);
    }

}
