package com.biz.crm.excel.component.validator.dms.order;

import com.biz.crm.eunm.dms.OrderEunm;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.mdm.availablelist.AvailableListFeign;
import com.biz.crm.nebular.dms.order.OrderImportVo;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.util.CollectionUtil;
import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.assertj.core.util.Lists;
import org.springframework.stereotype.Component;

/**
 * 订单导入校验类
 * @Author: chenrong
 * @Date: 2021/3/23 11:47
 */
@Component("dmsOrderValidator")
public class DmsOrderValidator implements ExcelImportValidator<OrderImportVo> {

  private static final String DATE_REGEX = "((\\d{2}(([02468][048])|([13579][26]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])))))|(\\d{2}(([02468][1235679])|([13579][01345789]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|(1[0-9])|(2[0-8]))))))(\\s(((0?[0-9])|([1][0-9])|([2][0-4]))\\:([0-5]?[0-9])((\\s)|(\\:([0-5]?[0-9])))))?";

  private static final String NUMBER_REGEX = "[+-]?\\d+(\\.\\d+)?";

  @Resource
  private AvailableListFeign availableListFeign;

  @Override
  public void validate(List<OrderImportVo> list, DefaultImportContext defaultImportContext) {
    list.forEach(li -> {
      this.validate(li.getCusCode(), li, "客户编码不能为空");
      this.validate(li.getCusName(), li, "客户名称不能为空");
      li.setOrderTypeCode(getOrderTypeCodeByName(li.getOrderTypeName()));
      this.validate(li.getOrderTypeCode(), li, "订单类型不正确");
      this.validate(li.getPickAddressCode(), li, "送达方不能为空");
      this.validate(li.getPickDate(), li, "送货日期不能为空");
      this.validatePattern(li.getPickDate(), li, DATE_REGEX, "送货日期格式不正确");
      li.setPickDate(li.getPickDate().replace("/", "-"));
      this.validate(li.getProductCode(), li, "商品编码不能为空");
      this.validate(li.getCount(), li, "购买数量不能为空");
      this.validatePattern(li.getCount(), li, NUMBER_REGEX, "购买数量格式不正确");
    });
    Map<String, List<OrderImportVo>> orderMap = list.stream().collect(Collectors.groupingBy(OrderImportVo::getCusCode));
    orderMap.forEach((k, v) -> {
      Set<String> productCodes = v.stream().collect(Collectors.groupingBy(OrderImportVo::getProductCode)).keySet();
      this.validateAndSetPrice(v, k, productCodes);
    });
  }

  /**
   * 校验并设置价格
   * @param orderList
   * @param cusCode
   * @param productCodes
   */
  private void validateAndSetPrice(List<OrderImportVo> orderList, String cusCode, Set<String> productCodes) {
    Map<String, CusAvailablelistVo> result = availableListFeign.
            findGoodsByCusAndGoodsCodes(Lists.newArrayList(productCodes), cusCode).getResult();
    if(!CollectionUtil.mapNotEmpty(result)) {
      result = Maps.newHashMap();
    }
    for (OrderImportVo order : orderList) {
      if (Objects.isNull(result.get(order.getProductCode()))) {
        order.appendErrorValidateMsg("客户".concat(order.getCusCode()).concat("-商品").concat(order.getProductCode()).concat("不可购，请先维护可购清单"));
        continue;
      }
      if (result.get(order.getProductCode()).getPrice() == null) {
        order.appendErrorValidateMsg("客户".concat(order.getCusCode()).concat("-商品").concat(order.getProductCode()).concat("价格不可用，请先维护价格"));
        continue;
      }
      order.setPrice(result.get(order.getProductCode()).getPrice());
      order.setAmount(new BigDecimal(order.getCount()).multiply(order.getPrice()));
    }

  }

  /**
   * 根据订单类型名称获取编码
   * @param name
   * @return
   */
  private String getOrderTypeCodeByName(String name) {
    OrderEunm.TYPE[] types = OrderEunm.TYPE.values();
    for (OrderEunm.TYPE type : types) {
      if (Objects.equals(type.getValue(), name)) {
        return type.getCode();
      }
    }
    return null;
  }

  /**
   * 校验数据格式
   * @param value
   * @param li
   * @param regex
   * @param msg
   */
  private void validatePattern(String value, OrderImportVo li, String regex, String msg) {
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(value);
    if (!matcher.matches()) {
      li.appendErrorValidateMsg(msg);
    }
  }

  /**
   * 校验单个字段非空
   * @param value
   * @param li
   * @param msg
   */
  private void validate(Object value, OrderImportVo li, String msg) {
    if (Objects.isNull(value) || Objects.equals("", value)) {
      li.appendErrorValidateMsg(msg);
    }
  }
}
