package com.biz.crm.excel.component.validator.dms.order;

import com.biz.crm.eunm.dms.OrderEunm;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.mdm.availablelist.AvailableListFeign;
import com.biz.crm.nebular.dms.order.OrderImportVo;
import com.biz.crm.nebular.dms.order.OrderUpdateImportVo;
import com.biz.crm.nebular.mdm.availablelist.CusAvailablelistVo;
import com.biz.crm.util.CollectionUtil;
import com.google.common.collect.Maps;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.assertj.core.util.Lists;
import org.springframework.stereotype.Component;

/**
 * @Author: chenrong
 * @Date: 2021/3/24 17:56
 */
@Component("orderUpdateValidator")
public class OrderUpdateValidator implements ExcelImportValidator<OrderUpdateImportVo> {

  private static final String NUMBER_REGEX = "[+-]?\\d+(\\.\\d+)?";

  @Resource
  private AvailableListFeign availableListFeign;

  /**
   * 导入校验方法
   * @param data
   * @param context
   * @return
   */
  @Override
  public void validate(List<OrderUpdateImportVo> data, DefaultImportContext context) {
    data.forEach(li -> {
      this.validate(li.getCusCode(), li, "客户编码不能为空");
      this.validate(li.getCusName(), li, "客户名称不能为空");
      this.validate(li.getOrderCode(), li, "订单编码不能为空");
      this.validate(li.getDetailId(), li, "订单明细id不能为空");
      li.setOrderTypeCode(getOrderTypeCodeByName(li.getOrderTypeName()));
      this.validate(li.getProductCode(), li, "商品编码不能为空");
      this.validate(li.getCount(), li, "购买数量不能为空");
      this.validate(li.getProductType(), li, "商品类型不能为空");
      this.validateProductType(li.getProductType(), li, "商品类型不正确");
      this.validatePattern(li.getCount(), li, NUMBER_REGEX, "购买数量格式不正确");
    });
    Map<String, List<OrderUpdateImportVo>> orderMap = data.stream().collect(Collectors.groupingBy(OrderUpdateImportVo::getOrderCode));
    orderMap.forEach((k, v) -> {
      Set<String> productCodes = v.stream().collect(Collectors.groupingBy(OrderUpdateImportVo::getProductCode)).keySet();
      this.validateAndSetPrice(v, k, productCodes);
    });
  }

  /**
   * 校验商品类型
   * @param productType
   * @param li
   * @param msg
   */
  private void validateProductType(String productType, OrderUpdateImportVo li, String msg) {
    if(OrderEunm.LineTypeEnum.getByCode(productType) == null) {
      li.appendErrorValidateMsg(msg);
    }
  }

  /**
   * 校验并设置价格
   * @param orderList
   * @param cusCode
   * @param productCodes
   */
  private void validateAndSetPrice(List<OrderUpdateImportVo> orderList, String cusCode, Set<String> productCodes) {
    Map<String, CusAvailablelistVo> result = availableListFeign.
            findGoodsByCusAndGoodsCodes(Lists.newArrayList(productCodes), cusCode).getResult();
    if(!CollectionUtil.mapNotEmpty(result)) {
      result = Maps.newHashMap();
    }
    for (OrderUpdateImportVo order : orderList) {
      if (Objects.isNull(result.get(order.getProductCode()))) {
        order.appendErrorValidateMsg("客户".concat(order.getCusCode()).concat("-商品").concat(order.getProductCode()).concat("不可购，请先维护可购清单"));
        continue;
      }
      if (result.get(order.getProductCode()).getPrice() == null) {
        order.appendErrorValidateMsg("客户".concat(order.getCusCode()).concat("-商品").concat(order.getProductCode()).concat("价格不可用，请先维护价格"));
        continue;
      }
      order.setPrice(result.get(order.getProductCode()).getPrice());
      order.setAmount(new BigDecimal(order.getCount()).multiply(order.getPrice()));
    }

  }

  /**
   * 根据订单类型名称获取编码
   * @param name
   * @return
   */
  private String getOrderTypeCodeByName(String name) {
    OrderEunm.TYPE[] types = OrderEunm.TYPE.values();
    for (OrderEunm.TYPE type : types) {
      if (Objects.equals(type.getValue(), name)) {
        return type.getCode();
      }
    }
    return null;
  }

  /**
   * 校验数据格式
   * @param value
   * @param li
   * @param regex
   * @param msg
   */
  private void validatePattern(String value, OrderUpdateImportVo li, String regex, String msg) {
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(value);
    if (!matcher.matches()) {
      li.appendErrorValidateMsg(msg);
    }
  }

  /**
   * 校验单个字段非空
   * @param value
   * @param li
   * @param msg
   */
  private void validate(Object value, OrderUpdateImportVo li, String msg) {
    if (Objects.isNull(value) || Objects.equals("", value)) {
      li.appendErrorValidateMsg(msg);
    }
  }
}
