package com.biz.crm.excel.component.validator.kms.orderform;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.common.AbstractImportVo;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.confadmin.model.KmsDirectSystemEntity;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.kms.orderform.KmsOrderFormImportVo;
import com.biz.crm.kaproduct.model.KmsTenantryDirectProductEntity;
import com.biz.crm.kms.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.kms.confadmin.mapper.KmsDirectSystemMapper;
import com.biz.crm.kms.form.orderform.mapper.KmsOrderFormDetailDwMapper;
import com.biz.crm.kms.form.orderform.mapper.KmsOrderGoodsDetailDwMapper;
import com.biz.crm.kms.kaproduct.mapper.KmsTenantryDirectProductMapper;
import com.biz.crm.kms.tenantrydirectcustomerorg.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.nebular.kms.kaproduct.req.KmsTenantryDirectProductReqVo;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectCustomerOrgReqVo;
import com.biz.crm.rawdata.model.KmsOrderFormDetailDwEntity;
import com.biz.crm.rawdata.model.KmsOrderGoodsDetailDwEntity;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.util.DateUtil;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.assertj.core.util.Lists;
import org.assertj.core.util.Sets;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author maoshen
 * @date 2021/8/26.
 */
@Slf4j
@Component("kmsOrderFormValidator")
public class KmsOrderFormValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<KmsOrderFormDetailDwMapper, KmsOrderFormDetailDwEntity, KmsOrderFormImportVo> implements ExcelImportValidator<KmsOrderFormImportVo> {

    @Value("${kms.config.tenantry-id:}")
    private String tenantryId;

    @Resource
    private KmsDirectSystemMapper kmsDirectSystemMapper;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;

    @Resource
    private KmsTenantryDirectProductMapper kmsTenantryDirectProductMapper;

    @Resource
    private KmsTenantryDirectCustomerOrgMapper kmsTenantryDirectCustomerOrgMapper;

    @Resource
    private KmsOrderGoodsDetailDwMapper kmsOrderGoodsDetailDwMapper;

    private Set<String> kaOrderNumSet = Sets.newHashSet();

    private static final Integer DATE_SIZE = 8;

    @Override
    public void validate(List<KmsOrderFormImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        // 检验商超
        validateSystem(data);

        // 检验商超门店
        validateKaStore(data);

        // 检验客户产品
        validateKaGoods(data);

        // 非空检验
        validateNotEmpty(data);

        // 转换数据
        convertData(data);

        // 重复检验
        validateExist(data);

        Map<String, List<KmsOrderFormImportVo>> validateFailMap = data.stream()
                .filter(o -> AbstractImportVo.ProcessTypeEnum.FAIL.equals(o.getProcessType()))
                .collect(Collectors.groupingBy(KmsOrderFormImportVo::getKaOrderNumber));
        data.stream().filter(o -> AbstractImportVo.ProcessTypeEnum.SUCCESS.equals(o.getProcessType())).forEach(o -> {
            List<KmsOrderFormImportVo> importFailVos = validateFailMap.get(o.getKaOrderNumber());
            if (!org.springframework.util.CollectionUtils.isEmpty(importFailVos)) {
                o.appendErrorValidateMsg("该单据存在校验失败的数据！");
            }
        });
    }

    /**
     * 检验重复
     *
     * @param data
     */
    private void validateExist(List<KmsOrderFormImportVo> data) {
        if (CollectionUtils.isEmpty(kaOrderNumSet)) {
            return;
        }
        List<KmsOrderGoodsDetailDwEntity> entities = kmsOrderGoodsDetailDwMapper
                .selectList(Wrappers.<KmsOrderGoodsDetailDwEntity>lambdaQuery()
                        .in(KmsOrderGoodsDetailDwEntity::getKaOrderNumber, kaOrderNumSet));
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        Map<String, KmsOrderGoodsDetailDwEntity> goodsDetailDwEntityMap = entities.stream()
                .collect(Collectors.toMap(o -> o.getKaOrderNumber() + "#" + o.getVersionNumber()
                        + "#" + o.getKaGoodsCode(), Function.identity()));
        data.forEach(x -> {
            String onlyKey = x.getKaOrderNumber() + "#" + x.getVersionNumber()
                    + "#" + x.getKaGoodsCode();
            if (goodsDetailDwEntityMap.containsKey(onlyKey)) {
                x.appendErrorValidateMsg("该单据商品已存在");
            }
        });
    }

    /**
     * 转换数据
     *
     * @param data
     */
    private void convertData(List<KmsOrderFormImportVo> data) {
        Set<String> kaOrderSet = Sets.newHashSet();
        data.forEach(x -> {
            x.setTenantryId(tenantryId);
            x.setInvoicesSource(KmsAdminEnum.AutoType.IMPORT.getDescription());
            kaOrderSet.add(x.getKaOrderNumber());
        });
        this.kaOrderNumSet = kaOrderSet;
    }

    /**
     * 检验商超
     *
     * @param data
     */
    private void validateSystem(List<KmsOrderFormImportVo> data) {
        Set<String> systemNames = Sets.newHashSet();
        data.forEach(x -> {
            if (StringUtils.isBlank(x.getKaName())) {
                x.appendErrorValidateMsg("【商超系统】未填写、");
            }
            if (StringUtils.isNotBlank(x.getKaName())) {
                systemNames.add(x.getKaName());
            }
        });
        if (CollectionUtils.isNotEmpty(systemNames)) {
            List<KmsDirectSystemEntity> entities = kmsDirectSystemMapper.selectList(Wrappers.<KmsDirectSystemEntity>lambdaQuery()
                    .in(KmsDirectSystemEntity::getDirectName, systemNames)
                    .eq(KmsDirectSystemEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .eq(KmsDirectSystemEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
            if (CollectionUtils.isEmpty(entities)) {
                return;
            }
            Map<String, KmsDirectSystemEntity> systemEntityMap = entities.stream()
                    .collect(Collectors.toMap(KmsDirectSystemEntity::getDirectName, Function.identity()));
            Set<String> directIds = Sets.newHashSet();
            data.forEach(x -> {
                if (StringUtils.isNotBlank(x.getKaName())) {
                    if (systemEntityMap.containsKey(x.getKaName())) {
                        x.setKaCode(systemEntityMap.get(x.getKaName()).getId());
                        directIds.add(x.getKaCode());
                    } else {
                        x.appendErrorValidateMsg("【商超系统(" + x.getKaName() + "）】不在维护中、");
                    }
                }
            });
        }
    }


    /**
     * 检验商超门店
     *
     * @param data
     */
    private void validateKaStore(List<KmsOrderFormImportVo> data) {
        Set<String> storeCodeSet = Sets.newHashSet();
        Set<String> storeNameSet = Sets.newHashSet();
        data.forEach(x -> {
            if (StringUtils.isBlank(x.getKaStoreCode()) || StringUtils.isBlank(x.getKaStoreName())) {
                x.appendErrorValidateMsg("【商超门店编码或名称】未填写、");
            } else {
                storeCodeSet.add(x.getKaStoreCode());
                storeNameSet.add(x.getKaStoreName());
            }
        });
        if (CollectionUtils.isEmpty(storeCodeSet) || CollectionUtils.isEmpty(storeNameSet)) {
            return;
        }
        List<KmsDirectStoreEntity> storeEntities = kmsDirectStoreMapper
                .selectList(Wrappers.<KmsDirectStoreEntity>lambdaQuery()
                        .in(KmsDirectStoreEntity::getStoreCode, storeCodeSet)
                        .in(KmsDirectStoreEntity::getStoreName, storeNameSet)
                        .eq(KmsDirectStoreEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                        .eq(KmsDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
        if (CollectionUtils.isEmpty(storeEntities)) {
            return;
        }
        Map<String, KmsDirectStoreEntity> storeEntityMap = Maps.newHashMap();
        Map<String, List<KmsDirectStoreEntity>> storeListMap = Maps.newHashMap();
        storeEntities.forEach(x -> {
            String keyForList = x.getDirectId() + x.getStoreCode();
            String key = x.getDirectId() + x.getStoreCode() + x.getStoreName();
            storeEntityMap.put(key, x);
            if (storeListMap.containsKey(keyForList)) {
                storeListMap.get(keyForList).add(x);
            } else {
                storeListMap.put(keyForList, Lists.newArrayList(x));
            }
        });
        data.forEach(x -> {
            if (StringUtils.isNotBlank(x.getKaCode()) && StringUtils.isNotBlank(x.getKaStoreCode()) && StringUtils.isNotBlank(x.getKaStoreName())) {
                String key = x.getKaCode() + x.getKaStoreCode() + x.getKaStoreName();
                String keyForList = x.getKaCode() + x.getKaStoreCode();
                if (!storeEntityMap.containsKey(key)) {
                    if (storeListMap.containsKey(keyForList)) {
                        x.appendErrorValidateMsg("【商超门店名称】填写错误、");
                    } else {
                        x.appendErrorValidateMsg("【商超门店】不在维护中、");
                    }
                }
            }
        });
    }


    /**
     * 检验客户产品
     *
     * @param data
     */
    private void validateKaGoods(List<KmsOrderFormImportVo> data) {
        Set<String> goodCodeSet = Sets.newHashSet();
        Set<String> kaCodeSet = Sets.newHashSet();
        data.forEach(x -> {
            if (StringUtils.isBlank(x.getKaGoodsCode())) {
                x.appendErrorValidateMsg("【客户产品编码和名称】未完整填写、");
            } else {
                goodCodeSet.add(x.getKaGoodsCode());
            }
            kaCodeSet.add(x.getKaCode());
        });
        if (CollectionUtils.isEmpty(goodCodeSet)) {
            return;
        }

        KmsTenantryDirectCustomerOrgReqVo customerOrgReqVo = new KmsTenantryDirectCustomerOrgReqVo();
        customerOrgReqVo.setDirectIdList(Lists.newArrayList(kaCodeSet));
        customerOrgReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        customerOrgReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        List<KmsTenantryDirectCustomerOrgEntity> customerOrgEntities = kmsTenantryDirectCustomerOrgMapper
                .selectListForExcel(customerOrgReqVo);
        if (CollectionUtils.isEmpty(customerOrgEntities)) {
            data.forEach(x -> {
                x.appendErrorValidateMsg("【客户产品不在维护中（未找到直营体系）】、");
            });
            return;
        }
        Map<String, List<KmsTenantryDirectCustomerOrgEntity>> customerOrgEntityMap = customerOrgEntities.stream()
                .collect(Collectors.groupingBy(KmsTenantryDirectCustomerOrgEntity::getDirectId));
        Map<String, List<KmsTenantryDirectCustomerOrgEntity>> cusOrgMap = customerOrgEntities.stream()
                .collect(Collectors.groupingBy(KmsTenantryDirectCustomerOrgEntity::getDirectId));
        data.forEach(x -> {
            if (!cusOrgMap.containsKey(x.getKaCode())) {
                x.appendErrorValidateMsg("商超【" + x.getKaName() + "】没有维护中的直营体系");
            } else {
                x.setBsDirectSystemId(cusOrgMap.get(x.getKaCode()).get(0).getId());
            }
        });
        KmsTenantryDirectProductReqVo productReqVo = new KmsTenantryDirectProductReqVo();
        productReqVo.setKaProductCodes(Lists.newArrayList(goodCodeSet));
        productReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        productReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<KmsTenantryDirectProductEntity> productEntities = kmsTenantryDirectProductMapper.selectListForExcel(productReqVo);
        if (CollectionUtils.isEmpty(productEntities)) {
            data.forEach(x -> {
                x.appendErrorValidateMsg("【客户产品不在维护中（未找到产品）】、");
            });
            return;
        }
        Map<String, KmsTenantryDirectProductEntity> productEntityMap = Maps.newHashMap();
        productEntities.forEach(x -> {
            productEntityMap.put(x.getBsDirectSystemId() + x.getKaProductCode(), x);
        });
        data.forEach(y -> {
            if (StringUtils.isNotBlank(y.getKaName()) && StringUtils.isNotBlank(y.getKaGoodsCode())
                    && StringUtils.isNotBlank(y.getKaGoodsName())) {
                if (customerOrgEntityMap.containsKey(y.getKaCode())) {
                    y.setBsDirectSystemId(customerOrgEntityMap.get(y.getKaCode()).get(0).getId());
                    if (!productEntityMap.containsKey(y.getBsDirectSystemId() + y.getKaGoodsCode())) {
                        y.appendErrorValidateMsg("【客户产品不在维护中（直营体系下没有该产品）】、");
                    }
                } else {
                    log.info(y.getKaName() + "下没有维护中直营体系（直营体系不对应)");
                    y.appendErrorValidateMsg("【没有维护中直营体系（无法检验产品）】、");
                }
            }
        });
    }

    /**
     * 非空 检验
     *
     * @param data
     */
    private void validateNotEmpty(List<KmsOrderFormImportVo> data) {
        data.forEach(x -> {
            if (StringUtils.isBlank(x.getKaOrderNumber())) {
                x.appendErrorValidateMsg("【商超验收单号】不能为空、");
            }
            if (StringUtils.isBlank(x.getVersionNumber())) {
                x.appendErrorValidateMsg("【订单版本号不能相同】");
            }
            if (StringUtils.isBlank(x.getOrderDate())) {
                x.appendErrorValidateMsg("【订货日期】不能为空");
            } else {
                if (x.getOrderDate().length() != DATE_SIZE) {
                    x.appendErrorValidateMsg("【订货日期】不正确");
                } else {
                    x.setOrderDate(DateUtil.format(DateUtil.getDateByFormat(
                            x.getOrderDate(), DateUtil.DEFAULT_MONTH_DAY_PATTERN),
                            DateUtil.DEFAULT_DAY_PATTERN));
                }
            }
            if (StringUtils.isBlank(x.getDeliveryDate())) {
                x.appendErrorValidateMsg("【送货日期】不能为空");
            } else {
                if (x.getDeliveryDate().length() != DATE_SIZE) {
                    x.appendErrorValidateMsg("【送货日期】不正确");
                } else {
                    x.setDeliveryDate(DateUtil.format(DateUtil.getDateByFormat(
                            x.getDeliveryDate(), DateUtil.DEFAULT_MONTH_DAY_PATTERN),
                            DateUtil.DEFAULT_DAY_PATTERN));
                }
            }
            if (StringUtils.isBlank(x.getUnitPrice())) {
                x.appendErrorValidateMsg("【订货单价(含税)】不能为空");
            }
            if (StringUtils.isBlank(x.getOrderAmount())) {
                x.appendErrorValidateMsg("【订单总额(含税)】不能为空");
            }
            if (StringUtils.isBlank(x.getCurUnit())) {
                x.appendErrorValidateMsg("【商超单位】不能为空");
            }
            if (StringUtils.isBlank(x.getCurUnitOrderQuantity())) {
                x.appendErrorValidateMsg("【订货数量】不能为空");
            }
        });
    }
}
