package com.biz.crm.excel.component.validator.mdm.customer;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.customer.MdmCustomerImportVo;
import com.biz.crm.excel.vo.mdm.terminal.MdmTerminalImportVo;
import com.biz.crm.mdm.customer.mapper.MdmCustomerMsgMapper;
import com.biz.crm.mdm.customer.model.MdmCustomerEntity;
import com.biz.crm.mdm.customerorg.entity.MdmCusOrgEntity;
import com.biz.crm.mdm.customerorg.mapper.MdmCusOrgMapper;
import com.biz.crm.mdm.org.entity.MdmOrgEntity;
import com.biz.crm.mdm.org.mapper.MdmOrgMapper;
import com.biz.crm.mdm.region.entity.MdmRegionEntity;
import com.biz.crm.mdm.region.mapper.MdmRegionMapper;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * @author: luoqi
 * @Date: 2020-12-11 10:50
 * @version: V1.0
 * @Description:
 */
@Slf4j
@Component("mdmCustomerImportValidator")
public class MdmCustomerImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmCustomerMsgMapper, MdmCustomerEntity, MdmCustomerImportVo> implements ExcelImportValidator<MdmCustomerImportVo> {

    @Resource
    private MdmCustomerMsgMapper mdmCustomerMsgMapper;

    @Resource
    private MdmOrgMapper mdmOrgMapper;

    @Resource
    private MdmCusOrgMapper mdmCusOrgMapper;

    @Resource
    private MdmRegionMapper mdmRegionMapper;

    private final String NUM_REGEX = "^[0-9]*$";


    @Override
    public void validate(List<MdmCustomerImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validCustomerCode(data);
        validCustomerName(data);
        validChannel(data);
        validCustomerType(data);
        validOrgCode(data);
        validCustomerOrgCode(data);
        validCustomerPhone(data);
        validProvince(data);
        validCity(data);
        validDistrict(data);
//        validUsername(data);
//        validLockState(data);
    }

    /**
     * 省校验
     * @param data
     */
    protected void validProvince(List<MdmCustomerImportVo> data){
        if(CollectionUtils.isEmpty(data)){
            return;
        }
        List<String> provinceNameList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getProvinceName()))
                .map(MdmCustomerImportVo::getProvinceName)
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(provinceNameList)){
            return;
        }
        Map<String, String> provinceRegionMap = Lists.partition(provinceNameList, 500)
                .parallelStream()
                .flatMap(x -> {
                    LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                            .in(MdmRegionEntity::getRegionName,x)
                            .eq(MdmRegionEntity::getRegionLevel, 1)
                            .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName);
                    return mdmRegionMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1));
        for (MdmCustomerImportVo mdmCustomerImportVo : data){
            if ( !StringUtils.isEmpty(mdmCustomerImportVo.getProvinceName())) {
                if (ObjectUtils.isEmpty(provinceRegionMap)||!provinceRegionMap.containsKey(mdmCustomerImportVo.getProvinceName())) {
                    mdmCustomerImportVo.appendErrorValidateMsg(mdmCustomerImportVo.getProvinceName()+"在行政区域不存在;");
                }else{
                    mdmCustomerImportVo.setProvinceCode(provinceRegionMap.get(mdmCustomerImportVo.getProvinceName()));
                }
            }
        }
    }

    /**
     * 市校验
     * @param data
     */
    protected void validCity(List<MdmCustomerImportVo> data){
        if(CollectionUtils.isEmpty(data)){
            return;
        }
        List<String> cityNameList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getCityName()))
                .map(MdmCustomerImportVo::getCityName)
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(cityNameList)){
            return;
        }
        Map<String, String> cityRegionMap = Lists.partition(cityNameList, 500)
                .parallelStream()
                .flatMap(x -> {
                    LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                            .in(MdmRegionEntity::getRegionName,x)
                            .eq(MdmRegionEntity::getRegionLevel, 2)
                            .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName);
                    return mdmRegionMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1));
        for (MdmCustomerImportVo mdmCustomerImportVo : data){
            if ( !StringUtils.isEmpty(mdmCustomerImportVo.getCityName())) {
                if (ObjectUtils.isEmpty(cityRegionMap)||!cityRegionMap.containsKey(mdmCustomerImportVo.getCityName())) {
                    mdmCustomerImportVo.appendErrorValidateMsg(mdmCustomerImportVo.getCityName()+"在行政区域不存在;");
                }else{
                    mdmCustomerImportVo.setCityCode(cityRegionMap.get(mdmCustomerImportVo.getCityName()));
                }
            }
        }
    }

    /**
     * 区校验
     * @param data
     */
    protected void validDistrict(List<MdmCustomerImportVo> data){
        if(CollectionUtils.isEmpty(data)){
            return;
        }
        List<String> districtNameList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getDistrictName()))
                .map(MdmCustomerImportVo::getDistrictName)
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(districtNameList)){
            return;
        }
        Map<String, String> districtRegionMap = Lists.partition(districtNameList, 500)
                .parallelStream()
                .flatMap(x -> {
                    LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                            .in(MdmRegionEntity::getRegionName,x)
                            .eq(MdmRegionEntity::getRegionLevel, 3)
                            .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName);
                    return mdmRegionMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1));
        for (MdmCustomerImportVo mdmCustomerImportVo : data){
            if ( !StringUtils.isEmpty(mdmCustomerImportVo.getDistrictName())) {
                if (ObjectUtils.isEmpty(districtRegionMap)||!districtRegionMap.containsKey(mdmCustomerImportVo.getDistrictName())) {
                    mdmCustomerImportVo.appendErrorValidateMsg(mdmCustomerImportVo.getDistrictName()+"在行政区域不存在;");
                }else{
                    mdmCustomerImportVo.setDistrictCode(districtRegionMap.get(mdmCustomerImportVo.getDistrictName()));
                }
            }
        }
    }

    /**
     * 校验锁定状态
     * @param data list
     */
//    protected void validLockState(List<MdmCustomerImportVo> data) {
//        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
//            String lockState = mdmCustomerImportVo.getLockState();
//            if(!StringUtils.isEmpty(lockState)){
//                if (!CrmEnableStatusEnum.contains(lockState)) {
//                    mdmCustomerImportVo.appendErrorValidateMsg("行号:"+mdmCustomerImportVo.getRowIndex()+",锁定状态"+lockState+"只能是003或009;");
//                }
//            }
//        }
//    }

//    /**
//     * 校验用户帐号
//     * @param data ist
//     */
//    protected void validUsername(List<MdmCustomerImportVo> data) {
//        long count = data.stream()
//                .filter(x -> !StringUtils.isEmpty(x.getUserName()))
//                .count();
//        if(count<1){
//            return;
//        }
//        LambdaQueryWrapper<MdmUserEntity> wrapper = Wrappers.<MdmUserEntity>lambdaQuery()
//                .eq(MdmUserEntity::getUserType, UserTypeEnum.CUSTOMER.getCode())
//                .select(MdmUserEntity::getUserName);
//        Set<String> dbUsernameSet = mdmUserMapper.selectList(wrapper).stream()
//                .filter(x -> x != null && !StringUtils.isEmpty(x.getUserName()))
//                .map(MdmUserEntity::getUserName)
//                .collect(Collectors.toSet());
//        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
//            String userName = mdmCustomerImportVo.getUserName();
//            if(!StringUtils.isEmpty(userName)){
//                if(CollectionUtils.isEmpty(dbUsernameSet)|| !dbUsernameSet.contains(userName)){
//                    mdmCustomerImportVo.appendErrorValidateMsg("行号:"+mdmCustomerImportVo.getRowIndex()+",用户帐号"+userName+"不存在;");
//                }
//            }
//        }
//    }

    /**
     * 校验客户组织编码
     *
     * @param data list
     */
    protected void validCustomerOrgCode(List<MdmCustomerImportVo> data) {
        long count = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getCustomerOrgCode()))
                .count();
        if (count < 1) {
            return;
        }
        LambdaQueryWrapper<MdmCusOrgEntity> wrapper = Wrappers.<MdmCusOrgEntity>lambdaQuery()
                .select(MdmCusOrgEntity::getCustomerOrgCode);
        Set<String> dbCustomerOrgSet = mdmCusOrgMapper.selectList(wrapper).stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getCustomerOrgCode()))
                .map(MdmCusOrgEntity::getCustomerOrgCode)
                .collect(Collectors.toSet());
        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
            String customerOrgCode = mdmCustomerImportVo.getCustomerOrgCode();
            if (!StringUtils.isEmpty(customerOrgCode)) {
                if (CollectionUtils.isEmpty(dbCustomerOrgSet) || !dbCustomerOrgSet.contains(customerOrgCode)) {
                    mdmCustomerImportVo.appendErrorValidateMsg("行号:" + mdmCustomerImportVo.getRowIndex() + ",客户组织编码" + customerOrgCode + "不存在;");
                }
            }
        }
    }

    /**
     * 校验电话格式
     *
     * @param data
     */
    protected void validCustomerPhone(List<MdmCustomerImportVo> data) {
        for (MdmCustomerImportVo item : data) {
            if (!StringUtils.isEmpty(item.getCustomerContact())) {
                if (!item.getCustomerContact().matches(NUM_REGEX)) {
                    item.appendErrorValidateMsg("客户联系方式仅能输入数字;");
                }
            }
            if (!StringUtils.isEmpty(item.getContactPhone())) {
                if (!item.getContactPhone().matches(NUM_REGEX)) {
                    item.appendErrorValidateMsg("联系人电话仅能输入数字;");
                }
            }
        }
    }

    /**
     * 校验组织编码
     *
     * @param data list
     */
    protected void validOrgCode(List<MdmCustomerImportVo> data) {
        if(CollectionUtils.isEmpty(data)){
            return;
        }
        List<String> orgCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getOrgCode()))
                .map(MdmCustomerImportVo::getOrgCode)
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(orgCodeList)){
            return;
        }
        Set<String> dbOrgCodeSet = Lists.partition(orgCodeList, 500).parallelStream()
                .flatMap(ol -> {
                    LambdaQueryWrapper<MdmOrgEntity> wrapper = Wrappers.<MdmOrgEntity>lambdaQuery()
                            .select(MdmOrgEntity::getOrgCode);
                    return mdmOrgMapper.selectList(wrapper).stream()
                            .filter(x -> x != null && !StringUtils.isEmpty(x.getOrgCode()))
                            .map(MdmOrgEntity::getOrgCode)
                            .collect(Collectors.toSet()).stream();
                })
                .collect(Collectors.toSet());

        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
            String orgCode = mdmCustomerImportVo.getOrgCode();
            if (!StringUtils.isEmpty(orgCode)) {
                if (orgCode.contains(",")) {
                    String[] rowOrgCodeList = orgCode.split(",");
                    for (String splitOrgCode : rowOrgCodeList) {
                        if (CollectionUtils.isEmpty(dbOrgCodeSet) || !dbOrgCodeSet.contains(splitOrgCode)) {
                            mdmCustomerImportVo.appendErrorValidateMsg("组织编码" + splitOrgCode + "不存在;");
                        }
                    }
                } else {
                    if (CollectionUtils.isEmpty(dbOrgCodeSet) || !dbOrgCodeSet.contains(orgCode)) {
                        mdmCustomerImportVo.appendErrorValidateMsg("组织编码" + orgCode + "不存在;");
                    }
                }
            }
        }
    }

    /**
     * 校验客户类型
     *
     * @param data list
     */
    protected void validCustomerType(List<MdmCustomerImportVo> data) {
        Map<String, String> customerTypeMap = DictUtil.dictMap("customer_type");
        Map<String, String> customerTypeNameCodeMap = new HashMap<>();
        for (Map.Entry<String, String> entry :
                customerTypeMap.entrySet()) {
            customerTypeNameCodeMap.put(entry.getValue(), entry.getKey());
        }
        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
            String customerTypeName = mdmCustomerImportVo.getCustomerTypeName();
            if (StringUtils.isEmpty(customerTypeName)) {
                mdmCustomerImportVo.appendErrorValidateMsg("客户类型未填写;");
            } else {
                if (!customerTypeNameCodeMap.containsKey(customerTypeName)) {
                    mdmCustomerImportVo.appendErrorValidateMsg("客户类型【" + customerTypeName + "】不存在;");
                } else {
                    mdmCustomerImportVo.setCustomerType(customerTypeNameCodeMap.get(customerTypeName));
                }
            }
        }
    }

    /**
     * 校验渠道
     *
     * @param data list
     */
    protected void validChannel(List<MdmCustomerImportVo> data) {
        Map<String, String> map = DictUtil.dictRevertMap("channel");
        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
            String channelName = mdmCustomerImportVo.getChannelName();
            if(!StringUtils.isEmpty(channelName)){
                if (map == null || map.size() < 1) {
                    mdmCustomerImportVo.appendErrorValidateMsg("数据字典channel;");
                    break;
                }
                String channelCode = map.get(channelName);
                if(StringUtils.isEmpty(channelCode)){
                    mdmCustomerImportVo.appendErrorValidateMsg("数据字典channel中:" + channelName + "不存在;");
                }else{
                    mdmCustomerImportVo.setChannel(channelCode);
                }
            }
        }
    }

    protected void validCustomerName(List<MdmCustomerImportVo> data) {
        for (MdmCustomerImportVo mdmCustomerImportVo : data) {
            if (StringUtils.isEmpty(mdmCustomerImportVo.getCustomerName())) {
                mdmCustomerImportVo.appendErrorValidateMsg("客户名称不能为空;");
            }
        }
    }

    /**
     * 校验客户编码
     *
     * @param data list
     */
    protected void validCustomerCode(List<MdmCustomerImportVo> data) {
        Set<String> collect = data.stream().filter(x -> !StringUtils.isEmpty(x.getCustomerCode())).map(MdmCustomerImportVo::getCustomerCode).collect(Collectors.toSet());
        if (collect.isEmpty()) {
            return;
        }
        LambdaQueryWrapper<MdmCustomerEntity> wrapper = Wrappers.<MdmCustomerEntity>lambdaQuery()
                .in(MdmCustomerEntity::getCustomerCode, collect)
                .select(MdmCustomerEntity::getCustomerCode);
        final Set<String> dbCustomerCodeSet = mdmCustomerMsgMapper.selectList(wrapper).stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getCustomerCode()))
                .map(MdmCustomerEntity::getCustomerCode)
                .collect(Collectors.toSet());
        Map<String, Integer> codeIndexMap = new HashMap<>(16);
        for (MdmCustomerImportVo item : data) {
            String customerCode = item.getCustomerCode();
            if (StringUtils.isEmpty(customerCode)) {
                break;
            }
            if (dbCustomerCodeSet.contains(customerCode)) {
                item.appendErrorValidateMsg("客户编码【" + customerCode + "】已存在;");
            } else if (codeIndexMap.containsKey(customerCode)) {
                item.appendErrorValidateMsg("客户编码【" + customerCode + "】与第" + codeIndexMap.get(customerCode) + "行重复;");
            } else {
                codeIndexMap.put(customerCode, item.getRowIndex());
            }
        }
    }
}
