package com.biz.crm.excel.component.validator.mdm.customerorg;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.customerorg.MdmCustomerOrgImportVo;
import com.biz.crm.mdm.customerorg.entity.MdmCusOrgEntity;
import com.biz.crm.mdm.customerorg.mapper.MdmCusOrgMapper;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author: luoqi
 * @Date: 2020-12-11 10:50
 * @version: V1.0
 * @Description:
 */
@Slf4j
@Component("mdmCustomerOrgImportValidator")
public class MdmCustomerOrgImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmCusOrgMapper, MdmCusOrgEntity, MdmCustomerOrgImportVo> implements ExcelImportValidator<MdmCustomerOrgImportVo> {

    @Resource
    private MdmCusOrgMapper mdmCusOrgMapper;

    @Override
    public void validate(List<MdmCustomerOrgImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validateRequired(data);
    }

    protected void validateRequired(List<MdmCustomerOrgImportVo> originList) {
        //客户组织类型名称：编码map
        final Map<String, String> cusOrgTypeValueCodeMap = DictUtil.dictRevertMap(DictConstant.CUS_ORG_TYPE);

        //客户组织层级名称：编码map
        final Map<String, String> cusOrgLevelValueCodeMap = DictUtil.dictRevertMap(DictConstant.CUS_ORG_LEVEL);

        //用于判断excel重复编码
        Map<String, Integer> originOrgCodeMap = new HashMap<>(16);
        //用于判断excel重复名称
        Map<String, Integer> originOrgNameMap = new HashMap<>(16);

        //数据库已有编码重复校验
        Set<String> orgCodeUniqueSet = new HashSet<>(16);
        Set<String> originOrgCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getCustomerOrgCode())).map(MdmCustomerOrgImportVo::getCustomerOrgCode).collect(Collectors.toSet());
        if (!originOrgCodeSet.isEmpty()) {
            List<List<String>> partition = Lists.partition(new ArrayList<>(originOrgCodeSet), 500);
            for (List<String> part :
                    partition) {
                QueryWrapper<MdmCusOrgEntity> orgCodeUniqueQuery = new QueryWrapper<>();
                orgCodeUniqueQuery.in("customer_org_code", part);
                List<MdmCusOrgEntity> partEntityList = mdmCusOrgMapper.selectList(orgCodeUniqueQuery);
                if (CollectionUtil.listNotEmptyNotSizeZero(partEntityList)) {
                    orgCodeUniqueSet.addAll(partEntityList.stream().map(MdmCusOrgEntity::getCustomerOrgCode).collect(Collectors.toSet()));
                }
            }
        }

        //数据库已有名称重复校验
        Set<String> orgNameUniqueSet = new HashSet<>(16);
        Set<String> originOrgNameSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getCustomerOrgName())).map(MdmCustomerOrgImportVo::getCustomerOrgName).collect(Collectors.toSet());
        if (!originOrgNameSet.isEmpty()) {
            List<List<String>> partition = Lists.partition(new ArrayList<>(originOrgNameSet), 500);
            for (List<String> part :
                    partition) {
                QueryWrapper<MdmCusOrgEntity> orgCodeUniqueQuery = new QueryWrapper<>();
                orgCodeUniqueQuery.in("customer_org_name", part);
                List<MdmCusOrgEntity> partEntityList = mdmCusOrgMapper.selectList(orgCodeUniqueQuery);
                if (CollectionUtil.listNotEmptyNotSizeZero(partEntityList)) {
                    orgNameUniqueSet.addAll(partEntityList.stream().map(MdmCusOrgEntity::getCustomerOrgCode).collect(Collectors.toSet()));
                }
            }
        }

        //数据库已有上级编码
        Set<String> parentCodeCheckSet = new HashSet<>(16);
        Set<String> originParentCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmCustomerOrgImportVo::getParentCode).collect(Collectors.toSet());
        if (!originParentCodeSet.isEmpty()) {
            List<List<String>> partition = Lists.partition(new ArrayList<>(originParentCodeSet), 500);
            for (List<String> part :
                    partition) {
                QueryWrapper<MdmCusOrgEntity> orgCodeUniqueQuery = new QueryWrapper<>();
                orgCodeUniqueQuery.in("customer_org_code", part);
                List<MdmCusOrgEntity> partEntityList = mdmCusOrgMapper.selectList(orgCodeUniqueQuery);
                if (CollectionUtil.listNotEmptyNotSizeZero(partEntityList)) {
                    parentCodeCheckSet.addAll(partEntityList.stream().map(MdmCusOrgEntity::getCustomerOrgCode).collect(Collectors.toSet()));
                }
            }
        }


        for (MdmCustomerOrgImportVo item :
                originList) {

            //校验编码
            if (StringUtils.isNotEmpty(item.getCustomerOrgCode())) {
                if (originOrgCodeMap.containsKey(item.getCustomerOrgCode())) {
                    item.appendErrorValidateMsg("客户组织编码与第" + originOrgCodeMap.get(item.getCustomerOrgCode()) + "行重复；");
                } else {
                    originOrgCodeMap.put(item.getCustomerOrgCode(), item.getRowIndex());
                }
                if (orgCodeUniqueSet.contains(item.getCustomerOrgCode())) {
                    item.appendErrorValidateMsg("组织编码已存在；");
                }
            }

            //校验名称
            if (StringUtils.isEmpty(item.getCustomerOrgName())) {
                item.appendErrorValidateMsg("缺失组织名称；");
            } else {
                if (originOrgNameMap.containsKey(item.getCustomerOrgName())) {
                    item.appendErrorValidateMsg("客户组织名称与第" + originOrgNameMap.get(item.getCustomerOrgName()) + "行重复；");
                } else {
                    originOrgNameMap.put(item.getCustomerOrgName(), item.getRowIndex());
                }
                if (orgNameUniqueSet.contains(item.getCustomerOrgName())) {
                    item.appendErrorValidateMsg("客户组织名称已存在；");
                }
            }

            //校验组织层级
            if (StringUtils.isEmpty(item.getCustomerOrgLevelName())) {
                item.appendErrorValidateMsg("缺失客户组织层级；");
            } else {
                if (!cusOrgLevelValueCodeMap.containsKey(item.getCustomerOrgLevelName())) {
                    item.appendErrorValidateMsg("客户组织层级无效；");
                } else {
                    item.setCustomerOrgLevel(cusOrgLevelValueCodeMap.get(item.getCustomerOrgLevelName()));
                }
            }

            //校验组织类型
            if (StringUtils.isEmpty(item.getCustomerOrgTypeName())) {
                item.appendErrorValidateMsg("缺失客户组织类型；");
            } else {
                if (!cusOrgTypeValueCodeMap.containsKey(item.getCustomerOrgTypeName())) {
                    item.appendErrorValidateMsg("客户组织类型无效；");
                } else {
                    item.setCustomerOrgType(cusOrgTypeValueCodeMap.get(item.getCustomerOrgTypeName()));
                }
            }
        }

        for (MdmCustomerOrgImportVo item :
                originList) {
            if (StringUtils.isEmpty(item.getParentCode())) {
                continue;
            }
            if (parentCodeCheckSet.contains(item.getParentCode())) {
                continue;
            }
            if (originOrgCodeSet.contains(item.getParentCode())) {
                continue;
            }
            item.appendErrorValidateMsg("上级编码无效；");
        }

        Set<String> errorCode = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getCustomerOrgCode()) && MdmCustomerOrgImportVo.ProcessTypeEnum.SUCCESS != item.getProcessType()).map(MdmCustomerOrgImportVo::getCustomerOrgCode).collect(Collectors.toSet());
        if (!errorCode.isEmpty()) {
            setErrorParent(originList, errorCode);
        }

    }

    private void setErrorParent(List<MdmCustomerOrgImportVo> originList, Set<String> errorCode) {
        if (errorCode.isEmpty()) {
            return;
        }
        final Set<String> codeOnTheTree = getCodeOnTheTree(originList);
        //TODO 校验不完善，如果某个节点校验未通过，则全部下级都不能通过
        originList.forEach(item -> {
            if (StringUtils.isNotEmpty(item.getParentCode()) && errorCode.contains(item.getParentCode())) {
                item.appendErrorValidateMsg("上级校验未通过；");
            }
            if (StringUtils.isNotEmpty(item.getCustomerOrgCode())) {
                if (!codeOnTheTree.contains(item.getCustomerOrgCode())) {
                    item.appendErrorValidateMsg("可能处在环形结构上或者上级处于环形结构上；");
                }
            }
        });
    }

    private Set<String> getCodeOnTheTree(List<MdmCustomerOrgImportVo> originList) {
        Set<String> orgCodeOnTheTree = new HashSet<>(16);
        List<MdmCustomerOrgImportVo> totalList = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getCustomerOrgCode())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(totalList)) {
            final Set<String> totalSet = totalList.stream().map(MdmCustomerOrgImportVo::getCustomerOrgCode).collect(Collectors.toSet());
            final Set<String> topCodeSet = totalList.stream().filter(item -> StringUtils.isEmpty(item.getParentCode()) || !totalSet.contains(item.getParentCode())).map(MdmCustomerOrgImportVo::getCustomerOrgCode).collect(Collectors.toSet());
            if (!topCodeSet.isEmpty()) {
                Set<String> curLevelSet = new HashSet<>(16);
                curLevelSet.addAll(topCodeSet);
                orgCodeOnTheTree.addAll(curLevelSet);
                while (!curLevelSet.isEmpty()) {
                    Set<String> collect = totalList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode()) && curLevelSet.contains(item.getParentCode()) && !item.getParentCode().equals(item.getCustomerOrgCode())).map(MdmCustomerOrgImportVo::getCustomerOrgCode).collect(Collectors.toSet());
                    curLevelSet.clear();
                    if (!collect.isEmpty()) {
                        orgCodeOnTheTree.addAll(collect);
                        curLevelSet.addAll(collect);
                    }
                }
            }
        }
        return orgCodeOnTheTree;
    }

}
