package com.biz.crm.excel.component.validator.mdm.function;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.constant.mdm.SqlConfigConstant;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.MdmPermissionObjEnum;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.function.MdmDataPermissionImportVo;
import com.biz.crm.mdm.function.entity.MdmFunctionEntity;
import com.biz.crm.mdm.function.entity.MdmFunctionSubEntity;
import com.biz.crm.mdm.function.mapper.MdmFunctionMapper;
import com.biz.crm.mdm.function.mapper.MdmFunctionSubMapper;
import com.biz.crm.mdm.role.entity.MdmRoleEntity;
import com.biz.crm.mdm.role.mapper.MdmRoleMapper;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-04-13 10:08
 **/
@Slf4j
@Component("mdmDataPermissionValidator")
public class MdmDataPermissionValidator implements ExcelImportValidator<MdmDataPermissionImportVo> {

    @Resource
    private MdmRoleMapper mdmRoleMapper;

    @Resource
    private MdmFunctionMapper mdmFunctionMapper;

    @Resource
    private MdmFunctionSubMapper mdmFunctionSubMapper;

    @Override
    public void validate(List<MdmDataPermissionImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validEmptyField(data);
        validRoleCode(data);
        validFunctionCode(data);
        validConfigCode(data);
        validPermissionObjName(data);
        validPermissionSearchTypeName(data);
    }

    protected void validPermissionSearchTypeName(List<MdmDataPermissionImportVo> data) {
        String orgDictTypeCode = "permission_obj_org";
        String positionDictTypeCode = "permission_obj_position";
        Map<String, String> orgMap = DictUtil.dictRevertMap(orgDictTypeCode);
        Map<String, String> positionMap = DictUtil.dictRevertMap(positionDictTypeCode);
        for (MdmDataPermissionImportVo importVo : data) {
            String permissionSearchTypeName = importVo.getPermissionSearchTypeName();
            if(StringUtils.isEmpty(permissionSearchTypeName)){
                importVo.appendErrorValidateMsg("查询方式不能为空;");
            }else if(!StringUtils.isEmpty(importVo.getPermissionObjCode())){
                String permissionObjCode = importVo.getPermissionObjCode();
                if (MdmPermissionObjEnum.POSITION.getCode().equals(permissionObjCode)) {
                    String permissionSearchType = positionMap.get(permissionSearchTypeName);
                    if (StringUtils.isEmpty(permissionObjCode)) {
                        importVo.appendErrorValidateMsg(permissionSearchType+"在数据字典"+positionDictTypeCode+"不存在;");
                    }else{
                        importVo.setPermissionSearchType(permissionSearchType);
                    }
                }
                if (MdmPermissionObjEnum.ORG.getCode().equals(permissionObjCode)) {
                    String permissionSearchType = orgMap.get(permissionSearchTypeName);
                    if (StringUtils.isEmpty(permissionObjCode)) {
                        importVo.appendErrorValidateMsg(permissionSearchType+"在数据字典"+orgDictTypeCode+"不存在;");
                    }else{
                        importVo.setPermissionSearchType(permissionSearchType);
                    }
                }
            }
        }
    }

    protected void validPermissionObjName(List<MdmDataPermissionImportVo> data) {
        String dictTypeCode = "permission_obj";
        Map<String, String> map = DictUtil.dictRevertMap(dictTypeCode);
        for (MdmDataPermissionImportVo importVo : data) {
            String permissionObjName = importVo.getPermissionObjName();
            if(StringUtils.isEmpty(permissionObjName)){
                importVo.appendErrorValidateMsg("查询对象不能为空;");
            }else{
                String permissionObjCode = map.get(permissionObjName);
                if(StringUtils.isEmpty(permissionObjCode)){
                    importVo.appendErrorValidateMsg(permissionObjName+"在数据字典"+dictTypeCode+"不存在;");
                }else{
                    importVo.setPermissionObjCode(permissionObjCode);
                }
            }
        }
    }

    protected void validConfigCode(List<MdmDataPermissionImportVo> data) {
        List<String> excelConfigList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getListConfigCode()))
                .map(MdmDataPermissionImportVo::getListConfigCode)
                .distinct()
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(excelConfigList)){
            return;
        }
        Set<String> dbFunctionSet = Lists.partition(excelConfigList, SqlConfigConstant.IN_SIZE)
                .parallelStream()
                .filter(x -> !CollectionUtils.isEmpty(x))
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmFunctionSubEntity> wrapper = Wrappers.<MdmFunctionSubEntity>lambdaQuery()
                            .in(MdmFunctionSubEntity::getFunctionCode, list)
                            .select(MdmFunctionSubEntity::getFunctionCode);
                    return mdmFunctionSubMapper.selectList(wrapper).stream();
                })
                .map(MdmFunctionSubEntity::getFunctionCode)
                .collect(Collectors.toSet());
        for (MdmDataPermissionImportVo importVo : data) {
            String listConfigCode = importVo.getListConfigCode();
            if(!StringUtils.isEmpty(listConfigCode)&& !dbFunctionSet.contains(listConfigCode)){
                importVo.appendErrorValidateMsg("列表编码"+listConfigCode+"不存在;");
            }
        }
    }

    protected void validFunctionCode(List<MdmDataPermissionImportVo> data) {
        List<String> excelFunctionCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmDataPermissionImportVo::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(excelFunctionCodeList)){
            return;
        }
        Map<String, String> dbFunctionMap = Lists.partition(excelFunctionCodeList, SqlConfigConstant.IN_SIZE).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmFunctionEntity> wrapper = Wrappers.<MdmFunctionEntity>lambdaQuery()
                            .in(MdmFunctionEntity::getFunctionCode, list)
                            .select(MdmFunctionEntity::getFunctionCode, MdmFunctionEntity::getEnableStatus);
                    return mdmFunctionMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmFunctionEntity::getFunctionCode, CrmBaseEntity::getEnableStatus, (x1, x2) -> x1));
        for (MdmDataPermissionImportVo importVo : data) {
            String functionCode = importVo.getFunctionCode();
            if(!StringUtils.isEmpty(functionCode)){
                if (!dbFunctionMap.containsKey(functionCode)) {
                    importVo.appendErrorValidateMsg("菜单编码"+functionCode+"不存在;");
                } else if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbFunctionMap.get(functionCode))) {
                    importVo.appendErrorValidateMsg("菜单编码"+functionCode+"未启用;");
                }
            }
        }
    }

    protected void validRoleCode(List<MdmDataPermissionImportVo> data) {
        List<String> excelRoleCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getRoleCode()))
                .map(MdmDataPermissionImportVo::getRoleCode)
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(excelRoleCodeList)) {
            return;
        }
        Map<String, String> dbRoleMap = Lists.partition(excelRoleCodeList, SqlConfigConstant.IN_SIZE).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmRoleEntity> wrapper = Wrappers.<MdmRoleEntity>lambdaQuery()
                            .in(MdmRoleEntity::getRoleCode, list)
                            .select(MdmRoleEntity::getRoleCode, MdmRoleEntity::getEnableStatus);
                    return mdmRoleMapper.selectList(wrapper).stream();
                }).collect(Collectors.toMap(MdmRoleEntity::getRoleCode, CrmBaseEntity::getEnableStatus, (x1, x2) -> x1));
        for (MdmDataPermissionImportVo importVo : data) {
            String roleCode = importVo.getRoleCode();
            if(!StringUtils.isEmpty(roleCode)){
                if (!dbRoleMap.containsKey(roleCode)) {
                    importVo.appendErrorValidateMsg("角色编码"+roleCode+"不存在;");
                } else if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbRoleMap.get(roleCode))) {
                    importVo.appendErrorValidateMsg("角色编码"+roleCode+"未启用;");
                }
            }
        }
    }

    protected void validEmptyField(List<MdmDataPermissionImportVo> data) {
        for (MdmDataPermissionImportVo importVo : data) {
            if (StringUtils.isEmpty(importVo.getRoleCode())) {
                importVo.appendErrorValidateMsg("角色编码不能为空;");
            }
            if (StringUtils.isEmpty(importVo.getFunctionCode())) {
                importVo.appendErrorValidateMsg("菜单编码不能为空;");
            }
            if (StringUtils.isEmpty(importVo.getListConfigCode())) {
                importVo.appendErrorValidateMsg("列表编码不能为空;");
            }
            if (StringUtils.isEmpty(importVo.getPermissionObjName())) {
                importVo.appendErrorValidateMsg("查询对象不能为空");
            }
            if (StringUtils.isEmpty(importVo.getPermissionSearchTypeName())) {
                importVo.appendErrorValidateMsg("查询方式不能为空;");
            }
        }
    }
}
