package com.biz.crm.excel.component.validator.mdm.function;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.constant.mdm.SqlConfigConstant;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.function.MdmRoleFunctionButtonImportVo;
import com.biz.crm.mdm.function.entity.MdmButtonEntity;
import com.biz.crm.mdm.function.entity.MdmFunctionEntity;
import com.biz.crm.mdm.function.entity.MdmFunctionRoleEntity;
import com.biz.crm.mdm.function.entity.MdmFunctionSubEntity;
import com.biz.crm.mdm.function.mapper.*;
import com.biz.crm.mdm.role.entity.MdmRoleEntity;
import com.biz.crm.mdm.role.mapper.MdmRoleMapper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author zxw
 * @date 2021-04-13 10:08
 **/
@Slf4j
@Component("mdmRoleFunctionButtonValidator")
public class MdmRoleFunctionButtonValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmFunctionRoleMapper, MdmFunctionRoleEntity, MdmRoleFunctionButtonImportVo> implements ExcelImportValidator<MdmRoleFunctionButtonImportVo> {

    @Resource
    private MdmRoleMapper mdmRoleMapper;

    @Resource
    private MdmFunctionMapper mdmFunctionMapper;

    @Resource
    private MdmFunctionRoleMapper mdmFunctionRoleMapper;

    @Resource
    private MdmFunctionSubMapper mdmFunctionSubMapper;

    @Resource
    private MdmButtonMapper mdmButtonMapper;


    @Override
    public void validate(List<MdmRoleFunctionButtonImportVo> data, DefaultImportContext context) {
        validRole(data);
        validFunction(data);
        validListConfig(data);
        validButton(data);
    }

    /**
     * 校验按钮
     * @param data data
     */
    protected void validButton(List<MdmRoleFunctionButtonImportVo> data) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        List<String> excelButtonCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getButtonCode()))
                .map(MdmRoleFunctionButtonImportVo::getButtonCode)
                .distinct()
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(excelButtonCodeList)){
            return;
        }

        Map<String,String> dbListButtonCodeMap = Lists.partition(excelButtonCodeList, SqlConfigConstant.IN_SIZE).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmButtonEntity> wrapper = Wrappers.<MdmButtonEntity>lambdaQuery()
                            .in(MdmButtonEntity::getButtonCode, excelButtonCodeList)
                            .select(MdmButtonEntity::getButtonCode,MdmButtonEntity::getEnableStatus);
                    return mdmButtonMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmButtonEntity::getButtonCode, CrmBaseEntity::getEnableStatus,(x1, x2)->x1));
        for (MdmRoleFunctionButtonImportVo importVo : data) {
            String buttonCode = importVo.getButtonCode();
            if(!StringUtils.isEmpty(buttonCode)){
                if(StringUtils.isEmpty(importVo.getListConfigCode())){
                    importVo.appendErrorValidateMsg("列表编码不能为空;");
                    continue;
                }
                if (!dbListButtonCodeMap.containsKey(buttonCode)) {
                    importVo.appendErrorValidateMsg("按钮编码:"+buttonCode+"不存在;");
                } else if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbListButtonCodeMap.get(buttonCode))) {
                    importVo.appendErrorValidateMsg("按钮编码:"+buttonCode+"未启用");
                }
            }
        }
    }

    /**
     * 校验功能列表
     * @param data data
     */
    protected void validListConfig(List<MdmRoleFunctionButtonImportVo> data) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        List<String> excelListConfigCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getListConfigCode()))
                .map(MdmRoleFunctionButtonImportVo::getListConfigCode)
                .distinct()
                .collect(Collectors.toList());
        if(CollectionUtils.isEmpty(excelListConfigCodeList)){
            return;
        }
        Set<String> dbListConfigCodeSet = Lists.partition(excelListConfigCodeList, SqlConfigConstant.IN_SIZE).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmFunctionSubEntity> wrapper = Wrappers.<MdmFunctionSubEntity>lambdaQuery()
                            .in(MdmFunctionSubEntity::getFunctionCode, excelListConfigCodeList)
                            .select(MdmFunctionSubEntity::getFunctionCode);
                    return mdmFunctionSubMapper.selectList(wrapper).stream();
                })
                .map(MdmFunctionSubEntity::getFunctionCode)
                .collect(Collectors.toSet());
        for (MdmRoleFunctionButtonImportVo importVo : data) {
            String listConfigCode = importVo.getListConfigCode();
            if(!StringUtils.isEmpty(listConfigCode)&&!dbListConfigCodeSet.contains(listConfigCode)){
                importVo.appendErrorValidateMsg("列表编码:"+listConfigCode+"不存在;");
            }
        }
    }

    /**
     * 校验菜单编码
     *
     * @param data data
     */
    protected void validFunction(List<MdmRoleFunctionButtonImportVo> data) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        Set<String> repeatFunctionRoleSet = new HashSet<>();
        for (MdmRoleFunctionButtonImportVo importVo : data) {
            if (StringUtils.isEmpty(importVo.getFunctionCode())) {
                importVo.appendErrorValidateMsg("菜单编码不能为空;");
            } else if (StringUtils.isEmpty(importVo.getListConfigCode()) && !StringUtils.isEmpty(importVo.getRoleCode())) {
                //角色编码不为空，列表编码为空
                if (!repeatFunctionRoleSet.add(importVo.getRoleCode() + ":" + importVo.getFunctionCode())) {
                    importVo.appendErrorValidateMsg("菜单编码:" + importVo.getFunctionCode() + ",角色编码:" + importVo.getRoleCode() + "重复");
                }
            }
        }

        List<String> excelFunctionCodeList = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmRoleFunctionButtonImportVo::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(excelFunctionCodeList)) {
            return;
        }
        Map<String, String> dbFunctionMap = Lists.partition(excelFunctionCodeList, SqlConfigConstant.IN_SIZE).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmFunctionEntity> wrapper = Wrappers.<MdmFunctionEntity>lambdaQuery()
                            .in(MdmFunctionEntity::getFunctionCode, list)
                            .select(MdmFunctionEntity::getFunctionCode, MdmFunctionEntity::getEnableStatus);
                    return mdmFunctionMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmFunctionEntity::getFunctionCode, CrmBaseEntity::getEnableStatus, (x1, x2) -> x1));
        for (MdmRoleFunctionButtonImportVo importVo : data) {
            String functionCode = importVo.getFunctionCode();
            if (!StringUtils.isEmpty(functionCode)) {
                if (!dbFunctionMap.containsKey(functionCode)) {
                    importVo.appendErrorValidateMsg("菜单编码:" + functionCode + "不存在;");
                } else if (!CrmEnableStatusEnum.ENABLE.getCode().equalsIgnoreCase(dbFunctionMap.get(functionCode))) {
                    importVo.appendErrorValidateMsg("菜单编码:" + functionCode + "未启用;");
                }
            }
        }

        //校验已经导入的关联信息
        Set<String> mayDbRoleCodeSet = new HashSet<>();
        Set<String> mayDbFunctionCodeSet = new HashSet<>();
        data.stream()
                .filter(x -> StringUtils.isEmpty(x.getListConfigCode()))
                .filter(x -> !StringUtils.isEmpty(x.getRoleCode()) && !StringUtils.isEmpty(x.getFunctionCode()))
                .forEach(x -> {
                    mayDbRoleCodeSet.add(x.getRoleCode());
                    mayDbFunctionCodeSet.add(x.getFunctionCode());
                });
        if (!CollectionUtils.isEmpty(mayDbRoleCodeSet) && !CollectionUtils.isEmpty(mayDbFunctionCodeSet)) {
            LambdaQueryWrapper<MdmFunctionRoleEntity> wrapper = Wrappers.<MdmFunctionRoleEntity>lambdaQuery()
                    .in(MdmFunctionRoleEntity::getRoleCode, mayDbRoleCodeSet)
                    .in(MdmFunctionRoleEntity::getFunctionCode, mayDbFunctionCodeSet)
                    .select(MdmFunctionRoleEntity::getRoleCode, MdmFunctionRoleEntity::getFunctionCode);

            List<MdmFunctionRoleEntity> mdmFunctionRoleEntities = mdmFunctionRoleMapper.selectList(wrapper);
            if (!CollectionUtils.isEmpty(mdmFunctionRoleEntities)) {
                for (MdmRoleFunctionButtonImportVo importVo : data) {
                    String roleCode = importVo.getRoleCode();
                    String functionCode = importVo.getFunctionCode();
                    String listConfigCode = importVo.getListConfigCode();
                    if (!StringUtils.isEmpty(roleCode) && !StringUtils.isEmpty(functionCode) && StringUtils.isEmpty(listConfigCode)) {
                        for (MdmFunctionRoleEntity mdmFunctionRoleEntity : mdmFunctionRoleEntities) {
                            if (roleCode.equals(mdmFunctionRoleEntity.getRoleCode()) && functionCode.equals(mdmFunctionRoleEntity.getFunctionCode())) {
                                importVo.appendErrorValidateMsg("角色编码:" + roleCode + ",菜单编码:" + functionCode + "已存在");
                            }
                        }
                    }
                }
            }
        }

    }

    /**
     * 校验角色编码
     *
     * @param data data
     */
    protected void validRole(List<MdmRoleFunctionButtonImportVo> data) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }

        for (MdmRoleFunctionButtonImportVo importVo : data) {
            if (StringUtils.isEmpty(importVo.getRoleCode())) {
                importVo.appendErrorValidateMsg("角色编码不能为空;");
            }
        }
        List<String> excelRoleCodeSet = data.stream()
                .filter(x -> !StringUtils.isEmpty(x.getRoleCode()))
                .map(MdmRoleFunctionButtonImportVo::getRoleCode)
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(excelRoleCodeSet)) {
            return;
        }

        Map<String, String> dbRoleMap = Lists.partition(excelRoleCodeSet, 500).parallelStream()
                .flatMap(list -> {
                    LambdaQueryWrapper<MdmRoleEntity> wrapper = Wrappers.<MdmRoleEntity>lambdaQuery()
                            .in(MdmRoleEntity::getRoleCode, list)
                            .select(MdmRoleEntity::getRoleCode, MdmRoleEntity::getEnableStatus);
                    return mdmRoleMapper.selectList(wrapper).stream();
                })
                .collect(Collectors.toMap(MdmRoleEntity::getRoleCode, CrmBaseEntity::getEnableStatus, (x1, x2) -> x1));
        for (MdmRoleFunctionButtonImportVo importVo : data) {
            String roleCode = importVo.getRoleCode();
            if (!StringUtils.isEmpty(roleCode)) {
                if (!dbRoleMap.containsKey(roleCode)) {
                    importVo.appendErrorValidateMsg("角色编码" + roleCode + "不存在;");
                } else if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbRoleMap.get(roleCode))) {
                    importVo.appendErrorValidateMsg("角色" + roleCode + "未启用;");
                }
            }
        }
    }
}
