package com.biz.crm.excel.component.validator.mdm.material;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.material.MdmMaterialImportVo;
import com.biz.crm.mdm.material.entity.MdmMaterialEntity;
import com.biz.crm.mdm.material.mapper.MdmMaterialMapper;
import com.biz.crm.mdm.product.entity.MdmProductLevelEntity;
import com.biz.crm.mdm.product.mapper.MdmProductLevelMapper;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @Author hej
 * @Created Date 2020-12-31 15:09
 **/
@Component("mdmMaterialImportValidator")
public class MdmMaterialImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmMaterialMapper, MdmMaterialEntity, MdmMaterialImportVo> implements ExcelImportValidator<MdmMaterialImportVo> {

    @Resource
    private MdmMaterialMapper mdmMaterialMapper;

    @Resource
    private MdmProductLevelMapper mdmProductLevelMapper;

    private Pattern pattern = Pattern.compile("^(\\+)?\\d+(\\.\\d+)?$");

    @Override
    public void validate(List<MdmMaterialImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        checkMaterialIsNull(data);
        checkMaterialCode(data);
        checkProductLevelCode(data);
        checkMaterialType(data);
        checkMaterialBaseUnit(data);
        checkMaterialStandardUnit(data);
        checkMaterialSaleUnit(data);
        checkNumber(data);
        checkSaleCompany(data);


    }

    /**
     * 验空
     *
     * @param data
     */
    protected void checkMaterialIsNull(List<MdmMaterialImportVo> data) {
        if (CollectionUtil.listNotEmptyNotSizeZero(data)) {
            data.forEach(m -> {
                //物料名称
                if (StringUtils.isEmpty(m.getMaterialName())) {
                    m.appendErrorValidateMsg("物料名称不能为空；");
                }
                //产品层级名称
//                if(StringUtils.isEmpty(m.getProductLevelName())){
//                    m.appendErrorValidateMsg("产品层级名称不能为空；");
//                }
                //产品层级编码
                if (StringUtils.isEmpty(m.getProductLevelCode())) {
                    m.appendErrorValidateMsg("产品层级编码不能为空；");
                }
                //物料类型
                if (StringUtils.isEmpty(m.getMaterialTypeName())) {
                    m.appendErrorValidateMsg("物料类型不能为空；");
                }
                //基本单位
                if (StringUtils.isEmpty(m.getBaseUnitName())) {
                    m.appendErrorValidateMsg("基本单位不能为空；");
                }
                //单位换算系数
                if (StringUtils.isEmpty(m.getUnitConversion())) {
                    m.appendErrorValidateMsg("单位换算系数不能为空；");
                }
            });
        }
    }

    /**
     * 校验物料编码
     *
     * @param data
     */
    protected void checkMaterialCode(List<MdmMaterialImportVo> data) {

        Set<String> collect = data.stream().filter(item -> StringUtils.isNotEmpty(item.getMaterialCode())).map(MdmMaterialImportVo::getMaterialCode).collect(Collectors.toSet());
        if (collect.isEmpty()) {
            return;
        }
        Set<String> materialSet = new HashSet<>();
        //获取所有已存在物料编码
        LambdaQueryWrapper<MdmMaterialEntity> wrapper = Wrappers.<MdmMaterialEntity>lambdaQuery().select(MdmMaterialEntity::getMaterialCode);
        wrapper.in(MdmMaterialEntity::getMaterialCode, collect);
        final Set<String> materialCodeSet = mdmMaterialMapper.selectList(wrapper).stream()
                .map(MdmMaterialEntity::getMaterialCode)
                .collect(Collectors.toSet());

        Map<String, Integer> codeIndexMap = new HashMap<>(16);
        for (MdmMaterialImportVo item : data) {
            if (StringUtils.isEmpty(item.getMaterialCode())) {
                continue;
            }
            if (materialCodeSet.contains(item.getMaterialCode())) {
                item.appendErrorValidateMsg("物料编码已存在；");
            } else if (codeIndexMap.containsKey(item.getMaterialCode())) {
                item.appendErrorValidateMsg("物料编码与第" + item.getMaterialCode() + "行重复；");
            } else {
                codeIndexMap.put(item.getMaterialCode(), item.getRowIndex());
            }
        }
    }

    /**
     * 校验产品层级编码
     *
     * @param data
     */
    protected void checkProductLevelCode(List<MdmMaterialImportVo> data) {
        Map<String, String> productMap = new HashMap<>();
        if (CollectionUtil.listNotEmptyNotSizeZero(data)) {
            //获取所有产品层级编码
            LambdaQueryWrapper<MdmProductLevelEntity> wrapper = Wrappers.<MdmProductLevelEntity>lambdaQuery().select(MdmProductLevelEntity::getProductLevelCode);
            List<MdmProductLevelEntity> mdmProductLevelEntities = mdmProductLevelMapper.selectList(wrapper);
            if (CollectionUtil.listNotEmptyNotSizeZero(mdmProductLevelEntities)) {
                mdmProductLevelEntities.stream()
                        .filter(m -> StringUtils.isNotEmpty(m.getProductLevelCode()))
                        .forEach(m -> productMap.put(m.getProductLevelCode(), null));
            }
            for (MdmMaterialImportVo datum : data) {
                if (!productMap.containsKey(datum.getProductLevelCode())) {
                    datum.appendErrorValidateMsg("当前层级编码" + datum.getProductLevelCode() + "不存在；");
//                    continue;
                }
//                String productName = productMap.get(datum.getProductLevelCode());
//                if(!productName.equals(datum.getProductLevelName())){
//                    datum.appendErrorValidateMsg("当前层级编码"+datum.getProductLevelCode()+"所属的层级名称应当是:"+productName+"；");
//                }
            }
        }
    }

    /**
     * 物料类型字典
     *
     * @param vos
     */
    protected void checkMaterialType(List<MdmMaterialImportVo> vos) {
        Map<String, String> map = DictUtil.dictMap("material_type");
        Map<String, String> channelMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
                map.entrySet()) {
            if (!channelMap.containsKey(entry.getValue())) {
                channelMap.put(entry.getValue(), entry.getKey());
            }
        }
        for (MdmMaterialImportVo materialImportVo : vos) {
            String materialTypeName = materialImportVo.getMaterialTypeName();
            if (!org.springframework.util.StringUtils.isEmpty(materialTypeName)) {
                if (!channelMap.containsKey(materialTypeName)) {
                    materialImportVo.appendErrorValidateMsg("行号:" + materialImportVo.getRowIndex() + "物料类型:" + materialTypeName + "不存在;");
                } else {
                    materialImportVo.setMaterialType(channelMap.get(materialTypeName));
                }
            }
        }
    }

    /**
     * 销售公司字典
     *
     * @param vos
     */
    protected void checkSaleCompany(List<MdmMaterialImportVo> vos) {
        Map<String, String> map = DictUtil.dictMap("sale_company");
        Map<String, String> channelMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
                map.entrySet()) {
            if (!channelMap.containsKey(entry.getValue())) {
                channelMap.put(entry.getValue(), entry.getKey());
            }
        }
        for (MdmMaterialImportVo materialImportVo : vos) {
            String saleCompanyName = materialImportVo.getSaleCompanyName();
            if (!org.springframework.util.StringUtils.isEmpty(saleCompanyName)) {
                if (!channelMap.containsKey(saleCompanyName)) {
                    materialImportVo.appendErrorValidateMsg("行号:" + materialImportVo.getRowIndex() + "销售公司:" + saleCompanyName + "不存在;");
                } else {
                    materialImportVo.setSaleCompany(channelMap.get(saleCompanyName));
                }
            }
        }
    }

    /**
     * 基本单位字典
     *
     * @param vos
     */
    protected void checkMaterialBaseUnit(List<MdmMaterialImportVo> vos) {
        Map<String, String> map = DictUtil.dictMap("material_base_unit");
        Map<String, String> channelMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
                map.entrySet()) {
            if (!channelMap.containsKey(entry.getValue())) {
                channelMap.put(entry.getValue(), entry.getKey());
            }
        }
        for (MdmMaterialImportVo materialImportVo : vos) {
            String baseUnitName = materialImportVo.getBaseUnitName();
            if (!org.springframework.util.StringUtils.isEmpty(baseUnitName)) {
                if (!channelMap.containsKey(baseUnitName)) {
                    materialImportVo.appendErrorValidateMsg("行号:" + materialImportVo.getRowIndex() + "物料基本单位:" + baseUnitName + "不存在;");
                } else {
                    materialImportVo.setBaseUnit(channelMap.get(baseUnitName));
                }
            }
        }
    }

    /**
     * 物料销售单位
     *
     * @param vos
     */
    protected void checkMaterialSaleUnit(List<MdmMaterialImportVo> vos) {
        Map<String, String> map = DictUtil.dictMap("material_sale_unit");
        Map<String, String> channelMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
                map.entrySet()) {
            if (!channelMap.containsKey(entry.getValue())) {
                channelMap.put(entry.getValue(), entry.getKey());
            }
        }
        for (MdmMaterialImportVo materialImportVo : vos) {
            String saleUnitName = materialImportVo.getSaleUnitName();
            if (!org.springframework.util.StringUtils.isEmpty(saleUnitName)) {
                if (!channelMap.containsKey(saleUnitName)) {
                    materialImportVo.appendErrorValidateMsg("行号:" + materialImportVo.getRowIndex() + "物料销售单位:" + saleUnitName + "不存在;");
                } else {
                    materialImportVo.setSaleUnit(channelMap.get(saleUnitName));
                }
            }
        }
    }

    /**
     * 价格校验
     *
     * @param vos
     */
    protected void checkNumber(List<MdmMaterialImportVo> vos) {
        if (CollectionUtil.listNotEmptyNotSizeZero(vos)) {
            vos.stream().filter(m -> StringUtils.isNotEmpty(m.getCostPrice()) && StringUtils.isNotEmpty(m.getUnitConversion()))
                    .forEach(m -> {
                        //判断是否为正整数
                        if (!pattern.matcher(m.getCostPrice()).matches()) {
                            m.appendErrorValidateMsg("成本价(元) 数字格式错误");
                        }
                        if (!pattern.matcher(m.getUnitConversion()).matches()) {
                            m.appendErrorValidateMsg("单位换算系数 数字格式错误");
                        }
                    });
        }
    }


    /**
     * 标准单位字典
     *
     * @param vos
     */
    protected void checkMaterialStandardUnit(List<MdmMaterialImportVo> vos) {
        Map<String, String> map = DictUtil.dictMap("material_standard_unit");
        Map<String, String> channelMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
            map.entrySet()) {
            if (!channelMap.containsKey(entry.getValue())) {
                channelMap.put(entry.getValue(), entry.getKey());
            }
        }
        for (MdmMaterialImportVo materialImportVo : vos) {
            String standardUnitName = materialImportVo.getStandardUnitName();
            if (!org.springframework.util.StringUtils.isEmpty(standardUnitName)) {
                if (!channelMap.containsKey(standardUnitName)) {
                    materialImportVo.appendErrorValidateMsg("行号:" + materialImportVo.getRowIndex() + "物料标准单位:" + standardUnitName + "不存在;");
                } else {
                    materialImportVo.setStandardUnit(channelMap.get(standardUnitName));
                }
            }
        }
    }

}
