package com.biz.crm.excel.component.validator.mdm.org;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.org.MdmOrgImportVo;
import com.biz.crm.mdm.org.entity.MdmOrgEntity;
import com.biz.crm.mdm.org.mapper.MdmOrgMapper;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;


/**
 * @author: luoqi
 * @Date: 2020-12-11 10:50
 * @version: V1.0
 * @Description:
 */
@Slf4j
@Component("mdmOrgImportValidator")
public class MdmOrgImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmOrgMapper, MdmOrgEntity, MdmOrgImportVo> implements ExcelImportValidator<MdmOrgImportVo> {

    @Autowired
    private MdmOrgMapper mdmOrgMapper;

    @Override
    public void validate(List<MdmOrgImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validateRequired(data);
    }

    protected void validateRequired(List<MdmOrgImportVo> originList) {
        final Map<String, String> orgTypeMap = DictUtil.dictMap(DictConstant.MDM_ORG_TYPE);
        //组织类型名称：编码map
        Map<String, String> orgTypeValueCodeMap = new HashMap<>(16);
        for (Map.Entry<String, String> entry :
                orgTypeMap.entrySet()) {
            if (!orgTypeValueCodeMap.containsKey(entry.getValue())) {
                orgTypeValueCodeMap.put(entry.getValue(), entry.getKey());
            }
        }

        //用于判断excel重复编码
        Map<String, Integer> originOrgCodeMap = new HashMap<>(16);

        //数据库已有编码重复校验
        Set<String> orgCodeUniqueSet = new HashSet<>(16);
        Set<String> originOrgCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getOrgCode())).map(MdmOrgImportVo::getOrgCode).collect(Collectors.toSet());
        if (!originOrgCodeSet.isEmpty()) {
            QueryWrapper<MdmOrgEntity> orgCodeUniqueQuery = new QueryWrapper<>();
            orgCodeUniqueQuery.in("org_code", originOrgCodeSet);
            List<MdmOrgEntity> orgCodeUniqueList = mdmOrgMapper.selectList(orgCodeUniqueQuery);
            if (CollectionUtil.listNotEmptyNotSizeZero(orgCodeUniqueList)) {
                orgCodeUniqueSet.addAll(orgCodeUniqueList.stream().map(MdmOrgEntity::getOrgCode).collect(Collectors.toSet()));
            }
        }

        //数据库已有上级编码
        Set<String> parentCodeCheckSet = new HashSet<>(16);
        Set<String> originParentCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmOrgImportVo::getParentCode).collect(Collectors.toSet());
        if (!originParentCodeSet.isEmpty()) {
            QueryWrapper<MdmOrgEntity> parentQuery = new QueryWrapper<>();
            parentQuery.in("org_code", originParentCodeSet);
            List<MdmOrgEntity> parentCodeCheckList = mdmOrgMapper.selectList(parentQuery);
            if (CollectionUtil.listNotEmptyNotSizeZero(parentCodeCheckList)) {
                parentCodeCheckSet.addAll(parentCodeCheckList.stream().map(MdmOrgEntity::getOrgCode).collect(Collectors.toSet()));
            }
        }


        for (MdmOrgImportVo item :
                originList) {

            //校验编码
            if (StringUtils.isNotEmpty(item.getOrgCode())) {
                if (originOrgCodeMap.containsKey(item.getOrgCode())) {
                    item.appendErrorValidateMsg("组织编码与第" + originOrgCodeMap.get(item.getOrgCode()) + "行重复；");
                } else {
                    originOrgCodeMap.put(item.getOrgCode(), item.getRowIndex());
                }
                if (orgCodeUniqueSet.contains(item.getOrgCode())) {
                    item.appendErrorValidateMsg("组织编码已存在；");
                }
            }

            //校验名称
            if (StringUtils.isEmpty(item.getOrgName())) {
                item.appendErrorValidateMsg("缺失组织名称；");
            }

            //校验组织层级
            if (StringUtils.isEmpty(item.getOrgTypeName())) {
                item.appendErrorValidateMsg("缺失组织层级；");
            } else {
                if (!orgTypeValueCodeMap.containsKey(item.getOrgTypeName())) {
                    item.appendErrorValidateMsg("组织层级无效；");
                } else {
                    item.setOrgType(orgTypeValueCodeMap.get(item.getOrgTypeName()));
                }
            }
        }

        for (MdmOrgImportVo item :
                originList) {
            if (StringUtils.isEmpty(item.getParentCode())) {
                continue;
            }
            if (parentCodeCheckSet.contains(item.getParentCode())) {
                continue;
            }
            if (originOrgCodeSet.contains(item.getParentCode())) {
                continue;
            }
            item.appendErrorValidateMsg("上级组织编码无效；");
        }

        Set<String> errorCode = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getOrgCode()) && MdmOrgImportVo.ProcessTypeEnum.SUCCESS != item.getProcessType()).map(MdmOrgImportVo::getOrgCode).collect(Collectors.toSet());
        if (!errorCode.isEmpty()) {
            setErrorParent(originList, errorCode);
        }

    }

    private void setErrorParent(List<MdmOrgImportVo> originList, Set<String> errorCode) {
        if (errorCode.isEmpty()) {
            return;
        }
        final Set<String> codeOnTheTree = getCodeOnTheTree(originList);
        //TODO 校验不完善，如果某个节点校验未通过，则全部下级都不能通过
        originList.forEach(item -> {
            if (StringUtils.isNotEmpty(item.getParentCode()) && errorCode.contains(item.getParentCode())) {
                item.appendErrorValidateMsg("上级校验未通过；");
            }
            if (StringUtils.isNotEmpty(item.getOrgCode())) {
                if (!codeOnTheTree.contains(item.getOrgCode())) {
                    item.appendErrorValidateMsg("可能处在环形结构上或者上级处于环形结构上；");
                }
            }
        });
    }

    private Set<String> getCodeOnTheTree(List<MdmOrgImportVo> originList) {
        Set<String> orgCodeOnTheTree = new HashSet<>(16);
        List<MdmOrgImportVo> totalList = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getOrgCode())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(totalList)) {
            final Set<String> totalSet = totalList.stream().map(MdmOrgImportVo::getOrgCode).collect(Collectors.toSet());
            final Set<String> topCodeSet = totalList.stream().filter(item -> StringUtils.isEmpty(item.getParentCode()) || !totalSet.contains(item.getParentCode())).map(MdmOrgImportVo::getOrgCode).collect(Collectors.toSet());
            if (!topCodeSet.isEmpty()) {
                Set<String> curLevelSet = new HashSet<>(16);
                curLevelSet.addAll(topCodeSet);
                orgCodeOnTheTree.addAll(curLevelSet);
                while (!curLevelSet.isEmpty()) {
                    Set<String> collect = totalList.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode()) && curLevelSet.contains(item.getParentCode()) && !item.getParentCode().equals(item.getOrgCode())).map(MdmOrgImportVo::getOrgCode).collect(Collectors.toSet());
                    curLevelSet.clear();
                    if (!collect.isEmpty()) {
                        orgCodeOnTheTree.addAll(collect);
                        curLevelSet.addAll(collect);
                    }
                }
            }
        }
        return orgCodeOnTheTree;
    }

}
