package com.biz.crm.excel.component.validator.mdm.price;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.PriceSettingFieldEnum;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.price.MdmPriceSettingImportVo;
import com.biz.crm.mdm.customer.mapper.MdmCustomerMsgMapper;
import com.biz.crm.mdm.customer.model.MdmCustomerEntity;
import com.biz.crm.mdm.org.entity.MdmOrgEntity;
import com.biz.crm.mdm.org.mapper.MdmOrgMapper;
import com.biz.crm.mdm.price.mapper.MdmPriceConditionGroupMapper;
import com.biz.crm.mdm.price.mapper.MdmPriceConditionGroupRelFieldMapper;
import com.biz.crm.mdm.price.mapper.MdmPriceConditionTypeMapper;
import com.biz.crm.mdm.price.mapper.MdmPriceSettingMapper;
import com.biz.crm.mdm.price.model.MdmPriceConditionGroupEntity;
import com.biz.crm.mdm.price.model.MdmPriceConditionGroupRelFieldEntity;
import com.biz.crm.mdm.price.model.MdmPriceConditionTypeEntity;
import com.biz.crm.mdm.price.model.MdmPriceSettingEntity;
import com.biz.crm.mdm.product.entity.MdmProductEntity;
import com.biz.crm.mdm.product.mapper.MdmProductMapper;
import com.biz.crm.mdm.terminal.entity.MdmTerminalEntity;
import com.biz.crm.mdm.terminal.mapper.MdmTerminalMapper;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.DictUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 价格导入验证
 *
 * @author zxw
 * @date 2021-01-25 14:28
 **/
@Slf4j
@Component("mdmPriceImportValidator")
public class MdmPriceImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmPriceSettingMapper, MdmPriceSettingEntity, MdmPriceSettingImportVo> implements ExcelImportValidator<MdmPriceSettingImportVo> {

    @Resource
    private MdmPriceConditionGroupMapper mdmPriceConditionGroupMapper;

    @Resource
    private MdmPriceConditionTypeMapper mdmPriceConditionTypeMapper;

    @Resource
    private MdmProductMapper mdmProductMapper;

    @Resource
    private MdmOrgMapper mdmOrgMapper;

    @Resource
    private MdmCustomerMsgMapper mdmCustomerMsgMapper;

    @Resource
    private MdmPriceConditionGroupRelFieldMapper mdmPriceConditionGroupRelFieldMapper;

    @Resource
    private MdmTerminalMapper mdmTerminalMapper;

    @Override
    public void validate(List<MdmPriceSettingImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validConditionGroupCode(data);
        validConditionTypeCode(data);
        validPrice(data);
        validDate(data);
        validCurrencyType(data);
        validProduct(data);
    }

    protected void validProduct(List<MdmPriceSettingImportVo> data) {
        LambdaQueryWrapper<MdmProductEntity> wrapper = Wrappers.<MdmProductEntity>lambdaQuery()
                .select(MdmProductEntity::getProductCode, MdmProductEntity::getBaseUnit, MdmProductEntity::getSaleUnit, MdmProductEntity::getEnableStatus);
        List<MdmProductEntity> mdmProductEntities = mdmProductMapper.selectList(wrapper);

        Map<String, MdmProductEntity> dbProductMap = Optional.ofNullable(mdmProductEntities).orElse(new ArrayList<>())
                .stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(MdmProductEntity::getProductCode, Function.identity()));

        Map<String, String> baseUnitMap = DictUtil.dictRevertMap("product_base_unit");
        Map<String, String> unitTypeMap = DictUtil.dictRevertMap("unit_type");
        Map<String, String> saleUnitMap = DictUtil.dictRevertMap("product_sale_unit");

        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            String productCode = mdmPriceSettingImportVo.getProductCode();
            if (StringUtils.isEmpty(productCode)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("商品编码不能为空;");
                continue;
            }
            MdmProductEntity mdmProductEntity = dbProductMap.get(productCode);
            if (mdmProductEntity == null) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("商品编码" + productCode + "不存在;");
                continue;
            }
            if (!CrmEnableStatusEnum.ENABLE.getCode().equals(mdmProductEntity.getEnableStatus())) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("商品编码" + productCode + "被禁用;");
                continue;
            }
            String unitType = mdmPriceSettingImportVo.getUnitType();
            if (StringUtils.isEmpty(unitType)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("单位类型不能为空;");
                continue;
            }
            if (unitTypeMap == null) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("数据字典unit_type不存在;");
                continue;
            }
            if (!unitTypeMap.containsKey(unitType)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("单位类型" + unitType + "在数据字典unit_type中不存在;");
                continue;
            }
            String priceUnit = mdmPriceSettingImportVo.getPriceUnit();
            if (StringUtils.isEmpty(priceUnit)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("价格单位不能为空;");
                continue;
            }

            String unitTypeKey = unitTypeMap.get(unitType);
            if ("baseUnit".equals(unitTypeKey)) {
                if (baseUnitMap == null) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("数据字典product_base_unit不存在;");
                    continue;
                }
                String basePriceUnit = baseUnitMap.get(priceUnit);
                if (StringUtils.isEmpty(basePriceUnit)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg(priceUnit + "在数据字典product_base_unit不存在;");
                    continue;
                }
                String baseUnit = mdmProductEntity.getBaseUnit();
                if (!basePriceUnit.equals(baseUnit)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg(priceUnit + "在商品" + productCode + "不存在;");
                }
            } else if ("saleUnit".equals(unitTypeKey)) {
                if (saleUnitMap == null) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("数据字典product_sale_unit不存在;");
                    continue;
                }
                String salePriceUnit = saleUnitMap.get(priceUnit);
                if (StringUtils.isEmpty(salePriceUnit)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg(priceUnit + "在数据字典product_sale_unit不存在;");
                    continue;
                }
                String saleUnit = mdmProductEntity.getSaleUnit();
                if (!salePriceUnit.equals(saleUnit)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg(priceUnit + "在商品" + productCode + "不存在;");
                }
            } else {
                mdmPriceSettingImportVo.appendErrorValidateMsg("不支持的数据字典类型:" + unitType + ";");
            }
        }
    }

    protected void validCurrencyType(List<MdmPriceSettingImportVo> data) {
        Map<String, String> map = DictUtil.dictRevertMap("currency_type");
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            String currencyType = mdmPriceSettingImportVo.getCurrencyType();
            if (StringUtils.isEmpty(currencyType)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("币种不能为空;");
                continue;
            }
            if (map == null) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("币种数据字典currency_type不存在;");
                continue;
            }
            if (!map.containsKey(currencyType)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("币种:" + currencyType + "在数据字典currency_type不存在");
            }
        }
    }

    protected void validDate(List<MdmPriceSettingImportVo> data) {
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            String beginDate = mdmPriceSettingImportVo.getBeginDate();
            if (StringUtils.isEmpty(beginDate)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("有效开始时间不能为空;");
                continue;
            }
            String endDate = mdmPriceSettingImportVo.getEndDate();
            if (StringUtils.isEmpty(endDate)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("有效结束时间不能为空;");
                continue;
            }
            if (!verifyDateFormat(beginDate, DateUtil.DEFAULT_DAY_PATTERN)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("开始时间格式应该为" + DateUtil.DEFAULT_DAY_PATTERN);
            }
            if (!verifyDateFormat(endDate, DateUtil.DEFAULT_DAY_PATTERN)) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("结束时间格式应该为" + DateUtil.DEFAULT_DAY_PATTERN);
            }
            if (beginDate.compareTo(endDate) >= 0) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("有效结束时间必须大于有效开始时间");
            }
        }
    }

    protected void validPrice(List<MdmPriceSettingImportVo> data) {
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            if (StringUtils.isEmpty(mdmPriceSettingImportVo.getPrice())) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("价格不能为空;");
            }
        }
    }

    protected void validConditionTypeCode(List<MdmPriceSettingImportVo> data) {
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            if (StringUtils.isEmpty(mdmPriceSettingImportVo.getConditionTypeCode())) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("条件类型编码不能为空;");
            }
        }
        LambdaQueryWrapper<MdmPriceConditionTypeEntity> wrapper = Wrappers.<MdmPriceConditionTypeEntity>lambdaQuery()
                .select(MdmPriceConditionTypeEntity::getConditionTypeCode);
        List<String> dbConditionTypeCodeList = mdmPriceConditionTypeMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getConditionTypeCode()))
                .map(MdmPriceConditionTypeEntity::getConditionTypeCode)
                .collect(Collectors.toList());
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            String conditionTypeCode = mdmPriceSettingImportVo.getConditionTypeCode();
            if (!StringUtils.isEmpty(conditionTypeCode)) {
                if (CollectionUtils.isEmpty(dbConditionTypeCodeList) || !dbConditionTypeCodeList.contains(conditionTypeCode)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("条件类型编码" + conditionTypeCode + "不存在;");
                }
            }
        }
    }

    protected void validConditionGroupCode(List<MdmPriceSettingImportVo> data) {
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            if (StringUtils.isEmpty(mdmPriceSettingImportVo.getConditionGroupCode())) {
                mdmPriceSettingImportVo.appendErrorValidateMsg("条件字段分类编码不能为空;");
            }
        }
        LambdaQueryWrapper<MdmPriceConditionGroupEntity> wrapper = Wrappers.<MdmPriceConditionGroupEntity>lambdaQuery()
                .select(MdmPriceConditionGroupEntity::getConditionGroupCode);
        List<String> dbConditionGroupCodeList = mdmPriceConditionGroupMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getConditionGroupCode()))
                .map(MdmPriceConditionGroupEntity::getConditionGroupCode)
                .collect(Collectors.toList());
        LambdaQueryWrapper<MdmPriceConditionGroupRelFieldEntity> fieldWrapper = Wrappers.<MdmPriceConditionGroupRelFieldEntity>lambdaQuery()
                .select(MdmPriceConditionGroupRelFieldEntity::getFieldCode, MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode);

        Map<String, List<String>> collect = mdmPriceConditionGroupRelFieldMapper.selectList(fieldWrapper).stream()
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(MdmPriceConditionGroupRelFieldEntity::getConditionGroupCode
                        , Collectors.mapping(MdmPriceConditionGroupRelFieldEntity::getFieldCode, Collectors.toList())));


        LambdaQueryWrapper<MdmCustomerEntity> customerWrapper = Wrappers.<MdmCustomerEntity>lambdaQuery()
                .select(MdmCustomerEntity::getCustomerCode, MdmCustomerEntity::getEnableStatus);
        Map<String, String> dbCustomerMap = mdmCustomerMsgMapper.selectList(customerWrapper).stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getCustomerCode()))
                .collect(Collectors.toMap(MdmCustomerEntity::getCustomerCode, CrmBaseEntity::getEnableStatus));
        LambdaQueryWrapper<MdmOrgEntity> orgWrapper = Wrappers.<MdmOrgEntity>lambdaQuery()
                .select(MdmOrgEntity::getOrgCode, MdmOrgEntity::getEnableStatus);

        Map<String, String> dbOrgMap = mdmOrgMapper.selectList(orgWrapper).stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getOrgCode()))
                .collect(Collectors.toMap(MdmOrgEntity::getOrgCode, CrmBaseEntity::getEnableStatus));
        LambdaQueryWrapper<MdmTerminalEntity> terminalMapper = Wrappers.<MdmTerminalEntity>lambdaQuery()
                .select(MdmTerminalEntity::getTerminalCode, MdmTerminalEntity::getEnableStatus);
        Map<String, String> dbTerminalMap = mdmTerminalMapper.selectList(terminalMapper).stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
                .collect(Collectors.toMap(MdmTerminalEntity::getTerminalCode, MdmTerminalEntity::getEnableStatus));

        Map<String, String> channelMap = DictUtil.dictRevertMap("channel");
        Map<String, String> priceGroupMap = DictUtil.dictRevertMap("price_group");
        for (MdmPriceSettingImportVo mdmPriceSettingImportVo : data) {
            String conditionGroupCode = mdmPriceSettingImportVo.getConditionGroupCode();
            if (!StringUtils.isEmpty(conditionGroupCode)) {
                if (CollectionUtils.isEmpty(dbConditionGroupCodeList) || !dbConditionGroupCodeList.contains(conditionGroupCode)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("条件字段分类编码" + conditionGroupCode + "不存在;");
                    continue;
                }
                List<String> fieldList = collect.get(conditionGroupCode);
                if (CollectionUtils.isEmpty(fieldList)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("条件字段分类编码" + conditionGroupCode + "未设置条件字段分类描述;");
                }

                String errorField = "";
                if (!fieldList.contains("orgCode") && !StringUtils.isEmpty(mdmPriceSettingImportVo.getOrgCode())) {
                    if (!StringUtils.isEmpty(errorField)) {
                        errorField += "和";
                    }
                    errorField += "组织";
                }
                if (!fieldList.contains("customerCode") && !StringUtils.isEmpty(mdmPriceSettingImportVo.getCustomerCode())) {
                    if (!StringUtils.isEmpty(errorField)) {
                        errorField += "和";
                    }
                    errorField += "客户";
                }
                if (!fieldList.contains("channel") && !StringUtils.isEmpty(mdmPriceSettingImportVo.getChannel())) {
                    if (!StringUtils.isEmpty(errorField)) {
                        errorField += "和";
                    }
                    errorField += "渠道";
                }
                if (!fieldList.contains("priceGroup") && !StringUtils.isEmpty(mdmPriceSettingImportVo.getPriceGroup())) {
                    if (!StringUtils.isEmpty(errorField)) {
                        errorField += "和";
                    }
                    errorField += "价格组";
                }
                if (!fieldList.contains("terminalCode") && !StringUtils.isEmpty(mdmPriceSettingImportVo.getTerminalCode())) {
                    if (!StringUtils.isEmpty(errorField)) {
                        errorField += "和";
                    }
                    errorField += "终端";
                }
                if (!StringUtils.isEmpty(errorField)) {
                    mdmPriceSettingImportVo.appendErrorValidateMsg("条件字段分类编码未包含" + errorField + "字段," + errorField + "字段不能输入信息");
                }
                for (String field : fieldList) {
                    PriceSettingFieldEnum match = PriceSettingFieldEnum.match(field);
                    switch (match) {
                        case ORG_CODE:
                            String orgCode = mdmPriceSettingImportVo.getOrgCode();
                            if (StringUtils.isEmpty(orgCode)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("组织编码不能为空;");
                                break;
                            }
                            if (!dbOrgMap.containsKey(orgCode)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("组织编码" + orgCode + "不存在;");
                                break;
                            }
                            if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbOrgMap.get(orgCode))) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("组织编码" + orgCode + "被禁用;");
                            }
                            break;
                        case CUSTOMER_CODE:
                            String customerCode = mdmPriceSettingImportVo.getCustomerCode();
                            if (StringUtils.isEmpty(customerCode)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("客户编码不能为空;");
                                break;
                            }
                            if (!dbCustomerMap.containsKey(customerCode)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("客户编码" + customerCode + "不存在;");
                                break;
                            }
                            if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbCustomerMap.get(customerCode))) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("客户编码" + customerCode + "被禁用;");
                            }
                            break;
                        case TERMINAL:
                            String terminalCode = mdmPriceSettingImportVo.getTerminalCode();
                            if(StringUtils.isEmpty(terminalCode)){
                                mdmPriceSettingImportVo.appendErrorValidateMsg("终端编码不能为空;");
                                break;
                            }
                            if (!dbTerminalMap.containsKey(terminalCode)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("终端编码" + terminalCode + "不存在");
                                break;
                            }
                            if (!CrmEnableStatusEnum.ENABLE.getCode().equals(dbTerminalMap.get(terminalCode))) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("终端编码" + terminalCode + "被禁用;");
                            }
                        case CHANNEL:
                            String channel = mdmPriceSettingImportVo.getChannel();
                            if (StringUtils.isEmpty(channel)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("渠道不能为空;");
                                break;
                            }
                            if (channelMap == null) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("数据字典channel不存在;");
                                break;
                            }
                            if (!channelMap.containsKey(channel)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("渠道" + channel + "不存在;");
                            }
                        case PRICE_GROUP:
                            String priceGroup = mdmPriceSettingImportVo.getPriceGroup();
                            if(StringUtils.isEmpty(priceGroup)){
                                mdmPriceSettingImportVo.appendErrorValidateMsg("价格组不能为空");
                                break;
                            }
                            if (priceGroupMap == null) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("数据字典price_group不存在;");
                                break;
                            }
                            if (!priceGroupMap.containsKey(priceGroup)) {
                                mdmPriceSettingImportVo.appendErrorValidateMsg("价格组" + priceGroup + "不存在;");
                            }
                        default:
                    }
                }
            }
        }
    }

    private boolean verifyDateFormat(String dataStr, String formatStr) {
        boolean flag = false;
        if (com.biz.crm.util.StringUtils.isNotEmpty(dataStr) && com.biz.crm.util.StringUtils.isNotEmpty(formatStr)) {
            try {
                SimpleDateFormat sdf = new SimpleDateFormat(formatStr);
                Date parse = sdf.parse(dataStr);
                String dateFormatStr = sdf.format(parse);
                if (dataStr.equals(dateFormatStr)) {
                    flag = true;
                }
            } catch (Exception e) {

            }
        }
        return flag;
    }
}
