package com.biz.crm.excel.component.validator.mdm.product;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.product.MdmProductImportVo;
import com.biz.crm.mdm.material.entity.MdmMaterialEntity;
import com.biz.crm.mdm.material.mapper.MdmMaterialMapper;
import com.biz.crm.mdm.product.entity.MdmProductEntity;
import com.biz.crm.mdm.product.entity.MdmProductLevelEntity;
import com.biz.crm.mdm.product.mapper.MdmProductLevelMapper;
import com.biz.crm.mdm.product.mapper.MdmProductMapper;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @Author hej
 * @Created Date 2020-12-30 15:57
 **/

@Slf4j
@Component("mdmProductImportValidator")
public class MdmProductImportValidator <M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmProductMapper, MdmProductEntity, MdmProductImportVo> implements ExcelImportValidator<MdmProductImportVo> {


    @Resource
    private MdmProductMapper mdmProductMapper;

    @Resource
    private MdmProductLevelMapper mdmProductLevelMapper;

    @Resource
    private MdmMaterialMapper mdmMaterialMapper;

    private Pattern pattern = Pattern.compile("[0-9]*");

    @Override
    public void validate(List<MdmProductImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        checkProductNull(data);
        checkProductCode(data);
        checkProductType(data);
        checkProductLevelCode(data);
        checkIsShelf(data);
        checkProductBaseUnit(data);
        checkProductSaleUnit(data);
        checkMaterialCode(data);
        checkCount(data);
    }

    /**
     * 校验空值
     */
    protected void checkProductNull(List<MdmProductImportVo> vos) {
        if(CollectionUtil.listNotEmptyNotSizeZero(vos)) {
            vos.forEach(m -> {
                if (StringUtils.isEmpty(m.getProductName())) {
                    m.appendErrorValidateMsg("商品名称不能为空；");
                }
                if (StringUtils.isEmpty(m.getProductTypeName())) {
                    m.appendErrorValidateMsg("商品类型不能为空；");
                }
                if (StringUtils.isEmpty(m.getIsShelfName())) {
                    m.appendErrorValidateMsg("上下架状态不能为空；");
                }
                if (StringUtils.isEmpty(m.getBeginDate())) {
                    m.appendErrorValidateMsg("开始时间不能为空；");
                }
                if (StringUtils.isEmpty(m.getEndDate())) {
                    m.appendErrorValidateMsg("结束时间不能为空；");
                }
//            if(StringUtils.isEmpty(m.getMaterialCode())){
//                m.appendErrorValidateMsg("行号:" + m.getRowIndex() + "物料编码不能为空");
//            }
//            if(StringUtils.isEmpty(m.getMaterialName())){
//                m.appendErrorValidateMsg("行号:" + m.getRowIndex() + "物料名称不能为空");
//            }
            });
        }
    }

    /**
     * 校验商品编码
     *
     * @param vos
     */
    protected void checkProductCode(List<MdmProductImportVo> vos) {
        Set<String> collect = vos.stream().filter(item -> StringUtils.isNotEmpty(item.getProductCode())).map(MdmProductImportVo::getProductCode).collect(Collectors.toSet());
        if (collect.isEmpty()) {
            return;
        }
        LambdaQueryWrapper<MdmProductEntity> wrapper = Wrappers.<MdmProductEntity>lambdaQuery().select(MdmProductEntity::getProductCode);
        wrapper.in(MdmProductEntity::getProductCode, collect);
        final Set<String> materialCodeSet = mdmProductMapper.selectList(wrapper).stream()
                .map(MdmProductEntity::getProductCode)
                .collect(Collectors.toSet());

        Map<String, Integer> codeIndexMap = new HashMap<>(16);
        for (MdmProductImportVo item : vos) {
            if (StringUtils.isEmpty(item.getProductCode())) {
                continue;
            }
            if (materialCodeSet.contains(item.getProductCode())) {
                item.appendErrorValidateMsg("商品编码已存在；");
            } else if (codeIndexMap.containsKey(item.getProductCode())) {
                item.appendErrorValidateMsg("商品编码与第" + item.getProductCode() + "行重复；");
            } else {
                codeIndexMap.put(item.getProductCode(), item.getRowIndex());
            }
        }

    }

    /**
     * 校验产品层级编码
     * @param vos
     */
    protected void checkProductLevelCode(List<MdmProductImportVo> vos){
        Set<String> productsSet = new HashSet<>();
        LambdaQueryWrapper<MdmProductLevelEntity> wrapper = Wrappers.<MdmProductLevelEntity>lambdaQuery()
                .select(MdmProductLevelEntity::getProductLevelCode);
        List<MdmProductLevelEntity> levelEntities = mdmProductLevelMapper.selectList(wrapper);
        if(CollectionUtil.listNotEmptyNotSizeZero(levelEntities)) {
            List<String> list = levelEntities.stream()
                    .filter(m -> StringUtils.isNotEmpty(m.getProductLevelCode()))
                    .map(m -> m.getProductLevelCode())
                    .collect(Collectors.toList());
            if(CollectionUtil.listNotEmptyNotSizeZero(list)){
                productsSet.addAll(list);
            }
        }
        for (MdmProductImportVo vo : vos) {
            String levelCode = vo.getProductLevelCode();
            if(StringUtils.isNotEmpty(levelCode) && productsSet.add(levelCode)){
                vo.appendErrorValidateMsg(" 产品层级编码【"+levelCode+"】不存在；");
            }
        }
    }

    /**
     * 检查物料编码
     * @param vos
     */
    protected void checkMaterialCode(List<MdmProductImportVo> vos){
        List<MdmProductImportVo> material = vos.stream().filter(m -> StringUtils.isNotEmpty(m.getMaterialCode())).
                collect(Collectors.toList());
        if(CollectionUtil.listNotEmptyNotSizeZero(material)) {
            Set<String> materialSet = new HashSet<>();
            LambdaQueryWrapper<MdmMaterialEntity> wrapper = Wrappers.<MdmMaterialEntity>lambdaQuery().select(MdmMaterialEntity::getMaterialCode);
            List<MdmMaterialEntity> mdmMaterialEntities = mdmMaterialMapper.selectList(wrapper);
            if (CollectionUtil.listNotEmptyNotSizeZero(mdmMaterialEntities)) {
                List<String> list = mdmMaterialEntities.stream()
                        .filter(m -> StringUtils.isNotEmpty(m.getMaterialCode()))
                        .map(m -> m.getMaterialCode())
                        .collect(Collectors.toList());
                if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                    materialSet.addAll(list);
                }
            }
            for (MdmProductImportVo vo : material) {
                if (materialSet.add(vo.getMaterialCode())) {
                    vo.appendErrorValidateMsg(" 物料编码【" + vo.getMaterialCode() + "】不存在；");
                    break;
                }
//                if(StringUtils.isEmpty(vo.getMaterialName())){
//                    vo.appendErrorValidateMsg(" 物料名称:" + vo.getMaterialCode() + "不能为空；");
//                }
            }
        }
    }

    /**
     * 数量校验
     * @param vos
     */
    protected void checkCount(List<MdmProductImportVo> vos){
        if(CollectionUtil.listNotEmptyNotSizeZero(vos)){
            vos.stream().filter(m-> !Objects.isNull(m.getCount())).forEach(m->{
                if(!pattern.matcher(m.getCount()).matches()){
                    m.appendErrorValidateMsg(" 数量：必须为正整数；");
                }
            });
        }
    }

    /**
     * 字典校验商品类型
     * @param vos
     */
    protected void checkProductType(List<MdmProductImportVo> vos){
        Map<String, String> map = DictUtil.dictMap("product_type");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmProductImportVo mdmProductImportVo : vos) {
            String productTypeName = mdmProductImportVo.getProductTypeName();
            if (!org.springframework.util.StringUtils.isEmpty(productTypeName)) {
                if (!channelMap.containsKey(productTypeName)) {
                    mdmProductImportVo.appendErrorValidateMsg("商品类型【" + productTypeName + "】不存在;");
                }else {
                    mdmProductImportVo.setProductType(channelMap.get(productTypeName));
                }
            }
        }
    }

    /**
     * 上下架状态校验
     * @param vos
     */
    protected void checkIsShelf(List<MdmProductImportVo> vos){
        Map<String, String> map = DictUtil.dictMap("is_shelf");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmProductImportVo mdmProductImportVo : vos) {
            String isShelfName = mdmProductImportVo.getIsShelfName();
            if (!org.springframework.util.StringUtils.isEmpty(isShelfName)) {
                if (!channelMap.containsKey(isShelfName)) {
                    mdmProductImportVo.appendErrorValidateMsg("上下架状态【" + isShelfName + "】不存在;");
                }else {
                    mdmProductImportVo.setIsShelf(channelMap.get(isShelfName));
                }
            }
        }
    }

    /**
     * 基本单位字典
     * @param vos
     */
    protected void checkProductBaseUnit(List<MdmProductImportVo> vos){
        Map<String, String> map = DictUtil.dictMap("product_base_unit");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmProductImportVo mdmProductImportVo: vos) {
            String baseUnitName = mdmProductImportVo.getBaseUnitName();
            if (!org.springframework.util.StringUtils.isEmpty(baseUnitName)) {
                if (!channelMap.containsKey(baseUnitName)) {
                    mdmProductImportVo.appendErrorValidateMsg("商品基本单位【" + baseUnitName + "】不存在;");
                }else {
                    mdmProductImportVo.setBaseUnit(channelMap.get(baseUnitName));
                }
            }
        }
    }

    /**
     * 物料销售单位
     * @param vos
     */
    protected void checkProductSaleUnit(List<MdmProductImportVo> vos){
        Map<String, String> map = DictUtil.dictMap("product_sale_unit");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmProductImportVo mdmProductImportVo : vos) {
            String saleUnitName = mdmProductImportVo.getSaleUnitName();
            if (!org.springframework.util.StringUtils.isEmpty(saleUnitName)) {
                if (!channelMap.containsKey(saleUnitName)) {
                    mdmProductImportVo.appendErrorValidateMsg("商品销售单位【" + saleUnitName + "】不存在;");
                }else {
                    mdmProductImportVo.setSaleUnit(channelMap.get(saleUnitName));
                }
            }
        }
    }

}
