package com.biz.crm.excel.component.validator.mdm.product;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.product.MdmProductLevelImportVo;
import com.biz.crm.mdm.product.entity.MdmProductLevelEntity;
import com.biz.crm.mdm.product.mapper.MdmProductLevelMapper;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author hej
 * @Created Date 2021-01-04 14:45
 **/

@Slf4j
@Component("mdmProductLevelImportValidator")
public class MdmProductLevelImportValidator <M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmProductLevelMapper, MdmProductLevelEntity, MdmProductLevelImportVo> implements ExcelImportValidator<MdmProductLevelImportVo> {

    @Resource
    private MdmProductLevelMapper mdmProductLevelMapper;


    @Override
    public void validate(List<MdmProductLevelImportVo> data, DefaultImportContext context) {
        checkProductLevelNull(data);
        checkProductLevelCode(data);
        checkProductLevelType(data);
    }

    /**
     * 校验空值
     */
    protected void checkProductLevelNull(List<MdmProductLevelImportVo> vos) {
        vos.stream().forEach(m -> {
            if (StringUtils.isEmpty(m.getProductLevelName())) {
                m.appendErrorValidateMsg("产品层级名称不能为空；");
            }
            if(StringUtils.isEmpty(m.getProductLevelTypeName())){
                m.appendErrorValidateMsg("产品层级类型不能为空；");
            }
        });
    }

    /**
     * 校验产品层级编码
     *
     * @param vos
     */
    protected void checkProductLevelCode(List<MdmProductLevelImportVo> vos) {

        Set<String> productLevelCodeSet = vos.stream().filter(item -> StringUtils.isNotEmpty(item.getProductLevelCode())).map(MdmProductLevelImportVo::getProductLevelCode).collect(Collectors.toSet());
        productLevelCodeSet.addAll(vos.stream().filter(item -> StringUtils.isNotEmpty(item.getParentCode())).map(MdmProductLevelImportVo::getParentCode).collect(Collectors.toSet()));
        if (productLevelCodeSet.isEmpty()) {
            return;
        }
        LambdaQueryWrapper<MdmProductLevelEntity> wrapper = Wrappers.<MdmProductLevelEntity>lambdaQuery()
                .in(MdmProductLevelEntity::getProductLevelCode, productLevelCodeSet)
                .select(MdmProductLevelEntity::getProductLevelCode);
        final Set<String> collect = mdmProductLevelMapper.selectList(wrapper).stream()
                .filter(m -> StringUtils.isNotEmpty(m.getProductLevelCode()))
                .map(MdmProductLevelEntity::getProductLevelCode)
                .collect(Collectors.toSet());
        Map<String, Integer> codeIndexMap = new HashMap<>(16);
        Set<String> productsSet = new HashSet<>();
        for (MdmProductLevelImportVo vo : vos) {
            if (StringUtils.isNotEmpty(vo.getProductLevelCode()) && collect.contains(vo.getProductLevelCode())) {
                vo.appendErrorValidateMsg("产品层级编码:" + vo.getParentCode() + "已存在；");
            } else if (codeIndexMap.containsKey(vo.getProductLevelCode())) {
                vo.appendErrorValidateMsg("产品层级编码:" + vo.getProductLevelCode() + "与第" + codeIndexMap.get(vo.getProductLevelCode()) + "行重复；");
            } else {
                productsSet.add(vo.getProductLevelCode());
                codeIndexMap.put(vo.getProductLevelCode(), vo.getRowIndex());
            }
        }

        productsSet.addAll(collect);
        for (MdmProductLevelImportVo vo : vos) {
            if (StringUtils.isNotEmpty(vo.getParentCode()) && !productsSet.contains(vo.getParentCode())) {
                vo.appendErrorValidateMsg("产品上层编码:" + vo.getParentCode() + "不存在；");
            }
        }
    }

    /**
     * 校验产品层级类型
     * @param vos
     */
    protected void checkProductLevelType(List<MdmProductLevelImportVo> vos){
        Map<String, String> map = DictUtil.dictMap("mdm_product_level_type");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmProductLevelImportVo mdmProductLevelImportVo : vos) {
            String productLevelTypeName = mdmProductLevelImportVo.getProductLevelTypeName();
            if (!org.springframework.util.StringUtils.isEmpty(productLevelTypeName)) {
                if (!channelMap.containsKey(productLevelTypeName)) {
                    mdmProductLevelImportVo.appendErrorValidateMsg("行号:" + mdmProductLevelImportVo.getRowIndex() + "产品层级类型字典mdm_product_level_type:" + productLevelTypeName + "不存在;");
                }else {
                    mdmProductLevelImportVo.setProductLevelType(channelMap.get(productLevelTypeName));
                }
            }
        }
    }

}
