package com.biz.crm.excel.component.validator.mdm.terminal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.terminal.MdmTerminalImportVo;
import com.biz.crm.mdm.customerorg.entity.MdmCusOrgEntity;
import com.biz.crm.mdm.customerorg.mapper.MdmCusOrgMapper;
import com.biz.crm.mdm.org.entity.MdmOrgEntity;
import com.biz.crm.mdm.org.mapper.MdmOrgMapper;
import com.biz.crm.mdm.region.entity.MdmRegionEntity;
import com.biz.crm.mdm.region.mapper.MdmRegionMapper;
import com.biz.crm.mdm.terminal.entity.MdmTerminalEntity;
import com.biz.crm.mdm.terminal.mapper.MdmTerminalMapper;
import com.biz.crm.mdm.terminal.mapper.MdmTerminalROrgMapper;
import com.biz.crm.nebular.mdm.constant.RegionLevelEnum;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.websocket.WebsocketUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author: luoqi
 * @Date: 2020-12-11 10:50
 * @version: V1.0
 * @Description:
 */
@Slf4j
@Component("mdmTerminalImportValidator")
public class MdmTerminalImportValidator <M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmTerminalMapper, MdmTerminalEntity, MdmTerminalImportVo> implements ExcelImportValidator<MdmTerminalImportVo> {

    private String phone = "^[1]+[3,8]+\\d{9}$";
    private final String NUM_REGEX = "^[0-9]*$";

    @Resource
    private MdmTerminalMapper mdmTerminalMapper;

    @Resource
    private MdmTerminalROrgMapper mdmTerminalROrgMapper;

    @Resource
    private MdmOrgMapper mdmOrgMapper;

    @Resource
    private MdmCusOrgMapper mdmCusOrgMapper;

    @Resource
    private MdmRegionMapper mdmRegionMapper;

    @Autowired
    private WebsocketUtil websocketUtil;

    @Override
    public void validate(List<MdmTerminalImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        validTerminalName(data,context);
        validTerminalCode(data,context);
        validOrgCode(data,context);
        validCustomerOrg(data);
        validChannel(data);
        validTerminalType(data);
        validRegion(data);
//        validProvince(data);
//        validCity(data);
//        validDistrict(data);
        validPhone(data);

    }

    protected void validPhone(List<MdmTerminalImportVo> data){
        for (MdmTerminalImportVo mdmTerminalImportVo : data){
            if (!StringUtils.isEmpty(mdmTerminalImportVo.getContactPhone())){
                if (!mdmTerminalImportVo.getContactPhone().matches(NUM_REGEX)){
                    mdmTerminalImportVo.appendErrorValidateMsg("联系人电话仅能输入数字;");
                }
            }
        }
    }

    /**
     * 行政区域校验
     *
     * @param data
     */
    protected void validRegion(List<MdmTerminalImportVo> data) {

        LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                .in(MdmRegionEntity::getRegionLevel, RegionLevelEnum.PROVINCE.getCode(), RegionLevelEnum.CITY.getCode(), RegionLevelEnum.COUNTY.getCode())
                .select(MdmRegionEntity::getRegionCode, MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionLevel, MdmRegionEntity::getParentCode);

        List<MdmRegionEntity> regionList = mdmRegionMapper.selectList(wrapper);
        final Map<String, String> provinceRevertMap = regionList.stream().filter(x -> RegionLevelEnum.PROVINCE.getCode().equals(x.getRegionLevel())).collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1));

        Map<String, List<MdmRegionEntity>> cityGroupMap = regionList.stream().filter(x -> RegionLevelEnum.CITY.getCode().equals(x.getRegionLevel()) && !StringUtils.isEmpty(x.getParentCode())).collect(Collectors.groupingBy(MdmRegionEntity::getParentCode));
        final Map<String, Map<String, String>> provinceCityRevertMap = cityGroupMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, v -> v.getValue().stream().collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1))));

        Map<String, List<MdmRegionEntity>> countyGroupMap = regionList.stream().filter(x -> RegionLevelEnum.COUNTY.getCode().equals(x.getRegionLevel()) && !StringUtils.isEmpty(x.getParentCode())).collect(Collectors.groupingBy(MdmRegionEntity::getParentCode));
        final Map<String, Map<String, String>> cityCountyRevertMap = countyGroupMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, v -> v.getValue().stream().collect(Collectors.toMap(MdmRegionEntity::getRegionName, MdmRegionEntity::getRegionCode, (x1, x2) -> x1))));

        for (MdmTerminalImportVo item : data) {
            if (!StringUtils.isEmpty(item.getProvinceName())) {
                if (provinceRevertMap.containsKey(item.getProvinceName())) {
                    String provinceCode = provinceRevertMap.get(item.getProvinceName());
                    item.setProvinceCode(provinceCode);
                    if (!StringUtils.isEmpty(item.getCityName())) {
                        Map<String, String> cityMap = provinceCityRevertMap.get(provinceCode);
                        if (cityMap != null && cityMap.containsKey(item.getCityName())) {
                            String cityCode = cityMap.get(item.getCityName());
                            item.setCityCode(cityCode);
                            if (!StringUtils.isEmpty(item.getDistrictName())) {
                                Map<String, String> countyMap = cityCountyRevertMap.get(cityCode);
                                if (countyMap != null && countyMap.containsKey(item.getDistrictName())) {
                                    item.setDistrictCode(countyMap.get(cityCode));
                                } else {
                                    item.appendErrorValidateMsg("区不存在；");
                                }
                            }
                        } else {
                            item.appendErrorValidateMsg("市不存在；");
                        }
                    }
                } else {
                    item.appendErrorValidateMsg("省不存在；");
                }
            }
        }
    }

    /**
     * 省校验
     * @param data
     */
    protected void validProvince(List<MdmTerminalImportVo> data){
        LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                .eq(MdmRegionEntity::getRegionLevel,1)
                .select(MdmRegionEntity::getRegionName);
        Set<String> set = mdmRegionMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getRegionName()))
                .map(MdmRegionEntity::getRegionName)
                .collect(Collectors.toSet());
        for (MdmTerminalImportVo mdmTerminalImportVo : data){
            if ( !StringUtils.isEmpty(mdmTerminalImportVo.getProvinceCode())) {
                if (!set.contains(mdmTerminalImportVo.getProvinceCode())) {
                    mdmTerminalImportVo.appendErrorValidateMsg("省不存在;");
                }
            }
        }
    }

    /**
     * 市校验
     * @param data
     */
    protected void validCity(List<MdmTerminalImportVo> data){
        LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                .eq(MdmRegionEntity::getRegionLevel,2)
                .select(MdmRegionEntity::getRegionName);
        Set<String> set = mdmRegionMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getRegionName()))
                .map(MdmRegionEntity::getRegionName)
                .collect(Collectors.toSet());
        for (MdmTerminalImportVo mdmTerminalImportVo : data){
            if ( !StringUtils.isEmpty(mdmTerminalImportVo.getCityCode())) {
                if (!set.contains(mdmTerminalImportVo.getCityCode())) {
                    mdmTerminalImportVo.appendErrorValidateMsg("市不存在;");
                }
            }
        }
    }

    /**
     * 区校验
     * @param data
     */
    protected void validDistrict(List<MdmTerminalImportVo> data){
        LambdaQueryWrapper<MdmRegionEntity> wrapper = Wrappers.<MdmRegionEntity>lambdaQuery()
                .eq(MdmRegionEntity::getRegionLevel,3)
                .select(MdmRegionEntity::getRegionName);
        Set<String> set = mdmRegionMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getRegionName()))
                .map(MdmRegionEntity::getRegionName)
                .collect(Collectors.toSet());
        for (MdmTerminalImportVo mdmTerminalImportVo : data){
            if ( !StringUtils.isEmpty(mdmTerminalImportVo.getDistrictCode())) {
                if (!set.contains(mdmTerminalImportVo.getDistrictCode())) {
                    mdmTerminalImportVo.appendErrorValidateMsg("区不存在;");
                }
            }
        }
    }

    /**
     * 校验渠道
     *
     * @param data list
     */
    protected void validChannel(List<MdmTerminalImportVo> data) {
        Map<String, String> map = DictUtil.dictMap("channel");
        Map<String,String> channelMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
                map.entrySet()){
            if (!channelMap.containsKey(entry.getValue())){
                channelMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmTerminalImportVo mdmTerminalImportVo : data) {
            String channel = mdmTerminalImportVo.getChannel();
            if (StringUtils.isEmpty(channel)) {
                mdmTerminalImportVo.appendErrorValidateMsg("渠道没有填写;");
            }else {
                if (!channelMap.containsKey(channel)){
                    mdmTerminalImportVo.appendErrorValidateMsg("渠道:" + channel + "不存在;");
                }
            }

        }
    }

    /**
     * 校验终端类型
     * @param data
     */
    protected void validTerminalType(List<MdmTerminalImportVo> data){
        Map<String,String> map = DictUtil.dictMap("terminal_type");
        Map<String,String> terminalTypeMap = new HashMap<>(16);
        for (Map.Entry<String,String> entry :
        map.entrySet()){
            if (!terminalTypeMap.containsKey(entry.getValue())){
                terminalTypeMap.put(entry.getValue(),entry.getKey());
            }
        }
        for (MdmTerminalImportVo mdmTerminalImportVo : data){
            String terminalType = mdmTerminalImportVo.getTerminalType();
            if (StringUtils.isEmpty(terminalType)) {
                mdmTerminalImportVo.appendErrorValidateMsg( "终端类型没有填写;");
            }else {
                if ( !terminalTypeMap.containsKey(terminalType)) {
                    mdmTerminalImportVo.appendErrorValidateMsg("终端类型:" + terminalType + "不存在;");
                }
            }

        }
    }

    /**
     * 校验客户组织编码
     *
     * @param data list
     */
    protected void validCustomerOrg(List<MdmTerminalImportVo> data) {
        LambdaQueryWrapper<MdmCusOrgEntity> wrapper = Wrappers.<MdmCusOrgEntity>lambdaQuery()
                .select(MdmCusOrgEntity::getCustomerOrgCode);
        Set<String> dbCustomerOrgCodeSet = mdmCusOrgMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getCustomerOrgCode()))
                .map(MdmCusOrgEntity::getCustomerOrgCode)
                .collect(Collectors.toSet());

        for (MdmTerminalImportVo mdmTerminalImportVo : data) {
            String customerOrgCode = mdmTerminalImportVo.getCustomerOrgCode();
            if (!StringUtils.isEmpty(customerOrgCode)) {
                if (CollectionUtils.isEmpty(dbCustomerOrgCodeSet) || !dbCustomerOrgCodeSet.contains(customerOrgCode)) {
                    mdmTerminalImportVo.appendErrorValidateMsg("客户组织编码:" + customerOrgCode + "不存在;");
                }
            }
        }
    }

    /**
     * 校验组织编码
     *
     * @param data list
     */
    protected void validOrgCode(List<MdmTerminalImportVo> data,DefaultImportContext context) {
        String sid = context.getImportParamVo().getWebSocketClientId();
        websocketUtil.sendMsg(sid,"开始校验组织编码");
        LambdaQueryWrapper<MdmOrgEntity> wrapper = Wrappers.<MdmOrgEntity>lambdaQuery()
                .select(MdmOrgEntity::getOrgCode);
        Set<String> dbOrgCodeSet = mdmOrgMapper.selectList(wrapper)
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getOrgCode()))
                .map(MdmOrgEntity::getOrgCode)
                .collect(Collectors.toSet());

        for (MdmTerminalImportVo mdmTerminalImportVo : data) {
            String orgCode = mdmTerminalImportVo.getOrgCode();
            if (StringUtils.isEmpty(orgCode)) {
                mdmTerminalImportVo.appendErrorValidateMsg("组织编码没有填写;");
            }else {
                if (orgCode.contains(",")) {
                    String[] currentOrgCodeList = orgCode.split(",");
                    for (String currentOrgCode : currentOrgCodeList) {
                        if (CollectionUtils.isEmpty(dbOrgCodeSet) || !dbOrgCodeSet.contains(currentOrgCode)) {
                            mdmTerminalImportVo.appendErrorValidateMsg("组织编码" + currentOrgCode + "不存在;");
                        }
                    }
                } else {
                    if (CollectionUtils.isEmpty(dbOrgCodeSet) || !dbOrgCodeSet.contains(orgCode)) {
                        mdmTerminalImportVo.appendErrorValidateMsg("组织编码:" + orgCode + "不存在;");
                    }
                }
            }
        }
    }

    /**
     * 校验终端编码
     *
     * @param data list
     */
    protected void validTerminalCode(List<MdmTerminalImportVo> data,DefaultImportContext context) {
        String sid = context.getImportParamVo().getWebSocketClientId();
        websocketUtil.sendMsg(sid,"开始校验终端编码");
        Set<String> collect = data.stream().filter(item -> !StringUtils.isEmpty(item.getTerminalCode())).map(MdmTerminalImportVo::getTerminalCode).collect(Collectors.toSet());
        if (collect.isEmpty()) {
            return;
        }
        Set<String> materialCodeSet = new HashSet<>();
        List<String> list = new ArrayList<>(collect);
        List<List<String>> lists = Lists.partition(list,500);
        for(List<String> item:lists){
            LambdaQueryWrapper<MdmTerminalEntity> wrapper = Wrappers.<MdmTerminalEntity>lambdaQuery().select(MdmTerminalEntity::getTerminalCode);
            wrapper.in(MdmTerminalEntity::getTerminalCode, item);
            Set<String>  materialCodeSetTemp = mdmTerminalMapper.selectList(wrapper).stream()
                    .map(MdmTerminalEntity::getTerminalCode)
                    .collect(Collectors.toSet());
            materialCodeSet.addAll(materialCodeSetTemp);
        }


        Map<String, Integer> codeIndexMap = new HashMap<>(16);
        for (MdmTerminalImportVo item : data) {
            if (StringUtils.isEmpty(item.getTerminalCode())) {
                continue;
            }
            if (materialCodeSet.contains(item.getTerminalCode())) {
                item.appendErrorValidateMsg("终端编码已存在；");
            } else if (codeIndexMap.containsKey(item.getTerminalCode())) {
                item.appendErrorValidateMsg("终端编码与第" + item.getTerminalCode() + "行重复；");
            } else {
                codeIndexMap.put(item.getTerminalCode(), item.getRowIndex());
            }
        }
    }

    /**
     * 校验终端名称和终端地址
     *
     * @param data list
     */
    protected void validTerminalName(List<MdmTerminalImportVo> data,DefaultImportContext context) {
        String sid = context.getImportParamVo().getWebSocketClientId();
        websocketUtil.sendMsg(sid,"校验终端名称和终端地址");
        for (MdmTerminalImportVo mdmTerminalVo : data) {
            if (StringUtils.isEmpty(mdmTerminalVo.getTerminalName())) {
                mdmTerminalVo.appendErrorValidateMsg("终端名称不能为空;");
            }
            if (StringUtils.isEmpty(mdmTerminalVo.getTerminalAddress())){
                mdmTerminalVo.appendErrorValidateMsg("终端地址不能为空;");
            }
        }
    }
}
