package com.biz.crm.excel.component.validator.mdm.user;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.mdm.user.MdmTerminalUserImportVo;
import com.biz.crm.mdm.role.entity.MdmRoleEntity;
import com.biz.crm.mdm.role.mapper.MdmRoleMapper;
import com.biz.crm.mdm.terminal.entity.MdmTerminalEntity;
import com.biz.crm.mdm.terminal.mapper.MdmTerminalMapper;
import com.biz.crm.mdm.user.entity.MdmUserEntity;
import com.biz.crm.mdm.user.mapper.MdmUserMapper;
import com.biz.crm.mdm.user.mapper.MdmUserRelTerminalMapper;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;


@Slf4j
@Component("mdmTerminalUserImportValidator")
public class MdmTerminalUserImportValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<MdmTerminalMapper, MdmTerminalEntity, MdmTerminalUserImportVo> implements ExcelImportValidator<MdmTerminalUserImportVo> {

    @Resource
    private MdmUserMapper mdmUserMapper;
    @Resource
    private MdmRoleMapper mdmRoleMapper;
    @Resource
    private MdmTerminalMapper mdmTerminalMapper;


    private final String PHONE_REGEX = "[1][\\d]{10}";

    /**
     * 导入校验方法
     *
     * @param data
     * @return
     */
    @Override
    public void validate(List<MdmTerminalUserImportVo> data, DefaultImportContext context) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }
        verify(data, context);
    }

    protected void verify(List<MdmTerminalUserImportVo> originList, DefaultImportContext context) {

        //登录名已存在校验map
        Map<String, MdmUserEntity> userNameCheckMap = new HashMap<>(16);
        Set<String> paramUserNameSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getUserName())).map(MdmTerminalUserImportVo::getUserName).collect(Collectors.toSet());
        if (!paramUserNameSet.isEmpty()) {
            QueryWrapper<MdmUserEntity> userNameQuery = new QueryWrapper<>();
            userNameQuery.in("user_name", paramUserNameSet);
            userNameQuery.select("user_name", "del_flag");
            List<MdmUserEntity> userNameQueryEntityList = mdmUserMapper.selectList(userNameQuery);
            if (CollectionUtil.listNotEmptyNotSizeZero(userNameQueryEntityList)) {
                userNameCheckMap = userNameQueryEntityList.stream().collect(Collectors.toMap(MdmUserEntity::getUserName, v -> v));
            }
        }

        //查询已存在的用户电话号码
        Set<String> existPhoneSet = new HashSet<>(16);
        Set<String> paramPhoneSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getEmail())).map(MdmTerminalUserImportVo::getUserPhone).collect(Collectors.toSet());
        if (!paramPhoneSet.isEmpty()) {
            List<String> list = new ArrayList<>(paramPhoneSet);
            List<List<String>> lists = Lists.partition(list, 500);
            for (List<String> item : lists) {
                QueryWrapper<MdmUserEntity> queryWrapper = new QueryWrapper<>();
                queryWrapper.in("user_phone", item);
                queryWrapper.select("user_phone");
                List<MdmUserEntity> entityList = mdmUserMapper.selectList(queryWrapper);
                if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                    existPhoneSet.addAll(entityList.stream().map(MdmUserEntity::getUserPhone).collect(Collectors.toList()));
                }
            }
        }

        //查询已存在的用户邮箱
        Set<String> existEmailSet = new HashSet<>(16);
        Set<String> paramEmailSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getEmail())).map(MdmTerminalUserImportVo::getEmail).collect(Collectors.toSet());
        if (!paramEmailSet.isEmpty()) {
            List<String> list = new ArrayList<>(paramEmailSet);
            List<List<String>> lists = Lists.partition(list, 500);
            for (List<String> item : lists) {
                QueryWrapper<MdmUserEntity> queryWrapper = new QueryWrapper<>();
                queryWrapper.in("email", item);
                queryWrapper.select("email");
                List<MdmUserEntity> entityList = mdmUserMapper.selectList(queryWrapper);
                if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
                    existEmailSet.addAll(entityList.stream().map(MdmUserEntity::getEmail).collect(Collectors.toList()));
                }
            }
        }

        //性别字典 名称：编码
        Map<String, String> genderValueCodeMap = DictUtil.dictRevertMap(DictConstant.GENDER);

        //权限角色校验map
        Map<String, String> roleCodeNameMap = new HashMap<>(16);
        Set<String> paramRoleCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getRoleCode())).map(MdmTerminalUserImportVo::getRoleCode).collect(Collectors.toSet());
        if (!paramRoleCodeSet.isEmpty()) {
            QueryWrapper<MdmRoleEntity> roleQuery = new QueryWrapper<>();
            roleQuery.in("role_code", paramRoleCodeSet);
            List<MdmRoleEntity> roleCodeQueryEntityList = mdmRoleMapper.selectList(roleQuery);
            if (CollectionUtil.listNotEmptyNotSizeZero(roleCodeQueryEntityList)) {
                roleCodeNameMap.putAll(roleCodeQueryEntityList.stream().collect(Collectors.toMap(MdmRoleEntity::getRoleCode, MdmRoleEntity::getRoleName)));
            }
        }

        //终端信息校验
        Set<String> terminalCodeSet = new HashSet<>(16);
        Set<String> paramTerminalCodeSet = originList.stream().filter(item -> StringUtils.isNotEmpty(item.getTerminalCodeName())).map(MdmTerminalUserImportVo::getTerminalCodeName).collect(Collectors.toSet());
        if (!paramTerminalCodeSet.isEmpty()){
            List<String> list = new ArrayList<>(paramTerminalCodeSet);
            List<List<String>> lists = Lists.partition(list, 500);
            for (List<String> strings : lists) {
                QueryWrapper<MdmTerminalEntity> queryWrapper = new QueryWrapper<>();
                queryWrapper.in("terminal_code",strings);
                queryWrapper.select("terminal_code");
                List<MdmTerminalEntity> mdmTerminalEntityList = mdmTerminalMapper.selectList(queryWrapper);
                if (CollectionUtil.listNotEmptyNotSizeZero(mdmTerminalEntityList)){
                    terminalCodeSet.addAll(mdmTerminalEntityList.stream().map(MdmTerminalEntity::getTerminalCode).collect(Collectors.toList()));
                }
            }
        }

        Map<String, Integer> userNameIndexMap = new HashMap<>(16);
        Map<String, Integer> phoneIndexMap = new HashMap<>(16);
        Map<String, Integer> emailIndexMap = new HashMap<>(16);
        for (MdmTerminalUserImportVo item :
                originList) {

            //校验账号
            if (StringUtils.isEmpty(item.getUserName())) {
                item.appendErrorValidateMsg("缺失账号；");
            } else {
                if (userNameIndexMap.containsKey(item.getUserName())) {
                    item.appendErrorValidateMsg("账号与第" + userNameIndexMap.get(item.getUserName()) + "行重复；");
                } else {
                    userNameIndexMap.put(item.getUserName(), item.getRowIndex());
                }
                if (userNameCheckMap.containsKey(item.getUserName())) {
                    MdmUserEntity mdmUserEntity = userNameCheckMap.get(item.getUserName());
                    if (CrmDelFlagEnum.NORMAL.getCode().equals(mdmUserEntity.getDelFlag())) {
                        item.appendErrorValidateMsg("账号已存在；");
                    } else {
                        item.appendErrorValidateMsg("账号已存在（逻辑删除）；");
                    }
                }
            }

            //校验姓名
            if (StringUtils.isEmpty(item.getFullName())) {
                item.appendErrorValidateMsg("缺失用户名；");
            }

            //校验性别
            if (StringUtils.isNotEmpty(item.getGenderName())) {
                if (!genderValueCodeMap.containsKey(item.getGenderName())) {
                    item.appendErrorValidateMsg("性别无效；");
                } else {
                    item.setGender(genderValueCodeMap.get(item.getGenderName()));
                }
            }

            //校验密码
            if (StringUtils.isEmpty(item.getUserPassword())) {
                item.appendErrorValidateMsg("缺失密码；");
            }

            //电话校验
            if (StringUtils.isNotEmpty(item.getUserPhone())) {
                if (!item.getUserPhone().matches(PHONE_REGEX)) {
                    item.appendErrorValidateMsg("联系电话格式不正确；");
                } else {
                    if (phoneIndexMap.containsKey(item.getUserPhone())) {
                        item.appendErrorValidateMsg("联系电话[" + item.getUserPhone() + "]在第" + phoneIndexMap.get(item.getUserPhone()) + "行已存在；");
                    } else if (existPhoneSet.contains(item.getUserPhone())) {
                        item.appendErrorValidateMsg("联系电话[" + item.getUserPhone() + "]已存在；");
                    } else {
                        phoneIndexMap.put(item.getUserPhone(), item.getRowIndex());
                    }
                }
            }

            //邮箱
            if (StringUtils.isNotEmpty(item.getEmail())) {
                if ((item.getEmail().length() - item.getEmail().replace("@", "").length()) != 1) {
                    item.appendErrorValidateMsg("邮箱格式不正确；");
                } else {
                    if (emailIndexMap.containsKey(item.getEmail())) {
                        item.appendErrorValidateMsg("邮箱[" + item.getEmail() + "]在第" + emailIndexMap.get(item.getEmail()) + "行已存在；");
                    } else if (existEmailSet.contains(item.getEmail())) {
                        item.appendErrorValidateMsg("邮箱[" + item.getEmail() + "]已存在；");
                    } else {
                        emailIndexMap.put(item.getEmail(), item.getRowIndex());
                    }
                }
            }

            //生效时间
            if (StringUtils.isEmpty(item.getStartTime())) {
                item.appendErrorValidateMsg("缺失生效时间；");
            } else {
                if (!verifyDateFormat(item.getStartTime(), DateUtil.DEFAULT_DAY_PATTERN)) {
                    item.appendErrorValidateMsg("生效时间格式不合法；");
                }
            }

            //失效时间
            if (StringUtils.isEmpty(item.getEndTime())) {
                item.appendErrorValidateMsg("缺失失效时间；");
            } else {
                if (!verifyDateFormat(item.getEndTime(), DateUtil.DEFAULT_DAY_PATTERN)) {
                    item.appendErrorValidateMsg("失效时间格式不合法；");
                }
            }

            //角色编码
            if (StringUtils.isNotEmpty(item.getRoleCode())) {
                if (!roleCodeNameMap.containsKey(item.getRoleCode())) {
                    item.appendErrorValidateMsg("权限角色编码无效；");
                }
            }
            //终端信息
            if(StringUtils.isNotEmpty(item.getTerminalCodeName())){
                if (!terminalCodeSet.contains(item.getTerminalCodeName())){
                    item.appendErrorValidateMsg("关联终端不存在；");
                }
            }
        }
    }

    private boolean verifyDateFormat(String dataStr, String formatStr) {
        boolean flag = false;
        if (StringUtils.isNotEmpty(dataStr) && StringUtils.isNotEmpty(formatStr)) {
            try {
                SimpleDateFormat sdf = new SimpleDateFormat(formatStr);
                Date parse = sdf.parse(dataStr);
                String dateFormatStr = sdf.format(parse);
                if (dataStr.equals(dateFormatStr)) {
                    flag = true;
                }
            } catch (Exception e) {

            }
        }
        return flag;
    }
}
