package com.biz.crm.excel.component.validator.sfa;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.excel.component.validator.AbstractExcelImportValidator;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.sfa.SfaAchievementRuleImportVo;
import com.biz.crm.mdm.user.MdmUserFeign;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.sfa.test.entity.SfaAchievementRuleEntity;
import com.biz.crm.sfa.test.entity.SfaIndexEntity;
import com.biz.crm.sfa.test.mapper.SfaAchievementRuleMapper;
import com.biz.crm.sfa.test.mapper.SfaIndexMapper;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 *  @author: yangshan
 *  @Date: 2021-3-3 10:56:18
 */
@Slf4j
@Component("sfaAchievementRuleValidator")
public class SfaAchievementRuleValidator<M extends BaseMapper<T>, T> extends AbstractExcelImportValidator<SfaAchievementRuleMapper, SfaAchievementRuleEntity, SfaAchievementRuleImportVo> implements ExcelImportValidator<SfaAchievementRuleImportVo> {

    @Resource
    private SfaIndexMapper indexMapper;
    @Resource
    private SfaAchievementRuleMapper sfaAchievementRuleMapper;
    @Resource
    private MdmUserFeign mdmUserFeign;
    private static List<SfaIndexEntity> sfaIndexEntityList;
    /**
     * 导入校验方法
     *
     * @param data
     * @return
     */
    @Override
    public void validate(List<SfaAchievementRuleImportVo> data, DefaultImportContext context) {
        if(CollectionUtils.isEmpty(data)){
            return;
        }

        if (CollectionUtil.listEmpty(sfaIndexEntityList)){
            this.findAllIndex();
        }

        HashSet<SfaAchievementRuleImportVo> voSet = new HashSet<>();
        data.forEach (v -> {
            StringBuilder msg = new StringBuilder();
            //对象校验
            this.doCheckObjectCodeAndName(v, msg);

            //绩效类型校验
            this.doCheckAchievementType(v, msg);

            //月度
            this.doCheckAchievementTypeMM(v, msg);

            //旬度
            this.doCheckAchievementTypeTD(v, msg);

            //自由时间
            this.doCheckAchievementTypeFT(v, msg);

            //权重
            if (StringUtils.isBlank(v.getWeight())){
                v.setWeight(BigDecimal.ZERO.toString());
            }
            this.doCheckWeight(v.getWeight(), msg);

            //目标数量
            this.doCheckTargetNum(v.getTargetNum(), msg);

            //指标校验
            this.doCheckIndex(v, msg);

            //单位校验
            if(StringUtils.isBlank(v.getUnit())){
                msg.append("绩效单位不能为空；");
            }

            if (StringUtils.isEmpty(v.getObjectCode())
                    || StringUtils.isEmpty(v.getAchievementDate())){
                msg.append("绩效时间设置失败；");
            }

            if (voSet.contains(v)){
                msg.append("数据重复；");
            }
            voSet.add(v);
            if (StringUtils.isNotBlank(msg.toString())){
                this.errorMsgAndSendWebsocket(v,msg.toString(), context);
            }
        });

        this.distinctData(data, voSet, context);
    }

    /**
     * 旬度 校验，设置值
     * @param v
     * @param msg
     */
    private void doCheckAchievementTypeTD(SfaAchievementRuleImportVo v, StringBuilder msg) {
        if (WorkSignEnum.achievementTimeType.TD.getDesc().equals(v.getAchievementType())){
            this.doCheckMouth(v, msg);
            if(StringUtils.isBlank(v.getAchievementDateTenDays())){
                msg.append("绩效旬度不能为空；");
            }
            if (!WorkSignEnum.achievementTimeType.XX.getDesc().equals(v.getAchievementDateTenDays())
                    && !WorkSignEnum.achievementTimeType.SX.getDesc().equals(v.getAchievementDateTenDays())){
                msg.append("绩效旬度格式不正确；");
            }
            StringBuilder achievementDate = new StringBuilder().append(v.getAchievementDateYear())
                    .append("年")
                    .append(v.getAchievementDateMonth())
                    .append("月");
            if (v.getAchievementDateTenDays().equals(WorkSignEnum.achievementTimeType.SX.getDesc())){
                achievementDate.append(WorkSignEnum.achievementTimeType.SX.getDesc());
            }
            if (v.getAchievementDateTenDays().equals(WorkSignEnum.achievementTimeType.XX.getDesc())){
                achievementDate.append(WorkSignEnum.achievementTimeType.XX.getDesc());
            }
            v.setAchievementType(WorkSignEnum.achievementTimeType.TD.getVal());
            v.setAchievementDate(achievementDate.toString());
            this.getDayOfDate(v,msg);
        }
    }

    /**
     * 自由时间 校验，设置值
     * @param v
     * @param msg
     */
    private void doCheckAchievementTypeFT(SfaAchievementRuleImportVo v, StringBuilder msg) {
        if (WorkSignEnum.achievementTimeType.FT.getDesc().equals(v.getAchievementType())){
            if(StringUtils.isBlank(v.getBeginDate())){
                msg.append("有效开始时间不能为空；");
            }
            if(StringUtils.isBlank(v.getEndDate())){
                msg.append("有效结束时间不能为空；");
            }
            LocalDate beginDate = null;
            try {
                beginDate = LocalDate.parse(v.getBeginDate(), CrmDateUtils.yyyyMMdd);
            } catch (Exception e) {
                msg.append("有效开始时间格式错误；");
            }
            LocalDate endDate = null;
            try {
                endDate = LocalDate.parse(v.getEndDate(), CrmDateUtils.yyyyMMdd);
            } catch (Exception e) {
                msg.append("有效结束时间格式错误；");
            }
            if (!(beginDate.isBefore(endDate))) {
                msg.append("有效开始时间不能大于有效结束时间；");
            }
            v.setAchievementType(WorkSignEnum.achievementTimeType.FT.getVal());
            v.setAchievementDate(v.getBeginDate() + "-" + v.getEndDate());
        }
    }

    /**
     * 月度 校验，设置值
     * @param v
     * @param msg
     */
    private void doCheckAchievementTypeMM(SfaAchievementRuleImportVo v, StringBuilder msg) {
        if (WorkSignEnum.achievementTimeType.MM.getDesc().equals(v.getAchievementType())){
            this.doCheckMouth(v, msg);
            v.setAchievementType(WorkSignEnum.achievementTimeType.MM.getVal());
            StringBuilder achievementDate = new StringBuilder().append(v.getAchievementDateYear())
                    .append("年")
                    .append(v.getAchievementDateMonth())
                    .append("月");
            v.setAchievementDate(achievementDate.toString());
            this.getDayOfDate(v, msg);
        }
    }

    /**
     * 去重分组，校验指标权重
     * @param data
     * @param voSet
     * @param context
     */
    public void distinctData(List<SfaAchievementRuleImportVo> data, HashSet<SfaAchievementRuleImportVo> voSet, DefaultImportContext context){
        //去重
        Map<String, List<SfaAchievementRuleImportVo>> dataList = voSet.stream().distinct().collect(Collectors.groupingBy(o -> o.getObjectPosCode() + "-" + o.getAchievementDate()));
        dataList.keySet().forEach(s -> {
            List<SfaAchievementRuleImportVo> vos = dataList.get(s);
            BigDecimal weightTotal = vos.stream().map(m -> new BigDecimal(m.getWeight())).reduce(BigDecimal.ZERO, BigDecimal::add);
            boolean one = weightTotal.compareTo(BigDecimal.ONE) == 0;
            boolean zero = weightTotal.compareTo(BigDecimal.ZERO) == 0;
            //绩效权重不为1和0
            if (!one && !zero){
                dataList.get(s).forEach(o -> {
                    if (data.contains(o)){
                        this.errorMsgAndSendWebsocket(o,"数据权重不合法", context);
                    }
                });
            }
            String[] split = s.split("-");
            this.sfaAchievementRuleMapper.delete(Wrappers.lambdaQuery(SfaAchievementRuleEntity.class)
                    .eq(SfaAchievementRuleEntity::getObjectPosCode, split[0])
                    .eq(SfaAchievementRuleEntity::getAchievementDate, split[1]));
        });
    }

    /**
     * 设置开始时间和结束时间
     * @param reqVo
     * @return
     */
    public SfaAchievementRuleImportVo getDayOfDate(SfaAchievementRuleImportVo reqVo, StringBuilder msg) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar cale = Calendar.getInstance();
        // 获取该月第一天和最后一天
        String firstDay, lastDay;
        String s = reqVo.getAchievementDateYear() + "-" + reqVo.getAchievementDateMonth() + "-15";
        Date date = null;
        try {
            date = format.parse(s);
        } catch (Exception e) {
            msg.append("时间格式错误；");
            return new SfaAchievementRuleImportVo();
        }
        cale.setTime(date);
        // 获取月的第一天
        cale.add(Calendar.MONTH, 0);
        cale.set(Calendar.DAY_OF_MONTH, 1);
        firstDay = format.format(cale.getTime());
        // 获取月的最后一天
        cale.add(Calendar.MONTH, 1);
        cale.set(Calendar.DAY_OF_MONTH, 0);
        lastDay = format.format(cale.getTime());

        if (com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getAchievementDateTenDays())){
            if (WorkSignEnum.achievementTimeType.SX.getDesc().equals(reqVo.getAchievementDateTenDays())){
                lastDay = reqVo.getAchievementDateYear() + "-" + reqVo.getAchievementDateMonth() + "-15";
            }else {
                firstDay = reqVo.getAchievementDateYear() + "-" + reqVo.getAchievementDateMonth() + "-15";
            }
        }
        reqVo.setBeginDate(firstDay);
        reqVo.setEndDate(lastDay);
        return reqVo;
    }

    /**
     * 权重校验
     * @param weight
     */
    public void doCheckWeight(String weight, StringBuilder msg){
        try {
            BigDecimal w = new BigDecimal(weight);
            if (w.compareTo(BigDecimal.ZERO) == -1 || w.compareTo(BigDecimal.ONE) == 1){
                msg.append("绩效权重不合法；");
            }
        }catch (Exception e){
            msg.append("绩效权重不合法；");
        }
    }

    /**
     * 目标数量校验
     * @param targetNum
     */
    public void doCheckTargetNum(String targetNum, StringBuilder msg){
        if(!StringUtils.isNumeric(targetNum)){
            msg.append("绩效目标数量不合法；");
        }
        try {
            int t = Integer.parseInt(targetNum);
            if (t < YesNoEnum.YesNoCodeNumberEnum.NO.getCode()){
                msg.append("绩效目标数量不合法；");
            }
        }catch (Exception e){
            msg.append("绩效目标数量不合法；");
        }
    }

    /**
     * 指标校验
     * @param vo
     */
    public void doCheckIndex(SfaAchievementRuleImportVo vo, StringBuilder msg){
        if (StringUtils.isEmpty(vo.getIndexCode())){
            msg.append("指标编码为空；");
        }
        if(StringUtils.isEmpty(vo.getIndexName())){
            msg.append("指标名称为空；");
        }
        if(StringUtils.isNotEmpty(vo.getIndexCode()) && StringUtils.isNotEmpty(vo.getIndexName())){
            Map<String, SfaIndexEntity> indexMap = sfaIndexEntityList.stream().collect(Collectors.toMap(s -> s.getIndexCode() + "-" + s.getIndexName(), Function.identity(), (k1, k2) -> k2));
            if (!indexMap.containsKey(vo.getIndexCode() + "-" + vo.getIndexName())){
                msg.append("指标不存在；");
            }
        }
    }

    public void findAllIndex(){
        sfaIndexEntityList = this.indexMapper.selectList(null);
    }

    /**
     * 校验user的username、realName、posName是否匹配
     * 如果是组织，校验组织的code和name是否匹配
     * @param vo
     * @param msg
     * @return
     */
    private void doCheckObjectCodeAndName(SfaAchievementRuleImportVo vo, StringBuilder msg){
        if (StringUtils.isEmpty(vo.getObjectCode())
                || StringUtils.isEmpty(vo.getObjectName())){
            msg.append("对象编码或对象名称为空；");
        }

        MdmOrgRespVo orgByCode = OrgUtil.getOrgByCode(vo.getObjectCode());
        MdmUserRespVo mdmUserRespVo = ApiResultUtil.objResult(mdmUserFeign.query(new MdmUserReqVo().setUserName(vo.getObjectCode())));
        if (ObjectUtils.isEmpty(orgByCode) && ObjectUtils.isEmpty(mdmUserRespVo)){
            msg.append("组织或用户不存在；");
            return;
        }
        //组织
        if (!ObjectUtils.isEmpty(orgByCode)){
            if (!orgByCode.getOrgName().equals(vo.getObjectName())){
                msg.append("组织编码与组织名称不匹配；");
            }
            if (StringUtils.isNotBlank(vo.getObjectPosName())){
                msg.append("导入为组织时，职位名称为空；");
            }
            return;
        }
        //user
        if (!ObjectUtils.isEmpty(mdmUserRespVo)){
            if (!mdmUserRespVo.getFullName().equals(vo.getObjectName())){
                msg.append("对象编码与对象名称不匹配；");
            }

            //职位匹配
            if (StringUtils.isEmpty(vo.getObjectPosCode())){
                msg.append("职位编码不能为空；");
            }

            if (StringUtils.isEmpty(vo.getObjectPosName())){
                msg.append("职位名称不能为空；");
            }

            MdmPositionRespVo position = PositionUtil.getPositionByCode(vo.getObjectPosCode());
            if (ObjectUtils.isEmpty(position)){
                msg.append("未找到该职位；");
            }

            if (StringUtils.isNotEmpty(vo.getObjectPosName()) && !ObjectUtils.isEmpty(position)){
                if (!vo.getObjectPosName().equals(position.getPositionName())){
                    msg.append("职位编码为职位名称不匹配；");
                }
            }

            //职位与人匹配
            List<MdmPositionRespVo> allPositionByUsername = PositionUtil.getAllPositionByUsername(vo.getObjectCode());
            if (CollectionUtil.listNotEmpty(allPositionByUsername)){
                List<String> allPosName = allPositionByUsername.stream().map(MdmPositionRespVo::getPositionName).collect(Collectors.toList());
                if (!allPosName.contains(vo.getObjectPosName())){
                    msg.append("职位与用户不匹配；");
                }
            }
        }
    }

    /**
     * 月度，参数校验
     * @param v
     */
    public void doCheckMouth(SfaAchievementRuleImportVo v, StringBuilder msg){
        if (!(StringUtils.isNumeric(v.getAchievementDateYear()) && StringUtils.isNumeric(v.getAchievementDateMonth()))){
            msg.append("绩效年月格式不正确；");
        }
        if (v.getAchievementDateYear().length() != 4){
            msg.append("绩效年份格式不正确；");
        }
        if (v.getAchievementDateMonth().length() == 1){
            v.setAchievementDateMonth(YesNoEnum.yesNoEnum.ZERO.getValue() + v.getAchievementDateMonth());
        }
        if (v.getAchievementDateMonth().length() != 2){
            msg.append("绩效月份格式不正确；");
        }
        int month = Integer.parseInt(v.getAchievementDateMonth());
        if (month > 12 || month < 0){
            msg.append("绩效月份格式不正确；");
        }
    }

    /**
     * 绩效类型校验
     */
    public void doCheckAchievementType(SfaAchievementRuleImportVo vo, StringBuilder msg){
        if(!vo.getAchievementType().equals(WorkSignEnum.achievementTimeType.MM.getDesc())
                && !vo.getAchievementType().equals(WorkSignEnum.achievementTimeType.TD.getDesc())
                && !vo.getAchievementType().equals(WorkSignEnum.achievementTimeType.FT.getDesc())){
            msg.append("绩效类型不正确；");
        }
    }
}
