package com.biz.crm.excel.component.validator.tpm.actdetail;

import com.alibaba.fastjson.JSON;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.common.DictItemVo;
import com.biz.crm.eunm.tpm.ActTypeEnum;
import com.biz.crm.excel.component.helper.MdmPropertyCheckHelper;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.util.DefaultImportContext;
import com.biz.crm.excel.vo.tpm.actdetail.TpmActDetailImportVo;
import com.biz.crm.mdm.material.MdmMaterialFeign;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.material.MdmMaterialRespVo;
import com.biz.crm.nebular.mdm.material.MdmMaterialUnitRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.tpm.act.req.TpmActDetailReqVo;
import com.biz.crm.nebular.tpm.costtypefine.resp.TpmCostTypeFineRespVo;
import com.biz.crm.tpm.act.TpmActFeign;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.UserRedis;
import com.biz.crm.util.UserUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yb
 * @version 1.0
 * @date 2021/12/9 9:49
 * @description 活动明细一些公用校验
 */
@Component("tpmActDetailCommonValidator")
public class TpmActDetailCommonValidator implements ExcelImportValidator<TpmActDetailImportVo> {

    @Resource
    private TpmActFeign tpmActFeign;

    @Resource
    private MdmPropertyCheckHelper mdmPropertyCheckHelper;

    @Resource
    private MdmMaterialFeign materialFeign;


    @Override
    public void validate(List<TpmActDetailImportVo> list, DefaultImportContext context) {
        if(CollectionUtils.isEmpty(list)){
            return;
        }
        String bizParams = context.getImportParamVo().getBizParams();
        String [] params = bizParams.split(",");
        String actType = params[0];
        String categoriesCode = params[1];
        TpmActDetailReqVo tpmActDetailReqVo = new TpmActDetailReqVo();
        tpmActDetailReqVo.setCategoriesCode(categoriesCode);
        tpmActDetailReqVo.setActType(actType);
        //查询提交的大类包含的活动细类信息
        Map<String ,TpmCostTypeFineRespVo> fineRespVoMap = ApiResultUtil.objResult(tpmActFeign.getCategoryFinesByCategoryCode(tpmActDetailReqVo), true)
                .stream()
                .collect(Collectors.toMap(TpmCostTypeFineRespVo::getFineCode, tpmCostTypeFineRespVo -> tpmCostTypeFineRespVo));
        UserRedis userRedis = UserUtils.getUser();
        if(ActTypeEnum.STABLE_CHARGE.getCode().equals(actType)){
            list.forEach(data -> {
                checkActCostTypeFine(data, fineRespVoMap);
                mdmPropertyCheckHelper.checkOrg(data, userRedis.getOrgcode(), true);
                checkCustomer(data, mdmPropertyCheckHelper.getCustomerMapByOrgCodes(Lists.newArrayList(data.getOrgCode())), "所填客户编码不是所填组织关联客户;");
                checkApplyAmount(data);
                checkForecastSalesAmount(data);
            });
        }else if(ActTypeEnum.PRACTICALITY_CHARGE.getCode().equals(actType)){
            list.forEach(data -> {
                checkActCostTypeFine(data, fineRespVoMap);
                checkCustomer(data, mdmPropertyCheckHelper.getCustomerMapByOrgCodes(Lists.newArrayList(data.getOrgCode())), "所填客户编码不是当前组织关联客户;");
                checkApplyAmount(data);
                checkMaterial(data);
            });
        }else if(ActTypeEnum.DEPARTMENT_CHARGE.getCode().equals(actType)){
            list.forEach(data -> {
                checkActCostTypeFine(data, fineRespVoMap);
                mdmPropertyCheckHelper.checkOrg(data, userRedis.getOrgcode(), true);
                checkApplyAmount(data);
                checkForecastSalesAmount(data);
            });
        }else {
            list.forEach(data -> {
                checkActCostTypeFine(data, fineRespVoMap);
                mdmPropertyCheckHelper.checkOrg(data, userRedis.getOrgcode(), true);
                checkApplyAmount(data);
                checkMaterial(data);
            });
        }
    }

    /**
     * 检查活动细类
     * @param data 数据源
     * @param fineRespVoMap 活动细类信息
     */
    private void checkActCostTypeFine(TpmActDetailImportVo data,  Map<String ,TpmCostTypeFineRespVo> fineRespVoMap){
      if(StringUtils.isEmpty(data.getFineCode())){
          data.appendErrorValidateMsg("活动细类编码不能为空;");
      }else {
          TpmCostTypeFineRespVo fineRespVo;
          if((fineRespVo = fineRespVoMap.get(data.getFineCode())) == null){
              data.appendErrorValidateMsg("所填活动细类编码与活动大类未进行关联;");
          }else {
              checkPayType(data, fineRespVo.getPayTypes());
              data.setFineName(fineRespVo.getFineName());
          }
      }
    }

    /**
     * 检查支付方式
     * @param data 数据源
     * @param payTypes 活动细类包含支付方式
     */
    private void checkPayType(TpmActDetailImportVo data, List<DictItemVo> payTypes){
        if(StringUtils.isEmpty(data.getPayTypeName())){
            data.appendErrorValidateMsg("支付方式不能为空;");
        }else {
            boolean has = false;
            for (DictItemVo payType : payTypes) {
                if(payType.getDictKey().equals(data.getPayTypeName())){
                    data.setPayType(payType.getDictValue());
                    has = true;
                    break;
                }
            }
            data.setPayTypes(payTypes);
            if(!has){
                data.appendErrorValidateMsg("所填支付方式非该细类指定的支付方式;");
            }
        }
    }

    /**
     * 检查客户信息
     * @param data 数据源
     * @param customerMsgRespVoMap 客户信息
     * @param msg 错误信息
     */
    private void checkCustomer(TpmActDetailImportVo data, Map<String, MdmCustomerMsgRespVo> customerMsgRespVoMap, String msg){
        if(StringUtils.isEmpty(data.getCustomerCode())){
            data.appendErrorValidateMsg("客户编码不能为空;");
        }else {
            MdmCustomerMsgRespVo customerMsgRespVo;
            if((customerMsgRespVo = customerMsgRespVoMap.get(data.getCustomerCode())) == null){
                data.appendErrorValidateMsg(msg);
            }else {
                data.setCustomerName(customerMsgRespVo.getCustomerName());
                mdmPropertyCheckHelper.checkTerminal(data, false);
            }
        }
    }

    /**
     * 检查物料信息以及所填的物料单位信息,数量,费用申请金额等
     * @param data 数据源
     */
    private void checkMaterial(TpmActDetailImportVo data){
        if(StringUtils.isEmpty(data.getMaterialCode())){
            data.appendErrorValidateMsg("物料编码不能为空;");
        }else {
            MdmMaterialRespVo materialRespVo = ApiResultUtil.objResult(materialFeign.detail(null, data.getMaterialCode()));
            if(materialRespVo == null){
                data.appendErrorValidateMsg("未找到物料信息;");
            }else {
                BigDecimal price = null;
                //设置一些物料属性包括物料名称,物料的单位集合
                data.setMaterialName(materialRespVo.getMaterialName());
                if(data.getMaterialUnitName().equals(materialRespVo.getBaseUnitName())){
                    price = new BigDecimal(materialRespVo.getCostPrice());
                    data.setMaterialUnit(materialRespVo.getBaseUnit());
                }else if(data.getMaterialUnitName().equals(materialRespVo.getSaleUnitName())){
                    price = new BigDecimal(materialRespVo.getCostPrice()).multiply(new BigDecimal(materialRespVo.getUnitConversion()));
                    if(price.compareTo(data.getMaterialPrice()) != 0){
                        data.appendErrorValidateMsg("所填单价不正确;");
                    }
                    data.setMaterialUnit(materialRespVo.getSaleUnit());
                }else {
                    data.appendErrorValidateMsg("未找到物料匹配单位;");
                }
                if(data.getNum() == null){
                    data.appendErrorValidateMsg("数量不能为空;");
                }else if(data.getNum() <= 0){
                    data.appendErrorValidateMsg("数量不能小于0;");
                }else {
                    if(price != null){
                        checkApplyAmountCompare(data, price.multiply(new BigDecimal(data.getNum())));
                    }
                }
            }
        }
    }

    /**
     * 检查费用申请金额
     * @param data 数据源
     * @param compare 数量乘于物料单价
     */
    private void checkApplyAmountCompare(TpmActDetailImportVo data, BigDecimal compare){
        if (data.getApplyAmount() != null && data.getApplyAmount().compareTo(compare) != 0){
            data.appendErrorValidateMsg("费用申请金额应等于数量乘于物料单价;");
        }
    }

    /**
     * 检查费用申请金额
     * @param data 数据源
     */
    private void checkApplyAmount(TpmActDetailImportVo data){
        if(data.getApplyAmount() == null){
            data.appendErrorValidateMsg("费用申请金额不能为空;");
        }else if(data.getApplyAmount().compareTo(BigDecimal.ZERO) <= 0){
            data.appendErrorValidateMsg("费用申请金额不能小于0;");
        }
    }

    /**
     * 检查预估销售额
     * @param data 数据源
     */
    private void checkForecastSalesAmount(TpmActDetailImportVo data){
        if(data.getForecastSalesAmount() == null){
            data.appendErrorValidateMsg("预估销售额不能为空;");
        }else if(data.getForecastSalesAmount().compareTo(BigDecimal.ZERO) <= 0){
            data.appendErrorValidateMsg("预估销售额不能小于0;");
        }
    }
}
