package com.biz.crm.excel.controller;

import com.alibaba.excel.support.ExcelTypeEnum;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.upload.UploadEnum;
import com.biz.crm.excel.controller.req.ExcelImportParamVo;
import com.biz.crm.excel.controller.resp.DemoFileResp;
import com.biz.crm.excel.controller.resp.UnAsyncImportRespVo;
import com.biz.crm.excel.service.ExcelImport;
import com.biz.crm.excel.service.impl.DefaultExcelImport;
import com.biz.crm.excel.util.ExcelImportUtil;
import com.biz.crm.excel.util.MqMessageParam;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.upload.excel.req.ExcelExportReqVo;
import com.biz.crm.upload.excel.ExcelExportFeign;
import com.biz.crm.util.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.*;
import java.nio.charset.StandardCharsets;


/**
 * 导入
 *  @author: luoqi
 *  @Date: 2020-12-9 19:27
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
@RestController
@RequestMapping("excelImportController")
@Api(tags = "导入excel ")
public class ExcelImportController {

    @Resource(name = "defaultExcelImport")
    private ExcelImport defaultExcelImport;
    @Resource
    private ExcelExportFeign excelExportFeign;
    @Resource
    private RocketMQProducer rocketMQProducer;
    /**
     * 异步导入excel
     * @return
     */
    @ApiOperation(value = "导入excel")
    @PostMapping("import")
    @CrmLog
    public Result doImport(@ApiParam("导入配置编码") @RequestParam("configCode") String configCode
            , @ApiParam("是否测试模式") @RequestParam(name = "test", defaultValue = "false") Boolean test
            , @ApiParam("webSocket连接id") @RequestParam("webSocketClientId") String webSocketClientId, @RequestPart(value = "file",required = false)MultipartFile file) {

        Assert.notNull(file, "请先选择文件");
        //校验
        this.check(webSocketClientId, configCode, file);

        //解析导入配置
        ExcelImportParamVo importParamVo = defaultExcelImport.resolveConfig(configCode);
        if(StringUtils.isBlank(importParamVo.getBeanNameAsSaver())){
            throw new BusinessException("必须配置保存逻辑[beanNameAsSaver]，请前往数据字典检查Excel导入配置[" + configCode + "]");
        }
        importParamVo.setWebSocketClientId(webSocketClientId);
        //创建导入任务
        this.createImportTask(importParamVo, file);
        importParamVo.setTest(test);
        this.defaultExcelImport.doAsyncImport(importParamVo, UserUtils.getToken(), this.createTempFile(file));
        return Result.ok();
    }
    /**
     * 发送mq
     * @param msgBody
     */
    private void sendMQ(String msgBody, ExcelImportParamVo importParamVo){
        String webSocketClientId = importParamVo.getWebSocketClientId();
        if(StringUtils.isBlank(webSocketClientId)){
            if(log.isInfoEnabled()){
                log.info("导入任务[" + importParamVo.getTaskCode() + "]:未提交 websocket id，忽略 websocket 消息发送！");
            }
            return;
        }
        MqMessageParam mqMessageParam = new MqMessageParam();
        mqMessageParam.setClientId(webSocketClientId);
        mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate(msgBody));
        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
        mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.EXCEL_IMPORT_WEBSOCKET_MSG);
        mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(mqMessageParam));
        this.rocketMQProducer.convertAndSend(mqMessageBody);
    }
    /**
     * 同步导入excel(不会保存到数据库)
     * @return
     */
    @ApiOperation(value = "同步导入excel(不会保存到数据库)")
    @PostMapping("unAsyncImportUnSave")
    @CrmLog
    public Result<UnAsyncImportRespVo> unAsyncImportUnSave(@ApiParam("导入配置编码") @RequestParam("configCode") String configCode, @ApiParam("自定义参数") @RequestParam("bizParams") String bizParams, @RequestPart("file")MultipartFile file) {

        //校验
        AssertUtils.isNotNull(file, "请选择需要导入的文件");
        AssertUtils.isNotEmpty(configCode, "导入配置编码不能为空");
        //解析导入配置
        ExcelImportParamVo importParamVo = defaultExcelImport.resolveConfig(configCode);
        importParamVo.setBizParams(bizParams);
        //创建导入任务
        this.createImportTask(importParamVo, file);
        return Result.ok(this.defaultExcelImport.doUnAsyncImport(importParamVo, this.createTempFile(file)));
    }

    /**
     * 创建导入任务
     * @param importParamVo
     * @param file
     */
    private void createImportTask(ExcelImportParamVo importParamVo, MultipartFile file){
        ExcelExportReqVo excelExportReqVo = this.buildExcelExportReqVo(importParamVo, file.getOriginalFilename());
        //创建导入任务记录
        Result<ExcelExportReqVo> result = excelExportFeign.saveExcelFileBackId(excelExportReqVo);
        if(!ApiResultUtil.checkResult(result) || null == result.getResult()){
            throw new BusinessException("创建导入任务失败，请重试！");
        }
        excelExportReqVo = result.getResult();
        importParamVo.setTaskId(excelExportReqVo.getId());
        importParamVo.setTaskCode(excelExportReqVo.getFileCode());
        String msg = "创建导入任务[" + importParamVo.getTaskCode() + "]成功！";
        log.warn(msg);
        this.sendMQ(msg, importParamVo);
    }
    /**
     * 生成导入任务记录
     * @param importParamVo
     * @return
     */
    private ExcelExportReqVo buildExcelExportReqVo(ExcelImportParamVo importParamVo, String baseFileName){
        ExcelExportReqVo vo = new ExcelExportReqVo();
//        vo.setObjectName(baseData.getObjectName());
//        vo.setFilePath(baseData.getUrl());
        vo.setFileName(baseFileName);
        vo.setExcelFileName(baseFileName);
//        if(null != failData){
//            vo.setObjectNameAsImportFail(failData.getObjectName());
//            vo.setFilePathAsImportFail(failData.getUrl());
//            vo.setFileNameAsImportFail(failFileName);
//            vo.setExcelFileNameAsImportFail(failFileName);
//        }
        vo.setFileSource(importParamVo.getFileNameAsDemo());
        vo.setFileParam(importParamVo.getImportConfigCode());
        vo.setFileType(UploadEnum.fileType.IMPORT.getVal());
        vo.setFileStatus(UploadEnum.fileStatus.IMPORT_IN_EXECUTION.getVal());
        vo.setWebSocketId(importParamVo.getWebSocketClientId());
        return vo;
    }


    private void check(String webSocketClientId, String importConfigCode, MultipartFile file){
        AssertUtils.isNotNull(file, "请选择需要导入的文件");
        AssertUtils.isNotEmpty(webSocketClientId, "Websocket连接码不能为空");
        AssertUtils.isNotEmpty(importConfigCode, "导入配置编码不能为空");
    }
    /**
     * 创建临时文件
     * @param file
     * @return
     */
    private File createTempFile(MultipartFile file){
        if(null == file){
            throw new BusinessException("请上传导入文件！");
        }
        BufferedInputStream bis = null;
        BufferedOutputStream bos = null;
        try {
            File fileTemp = File.createTempFile(file.getName() + UUIDGenerator.generate(), ExcelTypeEnum.XLSX.getValue());
            bis = new BufferedInputStream(file.getInputStream());
            //把文件流转为文件，保存在临时目录
            bos = new BufferedOutputStream(new FileOutputStream(fileTemp));
            int len = 0;
            byte[] buf = new byte[20 * 1024];//缓冲区
            while((len=bis.read(buf)) != -1){
                bos.write(buf, 0, len);
            }
            bos.flush();
            return fileTemp;
        } catch (IOException e) {
            log.warn("解析文件失败，请重试！", e);
            throw new BusinessException("解析文件失败，请重试！", e);
        }finally{
            try {
                if(bos!=null) {
                    bos.close();
                }
                if(bis!=null) {
                    bis.close();
                }
            } catch (IOException e) {
                log.warn("文件处理异常！", e);
                throw new BusinessException("文件处理异常，请重试！", e);
            }
        }
    }




    @SneakyThrows
    @ApiOperation(value = "导入excel,下载示例文件")
    @GetMapping("demoFile")
    @CrmLog
    public ResponseEntity<InputStreamResource> demoFile(@ApiParam("导入对象的类路径") @RequestParam String configCode) {
        DemoFileResp demoFileResp = this.defaultExcelImport.demoFile(configCode);


        String fileName = demoFileResp.getFileNameAsDemo() + ExcelTypeEnum.XLSX.getValue();
        if (!StringUtils.isEmpty(fileName)) {
            fileName = new String(fileName.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1);
        }
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentDispositionFormData("attachment", fileName);
        httpHeaders.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        try {
            return new ResponseEntity<>(new InputStreamResource(new FileInputStream(demoFileResp.getFile())), httpHeaders, HttpStatus.OK);
        } catch (FileNotFoundException e) {
            throw new BusinessException("解析文件失败，请重试！", e);
        }
    }

    @SneakyThrows
    @ApiOperation(value = "导入excel,下载示例文件,可传token")
    @GetMapping("demoFileTest")
    @CrmLog
    public ResponseEntity<InputStreamResource> demoFileTest(@ApiParam("导入对象的类路径") @RequestParam String configCode
            , @ApiParam("token") @RequestParam String token) {
        if(StringUtils.isNotBlank(token)){
            UserUtils.setToken(token);
        }
        return this.demoFile(configCode);
    }



}
