package com.biz.crm.excel.service.impl;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.AbstractImportVo;
import com.biz.crm.eunm.upload.UploadEnum;
import com.biz.crm.excel.controller.req.ExcelImportParamVo;
import com.biz.crm.excel.controller.resp.DemoFileResp;
import com.biz.crm.excel.controller.resp.UnAsyncImportRespVo;
import com.biz.crm.excel.service.ExcelImport;
import com.biz.crm.excel.util.CurrentLimitableImportListener;
import com.biz.crm.excel.util.DefaultExcelImportListener;
import com.biz.crm.excel.util.ExcelImportUtil;
import com.biz.crm.excel.util.MqMessageParam;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.upload.excel.req.ExcelExportReqVo;
import com.biz.crm.upload.excel.ExcelExportFeign;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.util.Set;

@Slf4j
@Component("defaultExcelImport")
public class DefaultExcelImport implements ExcelImport {
    @Value("${excel.export.downloadUrl}")
    private String EXCEL_DOWNLOAD_URL;
    @Resource
    private ExcelExportFeign excelExportFeign;
    @Resource
    private RocketMQProducer rocketMQProducer;

    /**
     * 异步导入
     * @param paramVo
     */
    @Override
    @Async
    public void doAsyncImport(ExcelImportParamVo paramVo, String token, File file) {
        try {
            //绑定会话token
            UserUtils.setToken(token);
            //listener
            CurrentLimitableImportListener listener = new CurrentLimitableImportListener(paramVo);
            listener.setEXCEL_DOWNLOAD_URL(this.EXCEL_DOWNLOAD_URL);
            EasyExcel.read(file, this.getClazz(paramVo.getClassPathAsImportVo()), listener).sheet().doRead();
        }catch (BusinessException e){
            String msg = "导入任务[" + paramVo.getTaskCode() + "]:导入异常>>>>>" + e.getMsg();
            log.error(msg, e);
            this.updateTaskStatus(paramVo, msg);
            this.sendMQ(msg, paramVo);
        }catch (Exception e){
            String msg = "导入任务[" + paramVo.getTaskCode() + "]:导入异常>>>>>" + e.getMessage();
            log.error(msg, e);
            this.updateTaskStatus(paramVo, msg);
            this.sendMQ(msg, paramVo);
        }

    }
    /**
     * 发送mq
     * @param msgBody
     */
    private void sendMQ(String msgBody, ExcelImportParamVo importParamVo){
        String webSocketClientId = importParamVo.getWebSocketClientId();
        if(StringUtils.isBlank(webSocketClientId)){
            if(log.isInfoEnabled()){
                log.info("导入任务[" + importParamVo.getTaskCode() + "]:未提交 websocket id，忽略 websocket 消息发送！");
            }
            return;
        }
        MqMessageParam mqMessageParam = new MqMessageParam();
        mqMessageParam.setClientId(webSocketClientId);
        mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate(msgBody));
        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
        mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.EXCEL_IMPORT_WEBSOCKET_MSG);
        mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(mqMessageParam));
        this.rocketMQProducer.convertAndSend(mqMessageBody);
    }
    private void updateTaskStatus(ExcelImportParamVo paramVo, String msg){
        ExcelExportReqVo excelExportReqVo = new ExcelExportReqVo();
        excelExportReqVo.setId(paramVo.getTaskId());
        excelExportReqVo.setFileStatus(UploadEnum.fileStatus.IMPORT_EXCEPTION.getVal());
        excelExportReqVo.setRemarks(msg);
        Result<String> result = excelExportFeign.update(excelExportReqVo);
        if(!ApiResultUtil.checkResult(result) ){
            log.error("导入任务[" + paramVo.getTaskCode() + "]: 同步任务状态失败！");
        }
    }


    /**
     *同步导入excel(不会保存到数据库)
     * @param importParamVo
     */
    @Override
    public UnAsyncImportRespVo doUnAsyncImport(ExcelImportParamVo importParamVo, File file) {
        importParamVo.setTest(true);
        //listener
        CurrentLimitableImportListener listener = new CurrentLimitableImportListener(importParamVo);
        EasyExcel.read(file, this.getClazz(importParamVo.getClassPathAsImportVo()), listener).sheet().doRead();
        UnAsyncImportRespVo respVo = new UnAsyncImportRespVo();
        respVo.setDataAsFail(listener.getDataAsFail());
        respVo.setDataAsSuccess(listener.getDataCache());
        respVo.setTaskCode(importParamVo.getTaskCode());
        respVo.setTaskId(importParamVo.getTaskId());
        return respVo;
    }


    /**
     * 下载示例文件
     * @param importConfigCode
     */
    @Override
    public DemoFileResp demoFile(String importConfigCode) {

        ExcelImportParamVo excelImportParamVo = this.resolveConfig(importConfigCode);

        String fileName = excelImportParamVo.getFileNameAsDemo();
        Class<? extends AbstractImportVo> clazz = this.getClazz(excelImportParamVo.getClassPathAsImportVo());
        Set<String> superIgnore;
        try {
            //子类定义的忽略字段
            superIgnore = clazz.newInstance().superIgnore();
        } catch (InstantiationException e) {
            log.warn("解析类实例化失败[" + clazz.getName() + "]，请重试！", e);
            throw new BusinessException("解析类实例化失败[" + clazz.getName() + "]，请重试！", e);
        } catch (IllegalAccessException e) {
            log.warn("解析类实例化失败[" + clazz.getName() + "]，请重试！", e);
            throw new BusinessException("解析类实例化失败[" + clazz.getName() + "]，请重试！", e);
        }
        //添加基类忽略的字段
        superIgnore.addAll(AbstractImportVo.BASE_IMPORT_COLUMNS);
        File file = ExcelImportUtil.writeFile(fileName, clazz
                , ExcelTypeEnum.XLSX, Lists.newArrayList(), superIgnore);
        DemoFileResp demoFileResp = CrmBeanUtil.copy(excelImportParamVo, DemoFileResp.class);
        demoFileResp.setFile(file);
        return demoFileResp;
    }


    /**
     * 解析导入配置
     * @param configCode
     * @return
     */
    @Override
    public ExcelImportParamVo resolveConfig(String configCode){

        String config = DictUtil.getDictValueMapsByCodes(DictConstant.EXCEL_IMPORT_CONFIG).get(configCode);
        if(StringUtils.isBlank(config)){
            throw new BusinessException("不存在的导入配置[" + configCode + "]");
        }
        ExcelImportParamVo paramVo = JsonPropertyUtil.toObject(config, ExcelImportParamVo.class);
        if(null == paramVo){
            throw new BusinessException("解析导入配置失败，请前往数据字典检查Excel导入配置[" + configCode + "]");
        }
//        if(StringUtils.isBlank(paramVo.getBeanNameAsTransactionManager())){
//            throw new BusinessException("必须配置事务管理器[beanNameAsTransactionManager]，请前往数据字典检查Excel导入配置[" + configCode + "]");
//        }

        if(StringUtils.isBlank(paramVo.getBeanNameAsValidator())){
            throw new BusinessException("必须配置校验逻辑[beanNameAsValidator]，请前往数据字典检查Excel导入配置[" + configCode + "]");
        }
        if(StringUtils.isBlank(paramVo.getClassPathAsImportVo())){
            throw new BusinessException("必须配置解析类路径[classPathAsImportVo]，请前往数据字典检查Excel导入配置[" + configCode + "]");
        }
        if(StringUtils.isBlank(paramVo.getFileNameAsDemo())){
            throw new BusinessException("必须配置示例文件名[fileNameAsDemo]，请前往数据字典检查Excel导入配置[" + configCode + "]");
        }
        paramVo.setImportConfigCode(configCode);
        return paramVo;
    }

    private Class<? extends AbstractImportVo> getClazz(String classPathAsImportVo){
        if(StringUtils.isBlank(classPathAsImportVo)){
            throw new BusinessException("必须配置解析类路径[classPathAsImportVo]");
        }
        try {
            return (Class<? extends AbstractImportVo>) Class.forName(classPathAsImportVo);
        } catch (ClassNotFoundException e) {
            log.warn("不存在的行解析类路径[" + classPathAsImportVo + "]，请重试！", e);
            throw new BusinessException("不存在的行解析类路径[" + classPathAsImportVo + "]，请重试！", e);
        }
    }




}
