package com.biz.crm.excel.service.impl;

import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.excel.component.ExportHelper;
import com.biz.crm.excel.component.export.extend.head.ExportHeadExtend;
import com.biz.crm.excel.controller.req.ExportParamVo;
import com.biz.crm.excel.service.IExcelService;
import com.biz.crm.excel.util.DefaultExportContext;
import com.biz.crm.excel.util.ExportTaskThread;
import com.biz.crm.mdm.columnconfig.MdmColumnConfigFeign;
import com.biz.crm.mdm.tableconfig.MdmFunctionSubFeign;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnConfigReqVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnExportRespVo;
import com.biz.crm.nebular.mdm.pageconfig.MdmFunctionSubReqVo;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.util.PageDataAdviser;
import com.biz.crm.util.Result;
import com.biz.crm.util.UserUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @Description:
 * @project：crm
 * @package：com.biz.crm.excel.service.impl
 * @author：lifei
 * @create：2020/12/6 5:41 下午
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "ExcelServiceImpl")
public class ExcelServiceImpl implements IExcelService {


    @Value("${excel.export.downloadUrl}")
    private String EXCEL_DOWNLOAD_URL;

    @Resource
    private ExportHelper exportHelper;
    @Resource
    private MdmFunctionSubFeign mdmFunctionSubFeign;
    @Resource
    private MdmColumnConfigFeign mdmColumnConfigFeign;
    //分页参数信息
    private static final Integer PAGE_SIZE = 200;
    //核心线程数
    private static final Integer CORE_POOL_SIZE = 10;

    //临时线程数
    private static final Integer MAXIMUM_POOL_SIZE = 20;

    //空闲时间清除临时线程数
    private static final Integer KEEP_ALIVE_TIME = 20;

    private static final Integer SIZE = 30;

    private static final BlockingQueue<Runnable> QUEUE = new LinkedBlockingQueue<>(SIZE);

    public static ThreadPoolExecutor THREAD_POOL_EXECUTOR = new ThreadPoolExecutor(CORE_POOL_SIZE, MAXIMUM_POOL_SIZE, KEEP_ALIVE_TIME, TimeUnit.SECONDS, QUEUE);



    /**
     * 动态头导出完成
     *
     * @param requestParamMap
     */
    @Override
    public void repeatedWriteExport(HashMap<String, Object> requestParamMap) {
        ExportParamVo exportParam = JsonPropertyUtil.toObject(requestParamMap, ExportParamVo.class);
        this.checkExportParamVo(exportParam);
        //准备本次导出任务的上下文数据
        DefaultExportContext exportContext = this.buildExportContext(exportParam);
        exportContext.setRequestParamMap(requestParamMap);
        exportContext.setPageDataAdviser(new PageDataAdviser(PAGE_SIZE));
        int threadSize = QUEUE.size();
        if (threadSize == SIZE) {
            try {
                QUEUE.put(() -> {
                        this.exportHelper.sendWebsocketMsgMQ("当前导出人数太多,正在排队处理,请稍后到下载中心查看", exportContext);
                    }
                );
            } catch (InterruptedException e) {
                log.warn("创建导出任务失败", e);
                throw new BusinessException("创建导出任务失败，请重试！", e);
            }
        }
        ExcelServiceImpl.THREAD_POOL_EXECUTOR.execute(new ExportTaskThread(exportContext, this.EXCEL_DOWNLOAD_URL));
    }

    /**
     * 准备本次导出任务的上下文数据
     * @param exportParam
     */
    private DefaultExportContext buildExportContext(ExportParamVo exportParam){
        //导出列头扩展
        ExportHeadExtend headExtend = this.getExportExtendProcessor(exportParam.getFunctionCode());
        DefaultExportContext exportContext = new DefaultExportContext();
        exportContext.setExportParam(exportParam);
        exportContext.setHeadExtend(headExtend);
        exportContext.setLoginUserToken(UserUtils.getToken());
        //获取功能名称
        this.loadFunctionName(exportContext);
        //获取动态头部配置
        this.loadExcelHeadConfig(exportContext);
        return exportContext;
    }

    /**
     * 获取动态头部list
     * @param exportContext
     */
    private void loadExcelHeadConfig(DefaultExportContext exportContext) {
        ExportParamVo exportParam = exportContext.getExportParam();
        //获取头部信息
        LinkedHashMap<String, MdmColumnExportRespVo> map = Maps.newLinkedHashMap();
        MdmColumnConfigReqVo mdmColumnConfigReqVo = new MdmColumnConfigReqVo();
        mdmColumnConfigReqVo.setParentCode(exportParam.getParentCode());
        mdmColumnConfigReqVo.setFunctionCode(exportParam.getFunctionCode());
//        mdmColumnConfigReqVo.setColumnExport(YesNoEnum.YesNoCodeNumberEnum.YES.getCode().toString());
        Result<List<MdmColumnExportRespVo>> result = mdmColumnConfigFeign.columnSelect(mdmColumnConfigReqVo);
        ApiResultUtil.checkResult(result);
        List<MdmColumnExportRespVo> mdmColumnExportRespVos = result.getResult();
        if(null != mdmColumnExportRespVos){
            for (MdmColumnExportRespVo data : mdmColumnExportRespVos) {
                String field = data.getField();
                if(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString().equals(data.getColumnExport())
                        || MdmColumnExportRespVo.checkTypecheckbox.equals(field)){
                    continue;
                }
                if(StringUtils.isEmpty(field) || StringUtils.isEmpty(data.getTitle())){
                    throw new BusinessException("列表字段配置错误，属性/列头名称不能为空");
                }
                map.put(field, data);
            }
        }
        exportContext.setHeadMap(map);
    }


    /**
     * 获取功能名称
     *
     * @param exportContext
     * @return
     */
    private void loadFunctionName( DefaultExportContext exportContext) {
        ExportParamVo exportParam = exportContext.getExportParam();
        MdmFunctionSubReqVo mdmFunctionSubReqVo = new MdmFunctionSubReqVo() {{
            this.setParentCode(exportParam.getParentCode());
            this.setFunctionCode(exportParam.getFunctionCode());
        }};
        //功能名称
        String functionName = ApiResultUtil.objResult(mdmFunctionSubFeign.functionSubSearch(mdmFunctionSubReqVo), true).getFunctionName();
        if(StringUtils.isBlank(functionName)) {
            throw new BusinessException("获取功能名称失败，请重试！");
        }
        exportContext.setFunctionName(functionName);
    }

    /**
     *
     * @param exportParam
     * @return
     */
    private void checkExportParamVo(ExportParamVo exportParam){
        if (StringUtils.isBlank(exportParam.getSid())) {
            throw new BusinessException("websocket连接id为空");
        }
        if (StringUtils.isBlank(exportParam.getFunctionCode())) {
            throw new BusinessException("列表功能编码为空");
        }
        if (StringUtils.isBlank(exportParam.getParentCode())) {
            throw new BusinessException("菜单编码为空");
        }
        if (StringUtils.isBlank(exportParam.getRequestUrl())) {
            throw new BusinessException("数据列表接口请求路径");
        }

    }

    /**
     * 获取导出列头扩展
     * @param functionCode
     * @return
     */
    private ExportHeadExtend getExportExtendProcessor(String functionCode){
        if(StringUtils.isBlank(functionCode)){
            return null;
        }
        String beanName = functionCode + ExportHeadExtend.beanNameAsSuffix;
        if(SpringApplicationContextUtil.getApplicationContext().containsBean(beanName)){
            ExportHeadExtend exportHeadExtend;
            try{
                exportHeadExtend = SpringApplicationContextUtil.getApplicationContext().getBean(beanName, ExportHeadExtend.class);
                return exportHeadExtend;
            }catch (Exception e){
                log.warn("讲道理不应该跑这里来的", e);
                return null;
            }
        }
        return null;

    }




}
