package com.biz.crm.excel.util;

import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.AbstractImportVo;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.eunm.upload.UploadEnum;
import com.biz.crm.excel.component.saver.ExcelImportSaver;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.controller.req.ExcelImportParamVo;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.upload.excel.req.ExcelExportReqVo;
import com.biz.crm.nebular.upload.vo.UploadVo;
import com.biz.crm.upload.excel.ExcelExportFeign;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.io.File;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * 可限流的监听器
 *
 *  @author: luoqi
 *  @Date: 2021-1-4 15:33
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
public class CurrentLimitableImportListener<T extends AbstractImportVo> extends AnalysisEventListener<T> {
    //MQ服务
    private RocketMQProducer rocketMQProducer = SpringApplicationContextUtil.getApplicationContext().getBean("rocketMQProducer", RocketMQProducer.class);
    //MQ消息体
    private RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
    //下载中心服务
    private ExcelExportUtil excelExportUtil = SpringApplicationContextUtil.getApplicationContext().getBean("excelExportUtil", ExcelExportUtil.class);

    private ExcelExportFeign excelExportFeign = SpringApplicationContextUtil.getApplicationContext().getBean(ExcelExportFeign.class);
    //当前处理数据行对象的 class
    private Class<? extends AbstractImportVo> dataClass;
    @Setter
    private String EXCEL_DOWNLOAD_URL;
    //导入配置编码
    @Getter
    private ExcelImportParamVo importParamVo;
    //当前待处理的缓存
    @Getter
    private List<T> dataCache = Lists.newArrayList();
    //处理失败的缓存
    @Getter
    private List<T> dataAsFail = Lists.newArrayList();

    private LocalDateTime beginDateTime = LocalDateTime.now();
    //校验逻辑
    private ExcelImportValidator<T> validator;
    //保存逻辑
    private ExcelImportSaver<T> saver;

    private boolean initTag = false;
    /**
     * 处理数据
     * @param context
     */
    private void doProcess(DefaultImportContext context){

        boolean errorFlag = this.validate(context);
        if(errorFlag){
            this.sendMQ("校验异常，跳过数据保存...");
            return;
        }

        Boolean test = this.importParamVo.getTest();
        if(null == test || test){
            this.dataCache.forEach(v -> {
                v.appendSuccessSaveMsg("跳过数据保存");
            });
            this.sendMQ("当前为测试模式，跳过数据保存...");
            return;
        }
        //保存
        this.save(context);
    }

    /**
     * 初始化数据资源
     */
    private void init(T data){

        if(this.initTag){
            return;
        }
        String msg = "导入任务[" + importParamVo.getTaskCode() + "]:初始化监听器资源...";
        log.warn(msg);
        this.sendMQ(msg);
        if(null == importParamVo){
            throw new BusinessException("无效的导入配置参数");
        }
        try{
            this.validator = SpringApplicationContextUtil.getApplicationContext().getBean(importParamVo.getBeanNameAsValidator(), ExcelImportValidator.class);
            if(!importParamVo.getTest()){
                this.saver = SpringApplicationContextUtil.getApplicationContext().getBean(importParamVo.getBeanNameAsSaver(), ExcelImportSaver.class);
            }
        }catch (Exception e){
            log.warn("导入任务[" + importParamVo.getTaskCode() + "]:未找到 ExcelImportSaver/ExcelImportValidator 实例，请重试！", e);
            this.sendMQ("监听器资源初始化失败：未找到 ExcelImportSaver/ExcelImportValidator 实例，请重试！");
            throw new BusinessException("未找到 ExcelImportSaver/ExcelImportValidator 实例，请重试！", e);
        }
        if(null == this.dataClass){
            this.dataClass = data.getClass();
        }
        this.initTag = true;
        this.sendMQ("监听器资源初始化完成...\n开始读取EXCEL数据...");
    }

    public CurrentLimitableImportListener(ExcelImportParamVo importParamVo){
        this.importParamVo = importParamVo;
    }


    /**
     * When analysis one row trigger invoke function.
     *
     * @param data    one row value. Is is same as {@link AnalysisContext#readRowHolder()}
     * @param context
     */
    @Override
    public void invoke(T data, AnalysisContext context) {
        this.init(data);
        ReadRowHolder rowHolder = context.readRowHolder();
        if(null != rowHolder){
            data.setRowIndex(rowHolder.getRowIndex());
        }
        //当前待处理的缓存
        this.dataCache.add(data);
    }



    /**
     * 校验
     */
    private boolean validate(DefaultImportContext context){
        //校验
        this.sendMQ("开始校验数据...");
        String msg = null;
        boolean errorFlag = false;
        try{
            this.validator.validate(this.dataCache, context);
            this.sendMQ("校验完成...");
        }catch (BusinessException e){
            errorFlag = true;
            msg = "导入任务[" + importParamVo.getTaskCode() + "]:已中断导入任务，校验异常：" + e.getMsg();
            log.error(msg, e);
        }catch (Exception e){
            errorFlag = true;
            msg = "导入任务[" + importParamVo.getTaskCode() + "]:已中断导入任务，校验异常：" + e.getClass().getName() + "[" + e.getMessage()+ "]";
            log.error(msg, e);
        }
        //处理异常
        if(errorFlag){
            for(T data : this.dataCache){
                data.appendErrorValidateMsg(msg);
            }
            this.sendMQ(msg);
        }
        //缓存处理失败的数据
        List<T> dataAsSuccessTemp = this.cacheFailData(this.dataCache);
        this.dataCache = dataAsSuccessTemp;
        return errorFlag;
    }

    /**
     * 保存
     */
    private void save(DefaultImportContext context){
        this.sendMQ("开始保存数据,总数：" + this.dataCache.size());
        String msg = null;
        boolean errorFlag = false;
        try{
            this.saver.save(this.dataCache, context);
            this.sendMQ("保存完成...");
        }catch (BusinessException e){
            errorFlag = true;
            msg = "导入任务[" + importParamVo.getTaskCode() + "]:已中断导入任务，保存异常：" + e.getMsg();
            log.warn(msg, e);
        }catch (Exception e){
            errorFlag = true;
            msg = "导入任务[" + importParamVo.getTaskCode() + "]:已中断导入任务，保存异常：" + e.getClass().getName() ;
            log.error(msg, e);
        }

        //处理异常
        if(errorFlag){
            for(T data : this.dataCache){
                data.appendErrorSaveMsg(msg);
            }
            this.sendMQ(msg);
        }
        //缓存处理失败的数据
        List<T> dataAsSuccessTemp = this.cacheFailData(this.dataCache);
        this.dataCache = dataAsSuccessTemp;
    }

    /**
     * 缓存处理失败的数据
     */
    private List<T> cacheFailData(List<T> dataList){
        if(null == dataList){
            return Lists.newArrayList();
        }
        Map<AbstractImportVo.ProcessTypeEnum, List<T>> typeMapData = dataList.stream().collect(Collectors.groupingBy(T:: getProcessType));
        //当前处理失败
        List<T> dataAsFailTemp = typeMapData.get(AbstractImportVo.ProcessTypeEnum.FAIL);
        if(!CollectionUtils.isEmpty(dataAsFailTemp)){
            //加入处理失败的缓存
            this.dataAsFail.addAll(dataAsFailTemp);
        }
        List<T> dataAsSuccessTemp = typeMapData.get(AbstractImportVo.ProcessTypeEnum.SUCCESS);
        if(null == dataAsSuccessTemp){
            dataAsSuccessTemp = Lists.newArrayList();
        }
        return dataAsSuccessTemp;
    }


    /**
     * 发送mq
     * @param msgBody
     */
    private void sendMQ(String msgBody){
        String webSocketClientId = this.importParamVo.getWebSocketClientId();
        if(StringUtils.isBlank(webSocketClientId)){
            if(log.isInfoEnabled()){
                log.info("导入任务[" + importParamVo.getTaskCode() + "]:未提交 websocket id，忽略 websocket 消息发送！");
            }
            return;
        }
        MqMessageParam mqMessageParam = new MqMessageParam();
        mqMessageParam.setClientId(webSocketClientId);
        mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate(msgBody));
        mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.EXCEL_IMPORT_WEBSOCKET_MSG);
        mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(mqMessageParam));
        this.rocketMQProducer.convertAndSend(mqMessageBody);
    }

    /**
     * 导入处理完成
     * 将处理失败的数据写入新的excel，并推送至下载中心
     * @param context
     */
    private void importFinish(AnalysisContext context){
        ExcelExportReqVo excelExportReqVo = this.uploadFileAndBuildExcelExportReqVo(context);
        if(null == excelExportReqVo){
            return;
        }
        //把上传文件和失败数据文件推送至下载中心
        excelExportReqVo.setId(this.importParamVo.getTaskId());
        this.sendMQ("同步下载中心任务状态...");
        Result<String> result = excelExportFeign.update(excelExportReqVo);
        if(!ApiResultUtil.checkResult(result)){
            this.sendMQ("同步下载中心任务状态失败：" + result.getMessage());
            return;
        }
        this.sendMQ("同步下载中心任务状态完成...");
        //发送最终导入结果消息
        this.sendFinishMsg(excelExportReqVo);
    }

    /**
     * 上传导入文件和失败数据文件
     * @param context
     * @return
     */
    private ExcelExportReqVo uploadFileAndBuildExcelExportReqVo(AnalysisContext context){
        //原文件
        File baseFile = context.readWorkbookHolder().getFile();
        List<UploadVo> resultAsBaseData;
        try {
            this.sendMQ("EXCEL原文件上传...");
            //原文件上传到文件服务器
            resultAsBaseData = ResultUtil.listResultFromJsonStr(this.excelExportUtil
                    .uploadFile(baseFile.getPath(), UserUtils.getToken()), UploadVo.class, true);
            this.sendMQ("EXCEL原文件上传完成...");
        }catch (BusinessException e){
            log.warn("导入任务[" + importParamVo.getTaskCode() + "]:EXCEL原文件上传失败：" + e.getMsg(), e);
            this.sendMQ("EXCEL原文件上传失败：" + e.getMsg());
            return null;
        }catch (Exception e){
            log.warn("导入任务[" + importParamVo.getTaskCode() + "]:EXCEL原文件上传失败：" + e.getMessage(), e);
            this.sendMQ("EXCEL原文件上传失败：" + e.getMessage());
            return null;
        }


        UploadEnum.fileStatus fileStatus = UploadEnum.fileStatus.IMPORT_SUCCESS;
        //处理错误数据
        UploadVo failData = null;
        String failFileName = "处理失败数据";
        if(!CollectionUtils.isEmpty(this.dataAsFail)){
            // 失败数是否等于业务数据总行数（fileStatus：全部失败，则失败；部分失败，则异常）
            fileStatus = this.dataAsFail.size() >= this.dataCache.size() ?
                    UploadEnum.fileStatus.IMPORT_FAIL : UploadEnum.fileStatus.IMPORT_EXCEPTION;
            List<UploadVo> temp;
            try {
                this.sendMQ("处理失败数据文件上传...");
                //错误数据文件上传到文件服务器
                temp = ResultUtil.listResultFromJsonStr(this.excelExportUtil.createExcelFile(failFileName, this.dataClass
                        , context.readWorkbookHolder().getExcelType(), dataAsFail, null), UploadVo.class, true);
                this.sendMQ("处理失败数据文件上传完成...");
            }catch (BusinessException e){
                log.error("导入任务[" + importParamVo.getTaskCode() + "]:处理失败数据文件上传失败：" + e.getMsg(), e);
                this.sendMQ("处理失败数据文件上传失败：" + e.getMsg());
                return null;
            }catch (Exception e){
                log.error("导入任务[" + importParamVo.getTaskCode() + "]:处理失败数据文件上传失败：" + e.getMessage(), e);
                this.sendMQ("处理失败数据文件上传失败：");
                return null;
            }
            failData = temp.get(0);
        }
        UploadVo baseData = resultAsBaseData.get(0);
        //组装下载中心参数
        return this.buildExcelExportReqVo(baseFile.getName(), baseData, failFileName, failData, fileStatus);
    }

    /**
     * 组装下载中心参数
     * @param baseFileName
     * @param baseData
     * @param failFileName
     * @param failData
     * @return
     */
    private ExcelExportReqVo buildExcelExportReqVo(String baseFileName, UploadVo baseData, String failFileName, UploadVo failData, UploadEnum.fileStatus fileStatus){
        ExcelExportReqVo vo = new ExcelExportReqVo();
        vo.setObjectName(baseData.getObjectName());
        vo.setFilePath(baseData.getUrl());
//        vo.setFileName(baseFileName);
//        vo.setExcelFileName(baseFileName);
        if(null != failData){
            vo.setObjectNameAsImportFail(failData.getObjectName());
            vo.setFilePathAsImportFail(failData.getUrl());
            vo.setFileNameAsImportFail(failFileName);
            vo.setExcelFileNameAsImportFail(failFileName);
        }
        vo.setFileSource(this.importParamVo.getFileNameAsDemo());
        vo.setFileParam(this.importParamVo.getImportConfigCode());

        vo.setFileType(UploadEnum.fileType.IMPORT.getVal());
        vo.setFileStatus(fileStatus.getVal());
        vo.setBeginDate(beginDateTime.format(CrmDateUtils.yyyyMMddHHmmss));
        vo.setEndDate(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss));
        return vo;
    }


    /**
     * 发送最终导入结果消息
     * @param excelExportReqVo
     */
    private void sendFinishMsg(ExcelExportReqVo excelExportReqVo){
        StringBuilder msg = new StringBuilder("当前任务数据总行数：");
        msg.append(this.dataCache.size() + this.dataAsFail.size());
        msg.append("条\n");
        msg.append("成功数：");
        msg.append(this.dataCache.size());
        msg.append("条\n");
        msg.append("失败数：");
        msg.append(this.dataAsFail.size());
        msg.append("条");
        if(StringUtils.isNotBlank(excelExportReqVo.getObjectNameAsImportFail())){
            msg.append("<a href='");
            msg.append(this.EXCEL_DOWNLOAD_URL);
            msg.append("?objectName=");
            msg.append(excelExportReqVo.getObjectNameAsImportFail());
            msg.append("'>[点击下载]</a>");
        }
        this.sendMQ(msg.toString());
    }

    /**
     * 组装已处理数据的消息内容
     * @return
     */
    private String buildDataCacheMsg(){
        StringBuilder msg = new StringBuilder();
        int i = 0;
        for(T data : this.dataCache){
            if(i > 0){
                msg.append("\n");
            }
            msg.append("第");
            msg.append(data.getRowIndex());
            msg.append("行（Excel行号）[校验: ");
            msg.append(data.getValidateMsg());
            msg.append(" -> 保存: ");
            msg.append(data.getSaveMsg());
            msg.append("]");
            i ++;
        }
        return msg.toString();
    }

    /**
     * if have something to do after all analysis
     *
     * @param analysisContext
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext analysisContext) {
        DefaultImportContext importContext = new DefaultImportContext();
        importContext.setContext(analysisContext);
        importContext.setImportParamVo(this.importParamVo);
        this.sendMQ("读取EXCEL数据完成，总共读取[" + this.dataCache.size() + "]行数据...");
        this.doProcess(importContext);
        //处理失败的数据推送至下载中心
        this.importFinish(analysisContext);

    }
}
