package com.biz.crm.excel.util;

import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.AbstractImportVo;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.upload.UploadEnum;
import com.biz.crm.excel.component.saver.ExcelImportSaver;
import com.biz.crm.excel.component.validator.ExcelImportValidator;
import com.biz.crm.excel.controller.req.ExcelImportParamVo;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.upload.excel.req.ExcelExportReqVo;
import com.biz.crm.nebular.upload.vo.UploadVo;
import com.biz.crm.upload.excel.ExcelExportFeign;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.io.File;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
public class DefaultExcelImportListener<T extends AbstractImportVo> extends AnalysisEventListener<T> {
    private DataSourceTransactionManager transactionManager;
    //MQ服务
    private RocketMQProducer rocketMQProducer = SpringApplicationContextUtil.getApplicationContext().getBean("rocketMQProducer", RocketMQProducer.class);
    //MQ消息体
    private RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
    //下载中心服务
    private ExcelExportUtil excelExportUtil = SpringApplicationContextUtil.getApplicationContext().getBean("excelExportUtil", ExcelExportUtil.class);

    private ExcelExportFeign excelExportFeign = SpringApplicationContextUtil.getApplicationContext().getBean(ExcelExportFeign.class);
    //当前处理数据行对象的 class
    private Class<? extends AbstractImportVo> dataClass;
    @Setter
    private String EXCEL_DOWNLOAD_URL;
    //导入配置编码
    @Getter
    private ExcelImportParamVo importParamVo;
    //缓存待处理数量
    @Getter
    @Setter
    private int processNo = 999;
    private String webSocketClientId;
    //当前待处理的缓存
    private List<T> dataCache = Lists.newArrayList();

    //当前处理失败的缓存
    private List<T> dataAsFail = Lists.newArrayList();
    //总行数计数
    private long dataTotal = 0;
    //总行数计数
    private LocalDateTime beginDateTime = LocalDateTime.now();
    //校验逻辑
    private ExcelImportValidator<T> validator;
    //保存逻辑
    private ExcelImportSaver<T> saver;
    private TransactionStatus transactionStatus;

    private boolean initTag = false;
    /**
     * 处理数据
     * @param finish
     * @param context
     */
    private void doProcess(boolean finish, DefaultImportContext context){
        //校验
        this.validate(finish, context);
        //保存
        this.save(finish, context);
        //发送 websocket 消息
        this.sendDataCacheMsg(finish);
    }

    /**
     * 初始化数据资源
     */
    private void init(){
        if(this.initTag){
            return;
        }
        if(null == importParamVo){
            throw new BusinessException("无效的导入配置参数");
        }
        try{
//            this.transactionManager = SpringApplicationContextUtil.getApplicationContext().getBean(importParamVo.getBeanNameAsTransactionManager(), DataSourceTransactionManager.class);
            this.saver = SpringApplicationContextUtil.getApplicationContext().getBean(importParamVo.getBeanNameAsSaver(), ExcelImportSaver.class);
            this.validator = SpringApplicationContextUtil.getApplicationContext().getBean(importParamVo.getBeanNameAsValidator(), ExcelImportValidator.class);
        }catch (Exception e){
            log.warn("未找到 ExcelImportSaver/ExcelImportValidator/TransactionManager 实例，请重试！", e);
            throw new BusinessException("未找到 ExcelImportSaver/ExcelImportValidator/TransactionManager 实例，请重试！", e);
        }

        this.webSocketClientId = importParamVo.getWebSocketClientId();

        DefaultTransactionDefinition def = new DefaultTransactionDefinition();
        def.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW); // 事物隔离级别，开启新事务。
        this.transactionStatus = this.transactionManager.getTransaction(def); // 获得事务状态
        this.initTag = true;
    }

    public DefaultExcelImportListener(ExcelImportParamVo importParamVo){
        this.importParamVo = importParamVo;
    }


    /**
     * When analysis one row trigger invoke function.
     *
     * @param data    one row value. Is is same as {@link AnalysisContext#readRowHolder()}
     * @param context
     */
    @Override
    public void invoke(T data, AnalysisContext context) {
        this.init();
        this.dataTotal ++;
        try {
            this.dataClass = data.getClass();
            ReadRowHolder rowHolder = context.readRowHolder();
            if(null != rowHolder){
                data.setRowIndex(rowHolder.getRowIndex());
            }
            //当前待处理的缓存
            this.dataCache.add(data);
//            this.doProcess(false, context);
        } catch (Exception e) {
            log.error("导入异常>>>>>>", e);
            transactionManager.rollback(transactionStatus);
            this.sendMQ(ExcelImportUtil.importSocketMsgGenerate("导入异常"));
            throw e;
        }

    }



    /**
     * 校验
     * @param finish
     */
    private void validate(boolean finish, DefaultImportContext context){
        if(this.dataCache.size() >= this.processNo || finish){
            boolean process = true;
            String msg = null;
            try{
                this.validator.validate(this.dataCache, context);
            }catch (BusinessException e){
                process = false;
                msg = e.getMsg();
            }catch (Exception e){
                process = false;
                msg = e.getMessage();
            }

            //处理异常
            if(!process){
                for(T data : this.dataCache){
                    data.setProcessType(AbstractImportVo.ProcessTypeEnum.SUCCESS);
                    data.appendErrorValidateMsg(msg);
                }
            }
            //缓存处理失败的数据
            List<T> dataAsSuccessTemp = this.cacheFailData(this.dataCache);
            this.dataCache = dataAsSuccessTemp;
        }

    }

    /**
     * 保存
     * @param finish
     */
    private void save(boolean finish, DefaultImportContext context){
        if(this.dataCache.size() >= this.processNo || finish){
            boolean process = true;
            String msg = null;
            try{
                this.saver.save(this.dataCache, context);
            }catch (BusinessException e){
                process = false;
                msg = e.getMsg();
            }catch (Exception e){
                process = false;
                msg = e.getMessage();
            }
            //处理异常
            if(!process){
                for(T data : this.dataCache){
                    data.setProcessType(AbstractImportVo.ProcessTypeEnum.FAIL);
                    data.appendErrorSaveMsg(msg);
                }
            }
            //缓存处理失败的数据
            this.cacheFailData(this.dataCache);

        }
    }

    /**
     * 缓存处理失败的数据
     */
    private List<T> cacheFailData(List<T> dataList){
        if(null == dataList){
            return Lists.newArrayList();
        }
        Map<AbstractImportVo.ProcessTypeEnum, List<T>> typeMapData = dataList.stream().collect(Collectors.groupingBy(T :: getProcessType));
        //当前处理失败
        List<T> dataAsFailTemp = typeMapData.get(AbstractImportVo.ProcessTypeEnum.FAIL);
        if(!CollectionUtils.isEmpty(dataAsFailTemp)){
            //加入处理失败的缓存
            this.dataAsFail.addAll(dataAsFailTemp);
        }
        List<T> dataAsSuccessTemp = typeMapData.get(AbstractImportVo.ProcessTypeEnum.SUCCESS);
        if(null == dataAsSuccessTemp){
            dataAsSuccessTemp = Lists.newArrayList();
        }
        return dataAsSuccessTemp;
    }

    /**
     * 发送当前已处理数据的消息
     * @param finish
     */
    private void sendDataCacheMsg(boolean finish){
        if(this.dataCache.size() >= this.processNo || finish){
            MqMessageParam mqMessageParam = new MqMessageParam();
//            TextMessage textMessage = new TextMessage();
            mqMessageParam.setClientId(this.webSocketClientId);
            mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate(this.buildDataCacheMsg()));
            this.sendMQ(JsonPropertyUtil.toJsonString(mqMessageParam));
            //清空待处理缓存
            this.dataCache = Lists.newArrayList();
        }
    }

    /**
     * 发送mq
     * @param msgBody
     */
    private void sendMQ(String msgBody){
        mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.EXCEL_IMPORT_WEBSOCKET_MSG);
        mqMessageBody.setMsgBody(msgBody);
        this.rocketMQProducer.convertAndSend(mqMessageBody);
    }

    /**
     * 导入处理完成
     * 将处理失败的数据写入新的excel，并推送至下载中心
     * @param context
     */
    private void importFinish(AnalysisContext context){
        ExcelExportReqVo excelExportReqVo = this.uploadFileAndBuildExcelExportReqVo(context);
        //把上传文件和失败数据文件推送至下载中心
        excelExportReqVo.setId(this.importParamVo.getTaskId());
        Result<String> result = excelExportFeign.update(excelExportReqVo);
        if(!ApiResultUtil.checkResult(result)){
            throw new BusinessException("文件推送失败[" + result.getMessage() + "]，请重试！");
        }
        //发送最终导入结果消息
        this.sendFinishMsg(excelExportReqVo);
    }

    /**
     * 上传导入文件和失败数据文件
     * @param context
     * @return
     */
    private ExcelExportReqVo uploadFileAndBuildExcelExportReqVo(AnalysisContext context){
        //原文件
        File baseFile = context.readWorkbookHolder().getFile();

        //原文件上传到文件服务器
        List<UploadVo> resultAsBaseData = ResultUtil.listResultFromJsonStr(this.excelExportUtil
                .uploadFile(baseFile.getPath(), UserUtils.getToken()), UploadVo.class, true);
        if(CollectionUtils.isEmpty(resultAsBaseData)){
            throw new BusinessException("原文件上传失败，请重试！");
        }

        UploadEnum.fileStatus fileStatus = UploadEnum.fileStatus.IMPORT_SUCCESS;
        //处理错误数据
        UploadVo failData = null;
        String failFileName = "处理失败数据";
        if(!CollectionUtils.isEmpty(this.dataAsFail)){
            // 失败数是否等于业务数据总行数（fileStatus：全部失败，则失败；部分失败，则异常）
            fileStatus = this.dataAsFail.size() >= this.dataTotal ?
                    UploadEnum.fileStatus.IMPORT_FAIL : UploadEnum.fileStatus.IMPORT_EXCEPTION;
            //错误数据文件上传到文件服务器
            List<UploadVo> tepm = ResultUtil.listResultFromJsonStr(this.excelExportUtil.createExcelFile(failFileName, this.dataClass
                    , context.readWorkbookHolder().getExcelType(), dataAsFail, null), UploadVo.class, true);
            if(CollectionUtils.isEmpty(tepm)){
                throw new BusinessException("处理失败数据文件上传失败，请重试！");
            }
            failData = tepm.get(0);
        }
        UploadVo baseData = resultAsBaseData.get(0);
        //组装下载中心参数
        return this.buildExcelExportReqVo(baseFile.getName(), baseData, failFileName, failData, fileStatus);
    }

    /**
     * 组装下载中心参数
     * @param baseFileName
     * @param baseData
     * @param failFileName
     * @param failData
     * @return
     */
    private ExcelExportReqVo buildExcelExportReqVo(String baseFileName, UploadVo baseData, String failFileName, UploadVo failData, UploadEnum.fileStatus fileStatus){
        ExcelExportReqVo vo = new ExcelExportReqVo();
        vo.setObjectName(baseData.getObjectName());
        vo.setFilePath(baseData.getUrl());
//        vo.setFileName(baseFileName);
//        vo.setExcelFileName(baseFileName);
        if(null != failData){
            vo.setObjectNameAsImportFail(failData.getObjectName());
            vo.setFilePathAsImportFail(failData.getUrl());
            vo.setFileNameAsImportFail(failFileName);
            vo.setExcelFileNameAsImportFail(failFileName);
        }
        vo.setFileSource(this.importParamVo.getFileNameAsDemo());
        vo.setFileParam(this.importParamVo.getImportConfigCode());

        vo.setFileType(UploadEnum.fileType.IMPORT.getVal());
        vo.setFileStatus(fileStatus.getVal());
        vo.setBeginDate(beginDateTime.format(CrmDateUtils.yyyyMMddHHmmss));
        vo.setEndDate(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss));
        return vo;
    }


    /**
     * 发送最终导入结果消息
     * @param excelExportReqVo
     */
    private void sendFinishMsg(ExcelExportReqVo excelExportReqVo){
        StringBuilder msg = new StringBuilder("当前任务数据总行数：");
//            + this.dataAsFail.size() + "条数据处理失败：已推送至下载中心 -> " + excelFileName
        msg.append(this.dataTotal);
        msg.append("条\n");
        msg.append("成功数：");
        msg.append(this.dataTotal - this.dataAsFail.size());
        msg.append("条\n");
        msg.append("失败数：");
        msg.append(this.dataAsFail.size());
        msg.append("条");
        if(StringUtils.isNotBlank(excelExportReqVo.getObjectNameAsImportFail())){
            msg.append("<a href='");
            msg.append(this.EXCEL_DOWNLOAD_URL);
            msg.append("?objectName=");
            msg.append(excelExportReqVo.getObjectNameAsImportFail());
            msg.append("'>[点击下载]</a>");
        }
        MqMessageParam mqMessageParam = new MqMessageParam();
        mqMessageParam.setClientId(this.webSocketClientId);

        mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate(msg.toString()));
        this.sendMQ(JsonPropertyUtil.toJsonString(mqMessageParam));
    }

    /**
     * 组装已处理数据的消息内容
     * @return
     */
    private String buildDataCacheMsg(){
        StringBuilder msg = new StringBuilder();
        int i = 0;
        for(T data : this.dataCache){
            if(i > 0){
                msg.append("\n");
            }
            msg.append("第");
            msg.append(data.getRowIndex());
            msg.append("行（Excel行号）[校验: ");
            msg.append(data.getValidateMsg());
            msg.append(" -> 保存: ");
            msg.append(data.getSaveMsg());
            msg.append("]");
            i ++;
        }
        return msg.toString();
    }

    /**
     * if have something to do after all analysis
     *
     * @param context
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext context) {
        try {
//            this.doProcess(true, context);
            //处理失败的数据推送至下载中心
            this.importFinish(context);
            transactionManager.commit(transactionStatus);
        } catch (BusinessException e) {
            log.error("导入异常>>>>>>", e);
            transactionManager.rollback(transactionStatus);
            MqMessageParam mqMessageParam = new MqMessageParam();
            mqMessageParam.setClientId(this.webSocketClientId);
            mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate("异常信息：" + e.getMsg()));
            this.sendMQ(JsonPropertyUtil.toJsonString(mqMessageParam));
            throw e;
        } catch (Exception e) {
            log.error("导入异常>>>>>>", e);
            transactionManager.rollback(transactionStatus);
            MqMessageParam mqMessageParam = new MqMessageParam();
            mqMessageParam.setClientId(this.webSocketClientId);
            mqMessageParam.setMsg(ExcelImportUtil.importSocketMsgGenerate("异常信息：" + e.getMessage()));
            this.sendMQ(JsonPropertyUtil.toJsonString(mqMessageParam));
            throw e;
        }

    }
}
