package com.biz.crm.excel.util;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.resttemplate.RestTemplateUtils;
import com.biz.crm.excel.component.ExportHelper;
import com.biz.crm.excel.component.export.extend.head.ExportHeadExtend;
import com.biz.crm.excel.controller.req.ExportParamVo;
import com.biz.crm.excel.service.impl.ExcelServiceImpl;
import com.biz.crm.mdm.columnconfig.MdmColumnConfigFeign;
import com.biz.crm.nebular.mdm.pageconfig.MdmColumnExportRespVo;
import com.biz.crm.nebular.upload.vo.UploadVo;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.util.PageDataAdviser;
import com.biz.crm.util.Result;
import com.biz.crm.util.ResultUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.UserUtils;
import com.biz.crm.websocket.user.endpoint.UserWebSocketHandler;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @Description:
 * @project：crm
 * @package：com.biz.crm.excel.util
 * @author：lifei
 * @create：2020/12/3 3:39 下午
 */
@Component
@Slf4j
public class ExcelExportUtil {


    @Value("${excel.export.basdir}")
    private String EXCEL_EXPORT_BASDIR;

    @Value("${excel.export.url}")
    private String URL;


    //上传文件请求地址
    @Value("${excel.export.uploadUrl}")
    private String UPLOAD_URL;

    //分页参数信息
    private static final Integer PAGE_SIZE = 100;

    //分页参数信息
    private static final Integer PAGE_NUM = 1;

    private static UserWebSocketHandler userWebSocketHandler = new UserWebSocketHandler();

    @Resource
    private RestTemplateUtils restTemplateUtils;

    @Autowired
    private ExcelServiceImpl excelService;

    @Resource
    private MdmColumnConfigFeign mdmColumnConfigFeign;
    @Resource
    private ExportHelper exportHelper;


    /**
     * 写入excel
     *
     * @param data
     * @param exportContext
     * @param excelWriter
     */
    private void againWrite(DefaultExportContext exportContext, List<?> data, ExcelWriter excelWriter) {
        excelWriter.write(data, exportContext.getWriteSheet());
        this.exportHelper.sendWebsocketMsgMQ("已写入临时文件：" + exportContext.getWrittenNo() + "/" + exportContext.getPageTotal() + "条数据", exportContext);
    }

    /**
     * 解析头部信息
     *
     * @param map
     * @return
     */
    private List<List<String>> resolveHeadMap(Map<String, MdmColumnExportRespVo> map) {
        List<List<String>> listList = Lists.newArrayList();
        for (Map.Entry<String, MdmColumnExportRespVo> m : map.entrySet()) {
            List<String> headList = new ArrayList<>();
            headList.add(m.getValue().getTitle());
            listList.add(headList);
        }
        return listList;
    }

    /**
     * 获取分页信息
     *
     * @return
     */
    private Result getPageData(DefaultExportContext exportContext) {
        try {
//            jsonObject.remove("parentCode");
//            jsonObject.remove("functionCode");
            ExportParamVo exportParam = exportContext.getExportParam();
            String requestUrl = exportParam.getRequestUrl();

            HashMap<String, Object> requestParamMap = exportContext.getRequestParamMap();
            PageDataAdviser pageDataAdviser = exportContext.getPageDataAdviser();
            Integer pageNum = pageDataAdviser.nextPage();
            requestParamMap.put("pageNum", pageNum);
            requestParamMap.put("pageSize", pageDataAdviser.getPageSize());
            requestUrl = URL + requestUrl;
//            requestUrl = "http://localhost:8307" + requestUrl;
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
            headers.set(GlobalParam.TOKEN, exportContext.getLoginUserToken());
            headers.set(GlobalParam.MENU_CODE, exportParam.getParentCode());
            headers.set(GlobalParam.FUNCTION_CODE, exportParam.getFunctionCode());
            if(log.isInfoEnabled()){
                log.info("导出任务 [" + exportContext.getTaskCode() + "]尝试读取列表数据:[第" + pageNum + "页,每页" + pageDataAdviser.getPageSize() + "条数据], params=" + JsonPropertyUtil.toJsonString(requestParamMap));
            }
            Result result = getRequestResult(requestUrl, requestParamMap, headers);
            ApiResultUtil.checkResult(result);
            return result;
        } catch (Exception e) {
            String msg = "导出任务 [" + exportContext.getTaskCode() + "]读取列表数据信息失败";
            log.error(msg, e);
            this.exportHelper.processException(exportContext, msg);
            return null;
        }
    }

    /**
     * 获取请求结果
     * @param requestUrl 请求url
     * @param requestParamMap 请求参数
     * @param headers 请求头
     * @return 请求结果
     */
    private Result getRequestResult(String requestUrl, HashMap<String, Object> requestParamMap, HttpHeaders headers) {
        ResponseEntity<Result> response;
        Result result;
        try {
            response = RestTemplateUtils.postForEntity(requestUrl, headers, requestParamMap, Result.class);
            result = response.getBody();
        } catch (Exception e) {
            //此处是新老规范post和get请求问题
            requestParamMap.put("page", requestParamMap.get("pageNum"));
            requestParamMap.put("size", requestParamMap.get("pageSize"));
            UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(requestUrl);
            requestParamMap.forEach(builder::queryParam);
            log.info("post接口请求错误,此处转化为get请求:{}",builder.build(false).toUriString());
            HttpEntity<?> requestEntity = new HttpEntity<>(headers);
            response = RestTemplateUtils.geTemplate().exchange(builder.build(false).toUriString(), HttpMethod.GET, requestEntity, Result.class);
            result = response.getBody();
            if (Objects.nonNull(result) && Objects.nonNull(result.getResult())) {
                JSONObject dataPage = JSON.parseObject(JSON.toJSONString(result.getResult()));
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("data", dataPage.get("records"));
                jsonObject.put("count", dataPage.get("total"));
                result.setResult(jsonObject);
            }
        }
        return result;
    }

    /**
     * 动态头-重复多次写入excel
     *
     * @param exportContext
     */
    public UploadVo repeatedWrite(DefaultExportContext exportContext) {

        Result result = this.getPageData(exportContext);

        PageResult pageResult = JsonPropertyUtil.toObject(JsonPropertyUtil.toJsonStringNotEmptyVal(result.getResult()), PageResult.class);
        Long pageTotal = pageResult.getCount();
        exportContext.setPageTotal(pageTotal);
        this.exportHelper.sendWebsocketMsgMQ("预计写入数据总条数：" + pageTotal, exportContext);
        //校验总条数，如果总条数为0 ，则直接中断任务
        this.verificationDataCount(exportContext, pageTotal);
        //列头扩展
        this.headExtend(exportContext, pageResult);
        //解析业务数据
        List<List<Object>> dataList = this.resolveBizData(exportContext, pageResult);

        File file = this.tempFile(exportContext);
        ExcelWriter excelWriter = null;
        try {
            List<List<String>> headList = this.resolveHeadMap(exportContext.getHeadMap());
            excelWriter = EasyExcel.write(file).head(headList).build();
            // 这里注意 如果同一个sheet只要创建一次
            exportContext.addWrittenNo(dataList.size());
            this.againWrite(exportContext, dataList, excelWriter);
            this.againGetPageData(exportContext, excelWriter);
            this.exportHelper.sendWebsocketMsgMQ("已完成全部数据写入,总共写入数据：" + exportContext.getPageTotal() + "条", exportContext);
        } finally {
            // 千万别忘记finish 会帮忙关闭流
            if (excelWriter != null) {
                excelWriter.finish();
            }
        }
        this.exportHelper.sendWebsocketMsgMQ("正在尝试上传文件，请稍后.......", exportContext);
        try {
            List<UploadVo> uploadVos = ResultUtil.listResultFromJsonStr(this.uploadFile(file.getPath(), exportContext.getLoginUserToken()), UploadVo.class, true);
            return uploadVos.get(0);
        } catch (Exception e) {
            String msg = "导出任务 [" + exportContext.getTaskCode() + "]上传导出文件失败";
            log.warn(msg, e);
            this.exportHelper.processException(exportContext, msg);
            return null;
        }
    }

    /**
     * 创建临时文件
     * @param exportContext
     * @return
     */
    private File tempFile(DefaultExportContext exportContext){
        try {
            this.exportHelper.sendWebsocketMsgMQ("创建临时文件,准备写入数据：[" + exportContext.getExcelFileName() + "]", exportContext);
            return File.createTempFile(exportContext.getExcelFileName(), ExcelTypeEnum.XLSX.getValue());
        } catch (IOException e) {
            log.warn("导出任务 [" + exportContext.getTaskCode() + "]创建临时文件失败", e);
            throw new BusinessException("导出任务 [" + exportContext.getTaskCode() + "]创建临时文件失败，请重试！", e);
        }
    }

    private List<Map> headValueExtend(DefaultExportContext exportContext, List<Map> dataMaps) {
        ExportHeadExtend headExtend = exportContext.getHeadExtend();
        if(null == headExtend){
            return dataMaps;
        }
        List<Map> dataMapsTemp = headExtend.getHeadsValue(exportContext, dataMaps);
        dataMaps = dataMapsTemp==null?dataMaps:dataMapsTemp;
        return dataMaps;
    }

    /**
     * 解析业务数据
     * @param exportContext
     * @param pageResult
     * @return
     */
    private List<List<Object>> resolveBizData(DefaultExportContext exportContext, PageResult pageResult){
        List<Map> dataMaps = pageResult.getData();
        //根据headVariableSum动态扩展表头
        Integer sumFirst = exportContext.getHeadVariableSum();
        //获取数据，改变headVariableSum
        dataMaps = this.headValueExtend(exportContext, dataMaps);
        Integer sumExtend = exportContext.getHeadVariableSum();
        if (sumExtend != 0){
            //取最大值作为扩展表头的最大数
            if (sumExtend > sumFirst){
                this.headExtend(exportContext, pageResult);
            }
        }
        Map<String, MdmColumnExportRespVo> headMap = exportContext.getHeadMap();
        List<List<Object>> dataList = new ArrayList<>();
        for (Map<String, String> data : dataMaps) {
            List<Object> dataRow = new ArrayList<>();
            for (Map.Entry<String, MdmColumnExportRespVo> head : headMap.entrySet()) {
                String column = head.getKey();
                String dictCode = head.getValue().getDictCode();
                String val =null;
                Object dataValue=data.get(column);
               if( dataValue instanceof String|| dataValue instanceof Number){
                   val= data.get(column);
               }else{
                   log.warn("接口返回类型不合规范 funciontCode=[{}],字段=[{}]",exportContext.getExportParam().getFunctionCode(),column);
               }


                //字典翻译
                if (StringUtils.isNotEmpty(val) && StringUtils.isNotEmpty(dictCode)){
                    String valAsDict = DictUtil.dictValue(dictCode, val);
                    if(org.apache.commons.lang3.StringUtils.isNotBlank(valAsDict)){
                        val = valAsDict;
                    }
                }
                dataRow.add(val);
            }
            dataList.add(dataRow);
        }
        return dataList;
    }




    /**
     * 再次得到分页数据信息并且写入
     *
     * @param excelWriter
     */
    private void againGetPageData(DefaultExportContext exportContext, ExcelWriter excelWriter) {
        PageDataAdviser pageDataAdviser = exportContext.getPageDataAdviser();
        while (pageDataAdviser.hasNext(exportContext.getPageTotal())) {
            Result result = this.getPageData(exportContext);
            PageResult pageResult = JsonPropertyUtil.toObject(JsonPropertyUtil.toJsonString(result.getResult()), PageResult.class);
//            this.verificationDataCount(exportContext, pageTotal);
            List<List<Object>> dataList = this.resolveBizData(exportContext, pageResult);

            exportContext.addWrittenNo(dataList.size());
            this.againWrite(exportContext, dataList, excelWriter);
        }
    }




    /**
     * 上传文件
     *
     * @param urlFile
     * @param token
     */
    public String uploadFile(String urlFile, String token) {
        String requestUrl = URL + UPLOAD_URL;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);
        headers.set(GlobalParam.TOKEN, token);
        String[] fileUrls = urlFile.split(",");
        MultiValueMap<String, Object> form = new LinkedMultiValueMap<>();
        for (int i = 0; i < fileUrls.length; i++) {
            //设置请求体，注意是LinkedMultiValueMap
            form.add("file", new FileSystemResource(fileUrls[i]));
        }
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(form, headers);
        String result = restTemplateUtils.postForObject(requestUrl, httpEntity);
        return result;
    }

    /**
     * 创建ExcelFile
     *
     * @param excelFileName
     * @param classObj
     * @param excelTypeEnum
     * @param data
     * @param excludeColumnFiledNames
     * @return
     */
    public String createExcelFile(String excelFileName, Class classObj, ExcelTypeEnum excelTypeEnum, List<?> data, Set<String> excludeColumnFiledNames) {
        if (null == classObj || org.apache.commons.lang3.StringUtils.isBlank(excelFileName)) {
            throw new BusinessException("类路径、文件名称不能为空！");
        }
        if (null == excelTypeEnum) {
            excelTypeEnum = ExcelTypeEnum.XLSX;
        }
        File file = ExcelImportUtil.writeFile(excelFileName, classObj, excelTypeEnum, data, excludeColumnFiledNames);
        return this.uploadFile(file.getPath(), UserUtils.getToken());
    }


    private void verificationDataCount(DefaultExportContext exportContext, Long pageTotal) {
        if (pageTotal == 0) {
            this.exportHelper.processException(exportContext, "导出总条数为0");
        }
    }

    /**
     * 列头扩展
     * @param exportContext
     */
    private void headExtend(DefaultExportContext exportContext, PageResult pageResult) {
        ExportHeadExtend headExtend = exportContext.getHeadExtend();
        if(null == headExtend){
            return;
        }
        ExportHeadExtend.ExportHeadExtendParam headExtendParam = new ExportHeadExtend.ExportHeadExtendParam();

        headExtendParam.setDataRow((Map)pageResult.getData().get(0));
        headExtendParam.setHeadVariableSum(exportContext.getHeadVariableSum());
        List<MdmColumnExportRespVo> heads = headExtend.getHeads(headExtendParam);
        if(null == heads){
            return;
        }
        Map<String, MdmColumnExportRespVo> headMap = exportContext.getHeadMap();
        for (MdmColumnExportRespVo head : heads) {
            if(null == head){
                continue;
            }
            if(StringUtils.isEmpty(head.getField()) || StringUtils.isEmpty(head.getTitle())){
                String msg = "扩展列头数据错误，属性/列头名称不能为空，请检查列头扩展类实例[" + exportContext.getFunctionName() + "]";
                this.exportHelper.delTask(msg, exportContext);
                throw new BusinessException(msg);
            }
            headMap.put(head.getField(),head);
        }
    }
}
