package com.biz.crm.confadmin.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.confadmin.mapper.KmsDirectSystemMapper;
import com.biz.crm.confadmin.model.KmsDirectOrderTypeEntity;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.confadmin.model.KmsDirectSystemEntity;
import com.biz.crm.confadmin.service.IKmsDirectOrderTypeService;
import com.biz.crm.confadmin.service.IKmsDirectSystemService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.nebular.kms.confadmin.req.KmsDirectSystemReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectOrderTypeRespVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectSystemRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 商超信息接口实现
 *
 * @author Gavin
 * @date 2021-05-06 13:51:45
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsDirectSystemServiceExpandImpl")
public class KmsDirectSystemServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsDirectSystemMapper, KmsDirectSystemEntity> implements IKmsDirectSystemService {

    @Resource
    private KmsDirectSystemMapper kmsDirectSystemMapper;

    @Resource
    private IKmsDirectOrderTypeService directOrderTypeService;

    @Resource
    private KmsDirectSystemHelper helper;

    @Resource
    private IKmsDirectOrderTypeService kmsDirectOrderTypeService;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsDirectSystemRespVo> findList(KmsDirectSystemReqVo reqVo) {
        Page<KmsDirectSystemRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsDirectSystemRespVo> list = kmsDirectSystemMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            helper.convertData(list);
        }
        return PageResult.<KmsDirectSystemRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsDirectSystemRespVo
     */
    @Override
    public KmsDirectSystemRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsDirectSystemEntity entity = kmsDirectSystemMapper.selectById(id);
        AssertUtils.isNotNull(entity, "未查询到数据信息");
        KmsDirectSystemRespVo directSystemRespVo = CrmBeanUtil.copy(entity, KmsDirectSystemRespVo.class);
        //查询单据类型
        List<KmsDirectOrderTypeEntity> orderTypeEntities = kmsDirectOrderTypeService.lambdaQuery()
                .eq(KmsDirectOrderTypeEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .eq(KmsDirectOrderTypeEntity::getDirectId, id).list();
        directSystemRespVo.setOrderTypeVos(CrmBeanUtil.copyList(orderTypeEntities, KmsDirectOrderTypeRespVo.class));
        return directSystemRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsDirectSystemReqVo reqVo) {
        helper.paramCheck(reqVo);
        KmsDirectSystemEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectSystemEntity.class);
        this.save(entity);
        reqVo.setId(entity.getId());
        //构建支持单据类型
        List<KmsDirectOrderTypeEntity> orderTypeEntities = helper.buildDirectOrderType(reqVo);
        directOrderTypeService.saveBatch(orderTypeEntities);
        KmsDirectSystemRespVo newData = CrmBeanUtil.copy(entity, KmsDirectSystemRespVo.class);
        newData.setOrderTypeVos(CrmBeanUtil.copyList(orderTypeEntities, KmsDirectOrderTypeRespVo.class));
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsDirectSystemReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        AssertUtils.isNotNull(kmsDirectSystemMapper.selectById(reqVo.getId()), "操作数据不存在");
        helper.paramCheck(reqVo);
        KmsDirectSystemEntity entity = CrmBeanUtil.copy(reqVo, KmsDirectSystemEntity.class);
        KmsDirectSystemRespVo oldData = this.query(reqVo.getId());
        this.updateById(entity);
        //构建支持单据类型
        List<KmsDirectOrderTypeEntity> allEntities = directOrderTypeService.lambdaQuery()
                .eq(KmsDirectOrderTypeEntity::getDirectId, entity.getId()).list();
        List<KmsDirectOrderTypeEntity> currEntities = helper.buildDirectOrderType(reqVo);
        //删除集合
        List<KmsDirectOrderTypeEntity> entities = Lists.newArrayList();
        allEntities.forEach(o -> {
            String orderType = o.getOrderType();
            KmsDirectOrderTypeEntity ot = currEntities.stream()
                    .filter(e -> e.getOrderType().equals(orderType)).findFirst().orElse(null);
            if (ObjectUtils.isEmpty(ot)) {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
                entities.add(o);
            }
        });
        currEntities.forEach(o -> {
            String orderType = o.getOrderType();
            KmsDirectOrderTypeEntity ot = allEntities.stream()
                    .filter(e -> e.getOrderType().equals(orderType)).findFirst().orElse(null);
            if (ObjectUtils.isEmpty(ot)) {
                entities.add(o);
            } else if (StringUtils.isEmpty(o.getId())) {
                o.setId(ot.getId());
                o.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
                entities.add(o);
            }
        });
        directOrderTypeService.saveOrUpdateBatch(entities);
        KmsDirectSystemRespVo newData = this.query(reqVo.getId());
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsDirectSystemEntity> kmsDirectSystemEntities = kmsDirectSystemMapper.selectBatchIds(ids);
        Set<String> directIds = Sets.newHashSet();
        Map<String, String> map = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(kmsDirectSystemEntities)) {
            kmsDirectSystemEntities.forEach(o -> {
                map.put(o.getId(), o.getDirectName());
                directIds.add(o.getId());
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
            List<KmsDirectStoreEntity> storeEntities = kmsDirectStoreMapper
                    .selectList(Wrappers.<KmsDirectStoreEntity>lambdaQuery()
                            .eq(KmsDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                            .in(KmsDirectStoreEntity::getDirectId, directIds));
            if (CollectionUtils.isNotEmpty(storeEntities)) {
                Set<String> names = Sets.newHashSet();
                Set<String> idsByDb = storeEntities.stream().map(KmsDirectStoreEntity::getDirectId).collect(Collectors.toSet());
                idsByDb.forEach(x -> {
                    if (map.containsKey(x)) {
                        names.add(map.get(x));
                    }
                });
                throw new BusinessException(names.toString() + "的商超系统存在门店，不可删除");
            }
        }
        this.updateBatchById(kmsDirectSystemEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为启用
        List<KmsDirectSystemEntity> kmsDirectSystemEntities = kmsDirectSystemMapper.selectBatchIds(ids);
        List<KmsDirectSystemRespVo> oldDataList = CrmBeanUtil.copyList(kmsDirectSystemEntities, KmsDirectSystemRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsDirectSystemEntities)) {
            kmsDirectSystemEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectSystemEntities);
        List<KmsDirectSystemRespVo> newDataList = CrmBeanUtil.copyList(kmsDirectSystemEntities, KmsDirectSystemRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(0).getId(), newDataList.get(0).getId(), oldDataList.get(0), newDataList.get(0));
        }
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为禁用
        List<KmsDirectSystemEntity> kmsDirectSystemEntities = kmsDirectSystemMapper.selectBatchIds(ids);
        List<KmsDirectSystemRespVo> oldDataList = CrmBeanUtil.copyList(kmsDirectSystemEntities, KmsDirectSystemRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsDirectSystemEntities)) {
            kmsDirectSystemEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsDirectSystemEntities);
        List<KmsDirectSystemRespVo> newDataList = CrmBeanUtil.copyList(kmsDirectSystemEntities, KmsDirectSystemRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(0).getId(), newDataList.get(0).getId(), oldDataList.get(0), newDataList.get(0));
        }
    }

    @Override
    public PageResult<KmsDirectSystemRespVo> findDirectSystemList(KmsDirectSystemReqVo reqVo) {
        Page<KmsDirectSystemRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsDirectSystemRespVo> respVos = kmsDirectSystemMapper.findSystemListByCusOrg(page,reqVo);
        return PageResult.<KmsDirectSystemRespVo>builder()
                .data(respVos)
                .count(page.getTotal())
                .build();
    }
}
