package com.biz.crm.orderform.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.nebular.kms.orderform.req.KmsOrderFormReqVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderFormExportRespVo;
import com.biz.crm.nebular.kms.orderform.resp.KmsOrderFormRespVo;
import com.biz.crm.orderform.model.KmsOrderFormEntity;
import com.biz.crm.orderform.service.IKmsOrderFormService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.List;

/**
 * 商超订货单表
 *
 * @author Gavin
 * @date 2021-05-12 18:17:09
 */
@Slf4j
@RestController
@RequestMapping("/kmsorderform")
@Api(tags = "商超单据-订货单")
public class KmsOrderFormController {
    @Autowired
    private IKmsOrderFormService kmsOrderFormService;

    @Resource
    private RedissonUtil redissonUtil;

    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/list")
    @CrmLog
    public Result<PageResult<KmsOrderFormRespVo>> list(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        PageResult<KmsOrderFormRespVo> result = kmsOrderFormService.findList(kmsOrderFormReqVo);
        return Result.ok(result);
    }


    @ApiOperation(value = "查询列表导出")
    @PostMapping("/listForExport")
    @CrmLog
    public Result<PageResult<KmsOrderFormExportRespVo>> listForExcel(@RequestBody KmsOrderFormReqVo reqVo) {
        PageResult<KmsOrderFormExportRespVo> result = kmsOrderFormService.listFprExport(reqVo);
        return Result.ok(result);
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @PostMapping("/query")
    @CrmLog

    public Result<KmsOrderFormRespVo> query(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        KmsOrderFormRespVo respVo = kmsOrderFormService.query(kmsOrderFormReqVo);
        return Result.ok(respVo);
    }

    /**
     * 新增
     */
    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        kmsOrderFormService.save(kmsOrderFormReqVo);
        return Result.ok();
    }

    /**
     * 更新
     */
    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        kmsOrderFormService.update(kmsOrderFormReqVo);
        return Result.ok("修改成功");
    }

    /**
     * 删除
     */
    @ApiOperation(value = "删除")
    @PostMapping("/delete")
    @CrmLog
    public Result delete(@RequestBody List<String> ids) {
        kmsOrderFormService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        kmsOrderFormService.enableBatch(kmsOrderFormReqVo);
        return Result.ok("启用成功");
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody KmsOrderFormReqVo kmsOrderFormReqVo) {
        kmsOrderFormService.disableBatch(kmsOrderFormReqVo);
        return Result.ok("禁用成功");
    }

    /**
     * 加急订单
     */
    @ApiOperation(value = "加急订单")
    @PostMapping("/expedited")
    @CrmLog
    public Result expedited(@RequestBody List<String> ids) {
        KmsOrderFormReqVo kmsOrderFormReqVo = new KmsOrderFormReqVo();
        kmsOrderFormReqVo.setIds(ids);
        kmsOrderFormService.expedited(kmsOrderFormReqVo);
        return Result.ok();
    }

    /**
     * 取消加急
     */
    @ApiOperation(value = "取消加急")
    @PostMapping("/cancelExpedited")
    @CrmLog
    public Result cancelExpedited(@RequestBody List<String> ids) {
        KmsOrderFormReqVo kmsOrderFormReqVo = new KmsOrderFormReqVo();
        kmsOrderFormReqVo.setIds(ids);
        kmsOrderFormService.cancelExpedited(kmsOrderFormReqVo);
        return Result.ok();
    }

    /**
     * 手动转换订货单
     */
    @ApiOperation(value = "手动转换订货单")
    @PostMapping("/manualTrans")
    @CrmLog
    public Result manualTrans(@RequestBody List<String> ids) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_TRANS_FORM);
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                KmsOrderFormReqVo kmsOrderFormReqVo = new KmsOrderFormReqVo();
                kmsOrderFormReqVo.setIds(ids);
                kmsOrderFormService.manualTrans(kmsOrderFormReqVo);
            } else {
                throw new BusinessException("有人正在进行单据转换，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("操作失败:" + e.getMessage());
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }

    /**
     * 单据推送查询单据信息
     */
    @ApiOperation(value = "单据推送查询单据信息")
    @PostMapping("/findPushList")
    @CrmLog
    public Result<List<KmsOrderFormRespVo>> findPushList(@RequestBody List<String> ids) {
        KmsOrderFormReqVo kmsOrderFormReqVo = new KmsOrderFormReqVo();
        kmsOrderFormReqVo.setIds(ids);
        List<KmsOrderFormRespVo> respVos = kmsOrderFormService.pushList(kmsOrderFormReqVo);
        return Result.ok(respVos);
    }

    /**
     * 单据推送
     */
    @ApiOperation(value = "单据推送")
    @PostMapping("/push")
    @CrmLog
    public Result push(@RequestBody KmsOrderFormReqVo params) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.PUSH_ORDER_FORM_TO_SAP);
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                AssertUtils.isNotEmpty(params.getIds(), "请选择需要推送的单据信息");
                List<KmsOrderFormEntity> kmsOrderFormEntities = kmsOrderFormService.listByIds(params.getIds());
                kmsOrderFormEntities.forEach(o -> {
                    //单据编码集合
                    if (!StringUtils.isEmpty(o.getSapNumber())) {
                        throw new BusinessException(o.getOrderNumber() + "订单已经推送SAP！");
                    }
                });
                params.getIds().forEach(o -> {
                    //数据推送
                    params.setId(o);
                    kmsOrderFormService.push(params);
                });
            } else {
                throw new BusinessException("有人正在推送订单，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("操作失败:" + e.getMessage());
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }
}
