package com.biz.crm.orderform.helper;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.eunm.kms.KmsEnum;
import com.biz.crm.grabrule.mapper.KmsDataAuthMapper;
import com.biz.crm.nebular.kms.rawdata.resp.KmsOrderFormDetailDwRespVo;
import com.biz.crm.nebular.mdm.CrmBaseVo;
import com.biz.crm.orderform.mapper.KmsOrderFormMapper;
import com.biz.crm.orderform.mapper.KmsOrderGoodsMapper;
import com.biz.crm.orderform.model.KmsOrderFormEntity;
import com.biz.crm.orderform.model.KmsOrderGoodsEntity;
import com.biz.crm.rawdata.model.KmsOrderFormDetailDwEntity;
import com.biz.crm.rawdata.model.KmsOrderGoodsDetailDwEntity;
import com.biz.crm.rawdata.service.IKmsOrderFormDetailDwService;
import com.biz.crm.rawdata.service.IKmsOrderGoodsDetailDwService;
import com.biz.crm.trans.service.IKmsTransDataService;
import com.biz.crm.trans.vo.TransDataGoodsVo;
import com.biz.crm.trans.vo.TransDataStoreVo;
import com.biz.crm.trans.vo.TransDataVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.orderform.helper
 * @ClassName KmsOrderFormHelper
 * @Author Gavin
 * @Date 2021/5/13 上午10:19
 * @Description 订货单助手
 */
@Slf4j
@Component
public class KmsOrderFormHelper {

    @Resource
    private KmsOrderFormMapper kmsOrderFormMapper;

    @Resource
    private KmsOrderGoodsMapper kmsOrderGoodsMapper;

    @Resource
    private IKmsOrderFormDetailDwService kmsOrderFormDetailDwService;

    @Resource
    private IKmsOrderGoodsDetailDwService kmsOrderGoodsDetailDwService;

    @Resource
    private IKmsTransDataService kmsTransDataService;

    @Resource
    private KmsDataAuthMapper kmsDataAuthMapper;


    /**
     * 处理订货单版本信息
     *
     * @param dwRespVo
     * @return
     */
    public KmsOrderFormEntity transFormVersion(KmsOrderFormDetailDwRespVo dwRespVo) {
        String orderNumber = dwRespVo.getOrderNumber();
        //根据订单号查询原始单据
        List<KmsOrderFormDetailDwEntity> orderDwEntities = kmsOrderFormDetailDwService.lambdaQuery()
                .eq(KmsOrderFormDetailDwEntity::getOrderNumber, orderNumber).list();
        //根据版本号排序取最大
        KmsOrderFormDetailDwEntity dwEntity = orderDwEntities.stream()
                .sorted(Comparator.comparing(KmsOrderFormDetailDwEntity::getVersionNumber).reversed())
                .findFirst().orElse(null);
        //查询是否存在已转换数据
        LambdaQueryWrapper<KmsOrderFormEntity> build = new LambdaQueryWrapper<KmsOrderFormEntity>()
                .eq(KmsOrderFormEntity::getOrderNumber, orderNumber);
        KmsOrderFormEntity orderFormEntity = kmsOrderFormMapper.selectOne(build);
        if (ObjectUtils.isEmpty(orderFormEntity)) {
            orderFormEntity = CrmBeanUtil.copy(dwEntity, KmsOrderFormEntity.class);
            //赋予权限信息
            orderFormEntity.setBsDirectSystemId(dwRespVo.getBsDirectSystemId());
            orderFormEntity.setCreateCode(dwRespVo.getCreateCode());
            orderFormEntity.setCreateName(dwRespVo.getCreateName());
            orderFormEntity.setCreateOrgCode(dwRespVo.getCreateOrgCode());
            orderFormEntity.setCreateOrgName(dwRespVo.getCreateOrgName());
            orderFormEntity.setCreatePosCode(dwRespVo.getCreatePosCode());
            orderFormEntity.setCreatePosName(dwRespVo.getCreatePosName());
        } else if (!YesNoEnum.yesNoEnum.no.getValue().equals(orderFormEntity.getIsTrans())) {
            BeanUtils.copyProperties(dwEntity, orderFormEntity, KmsUtils.ignoreProperties());
            orderFormEntity.setRemarks(dwEntity.getRemarks());
        }
        //更改转换标示为已转换
        orderDwEntities.forEach(dw -> {
            dw.setTransStatus(YesNoEnum.yesNoEnum.ONE.getValue());
        });
        kmsOrderFormDetailDwService.updateBatchById(orderDwEntities);
        return orderFormEntity;
    }

    /**
     * 处理订货单商品版本数据
     *
     * @param orderFormEntities
     * @param orderNumbers
     */
    public List<KmsOrderGoodsEntity> transGoodsVersion(KmsOrderFormEntity orderFormEntity) {
        String orderNumber = orderFormEntity.getOrderNumber();
        Integer versionNumber = orderFormEntity.getVersionNumber();
        //删除原订单商品明细
        LambdaUpdateWrapper<KmsOrderGoodsEntity> wrapper = new LambdaUpdateWrapper<KmsOrderGoodsEntity>()
                .eq(KmsOrderGoodsEntity::getOrderNumber, orderNumber);
        List<KmsOrderGoodsEntity> deleteOrderGoodsEntities = kmsOrderGoodsMapper.selectList(wrapper);
        Map<String, KmsOrderGoodsEntity> goodsMap = deleteOrderGoodsEntities.stream()
                .collect(Collectors.toMap(o -> o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode(), Function.identity()));
        //订单号查询原始订货单商品详情信息
        List<KmsOrderGoodsDetailDwEntity> detailDwEntities = kmsOrderGoodsDetailDwService.lambdaQuery()
                .eq(KmsOrderGoodsDetailDwEntity::getOrderNumber, orderNumber)
                .eq(KmsOrderGoodsDetailDwEntity::getVersionNumber, versionNumber)
                .list();
        Map<String, KmsOrderGoodsDetailDwEntity> dwEntityMap = detailDwEntities.stream()
                .collect(Collectors.toMap(o -> o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode(), Function.identity(), (v1, v2) -> v1));

        //订单号+版本号查询原始订货单商品详情信息
        Map<String, KmsOrderGoodsEntity> onlyMap = Maps.newHashMap();
        detailDwEntities.forEach(o -> {
            String key = o.getOrderNumber() + "-" + o.getVersionNumber() + "-" + o.getKaGoodsCode();
            KmsOrderGoodsEntity goodsEntity = CrmBeanUtil.copy(o, KmsOrderGoodsEntity.class);
            goodsEntity.setOrderAmount(StringUtils.isEmpty(goodsEntity.getOrderAmount()) ? "0" : goodsEntity.getOrderAmount());
            goodsEntity.setOrderAmountNot(StringUtils.isEmpty(goodsEntity.getOrderAmountNot()) ? "0" : goodsEntity.getOrderAmountNot());
            goodsEntity.setCurUnitOrderQuantity(StringUtils.isEmpty(goodsEntity.getCurUnitOrderQuantity()) ? "0" : goodsEntity.getCurUnitOrderQuantity());
            goodsEntity.setRemarks(o.getRemarks());
            //判断是否重复
            if (!ObjectUtils.isEmpty(onlyMap.get(key))) {
                KmsOrderGoodsEntity en = onlyMap.get(key);
                en.setOrderAmount(new BigDecimal(en.getOrderAmount())
                        .add(new BigDecimal(goodsEntity.getOrderAmount())).toString());
                en.setOrderAmountNot(new BigDecimal(en.getOrderAmountNot())
                        .add(new BigDecimal(goodsEntity.getOrderAmountNot())).toString());
                en.setCurUnitOrderQuantity(new BigDecimal(en.getCurUnitOrderQuantity())
                        .add(new BigDecimal(goodsEntity.getCurUnitOrderQuantity())).toString());
                onlyMap.put(key, en);
            } else {
                KmsOrderGoodsEntity entity = ObjectUtils.isEmpty(goodsMap.get(key))
                        ? new KmsOrderGoodsEntity() : goodsMap.get(key);
                BeanUtils.copyProperties(goodsEntity, entity, KmsUtils.ignoreProperties());
                onlyMap.put(key, entity);
            }
        });
        //删除数据
        List<String> goodsIds = Lists.newArrayList();
        goodsMap.forEach((k, v) -> {
            if (ObjectUtils.isEmpty(dwEntityMap.get(k))) {
                goodsIds.add(v.getId());
            }
        });
        if (!CollectionUtils.isEmpty(goodsIds)) {
            kmsOrderGoodsMapper.deleteBatchIds(goodsIds);
        }
        List<KmsOrderGoodsEntity> onlyList = Lists.newArrayList();
        onlyMap.forEach((k, v) -> {
            onlyList.add(v);
        });
        return onlyList;
    }

    /**
     * 数据转换
     *
     * @param orderFormEntity
     * @param goodsEntities
     */
    public void transData(KmsOrderFormEntity orderFormEntity, List<KmsOrderGoodsEntity> goodsEntities) {
        //封装转换数据信息
        TransDataVo transDataVo = new TransDataVo();
        transDataVo.setOrderType(KmsAdminEnum.OrderType.ORDER.getValue());
        orderFormEntity.setOrderStatusMsg("");
        TransDataStoreVo dataStoreVo = CrmBeanUtil.copy(orderFormEntity, TransDataStoreVo.class);
        dataStoreVo.setSoldToPartyCode(null);
        dataStoreVo.setSoldToPartyName(null);
        dataStoreVo.setKaOrderNumber(orderFormEntity.getKaOrderNumber());
        dataStoreVo.setOrderStatusMsg("");
        dataStoreVo.setKaName(orderFormEntity.getKaName());
        transDataVo.setStoreVos(dataStoreVo);
        if (!StringUtils.isEmpty(orderFormEntity.getCustomerOrgCode())) {
            //当前时间加时间阈值
            Map<String, String> dateMap = DictUtil.dictMap(KmsConstant.KMS_ORDER_WEEK_PRODUCT_DATE);
            Calendar cl = Calendar.getInstance();
            //获取当前用户所属公司
            String day = dateMap.get(orderFormEntity.getCustomerOrgCode());
            AssertUtils.isNotEmpty(day, "请配置星期物料转换时间阈值");
            int dayInt = Integer.parseInt(day);
            cl.add(Calendar.DAY_OF_MONTH, dayInt);
            transDataVo.setOrderDate(DateUtil.format(cl.getTime(), DateUtil.DEFAULT_DAY_PATTERN));
        } else {
            transDataVo.setOrderDate(orderFormEntity.getOrderDate());
        }
        //单据商品Map
        Map<String, List<KmsOrderGoodsEntity>> goodsEntityMap = Maps.newHashMap();
        goodsEntities.forEach(o -> {
            List<KmsOrderGoodsEntity> entities = goodsEntityMap.get(o.getOrderNumber());
            if (CollectionUtils.isEmpty(entities)) {
                entities = Lists.newArrayList();
            }
            entities.add(o);
            goodsEntityMap.put(o.getOrderNumber(), entities);
            TransDataGoodsVo dataGoodsVo = CrmBeanUtil.copy(o, TransDataGoodsVo.class);
            dataGoodsVo.setCurUnitOrderQuantity(o.getCurUnitOrderQuantity());
            dataGoodsVo.setKaOrderNumber(o.getKaOrderNumber());
            dataGoodsVo.setOrderStatusMsg("");
            dataGoodsVo.setKaName(o.getKaName());
            dataGoodsVo.setOrderDate(transDataVo.getOrderDate());
            transDataVo.getGoodsVos().add(dataGoodsVo);
        });
        //数据转换
        kmsTransDataService.transData(transDataVo);
        //结果数据处理
        resultProcessing(orderFormEntity, goodsEntities, transDataVo);
    }

    /**
     * 数据转换
     *
     * @param orderFormEntity
     * @param goodsEntities
     * @param transDataVo
     */
    private void resultProcessing(KmsOrderFormEntity orderFormEntity, List<KmsOrderGoodsEntity> goodsEntities, TransDataVo transDataVo) {
        String orderNumber = orderFormEntity.getOrderNumber();
        //转换结果信息
        TransDataStoreVo transDataStoreVo = transDataVo.getStoreVos();
        //门店处理
        BeanUtils.copyProperties(transDataStoreVo, orderFormEntity, Arrays.stream(CrmBaseVo.class.getDeclaredFields())
                .map(Field::getName)
                .collect(Collectors.toList()).toArray(new String[]{}));
        //商品处理
        Map<String, TransDataGoodsVo> dataGoodsVoMap = transDataVo.getGoodsVos().stream()
                .collect(Collectors.toMap(TransDataGoodsVo::getKaGoodsCode, Function.identity()));
        goodsEntities.forEach(goods -> {
            //商品行信息处理
            String kaGoodsCode = goods.getKaGoodsCode();
            TransDataGoodsVo transDataGoodsVo = dataGoodsVoMap.get(kaGoodsCode);
            BeanUtils.copyProperties(transDataGoodsVo, goods, Arrays.stream(CrmBaseVo.class.getDeclaredFields())
                    .map(Field::getName)
                    .collect(Collectors.toList()).toArray(new String[]{}));
            if (!StringUtils.isEmpty(goods.getOrderStatusMsg())) {
                goods.setOrderStatus(KmsEnum.OrderStatus.S100.getValue());
            } else {
                goods.setOrderStatus(KmsEnum.OrderStatus.S400.getValue());
                goods.setOrderStatusMsg("转换成功");
            }
        });
        //确认转换状态
        String orderStatusMsg = orderFormEntity.getOrderStatusMsg();
        List<KmsOrderGoodsEntity> errorList = goodsEntities.stream()
                .filter(goods -> KmsEnum.OrderStatus.S100.getValue().equals(goods.getOrderStatus())).collect(Collectors.toList());
        if (!StringUtils.isEmpty(orderStatusMsg) || !CollectionUtils.isEmpty(errorList)) {
            orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S100.getValue());
            if (!CollectionUtils.isEmpty(errorList) && !StringUtils.isEmpty(orderStatusMsg)) {
                orderFormEntity.setOrderStatusMsg(orderStatusMsg + "|商品行信息存在转换失败数据");
            } else if (!CollectionUtils.isEmpty(errorList) && StringUtils.isEmpty(orderStatusMsg)) {
                orderFormEntity.setOrderStatusMsg("商品行信息存在转换失败数据");
            }
        } else {
            orderFormEntity.setOrderStatus(KmsEnum.OrderStatus.S400.getValue());
        }
        if (StringUtils.isEmpty(orderFormEntity.getOrderStatusMsg())) {
            orderFormEntity.setOrderStatusMsg("转换成功");
        }
    }
}