package com.biz.crm.salesdata.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.kms.rawdata.resp.KmsSaleDataFormDetailDwRespVo;
import com.biz.crm.nebular.kms.salesdata.req.KmsSalesDataReqVo;
import com.biz.crm.nebular.kms.salesdata.resp.KmsSalesDataRespVo;
import com.biz.crm.salesdata.helper.KmsSalesDataHelper;
import com.biz.crm.salesdata.mapper.KmsSalesDataMapper;
import com.biz.crm.salesdata.model.KmsSalesDataEntity;
import com.biz.crm.salesdata.service.IKmsSalesDataService;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectStoreMapper;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.trans.helper.KmsTransDataHelper;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 销售数据表接口实现
 *
 * @author gavin
 * @date 2021-11-24 14:22:40
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsSalesDataServiceExpandImpl")
public class KmsSalesDataServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsSalesDataMapper, KmsSalesDataEntity> implements IKmsSalesDataService {

    @Resource
    private KmsSalesDataMapper kmsSalesDataMapper;

    @Resource
    private KmsSalesDataHelper kmsSalesDataHelper;

    @Resource
    private KmsTenantryDirectCustomerOrgMapper kmsTenantryDirectCustomerOrgMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private MdmTerminalFeign mdmTerminalFeign;

    @Resource
    private MdmCustomerMsgFeign mdmCustomerMsgFeign;

    @Resource
    private KmsTransDataHelper kmsTransDataHelper;

    @Resource
    private KmsTenantryDirectStoreMapper kmsTenantryDirectStoreMapper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsSalesDataRespVo> findList(KmsSalesDataReqVo reqVo) {
        Page<KmsSalesDataRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsSalesDataRespVo> list = kmsSalesDataMapper.findList(page, reqVo);
        return PageResult.<KmsSalesDataRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsSalesDataRespVo
     */
    @Override
    public KmsSalesDataRespVo query(String id) {
        AssertUtils.isNotEmpty(id, "id不能为空!");
        KmsSalesDataEntity entity = kmsSalesDataMapper.selectById(id);
        KmsSalesDataRespVo respVo = CrmBeanUtil.copy(entity, KmsSalesDataRespVo.class);
        KmsTenantryDirectCustomerOrgEntity bs = kmsTenantryDirectCustomerOrgMapper.selectById(respVo.getBsDirectSystemId());
        if (bs != null) {
            respVo.setKaCode(bs.getBsDirectSystemCode());
            respVo.setKaName(bs.getBsDirectSystemName());
        }
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsSalesDataReqVo reqVo) {

    }

    /**
     * 编辑
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsSalesDataReqVo reqVo) {

    }

    /**
     * 删除
     *
     * @param ids
     * @return void
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {

    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsSalesDataReqVo reqVo) {
        //设置状态为启用
        List<KmsSalesDataEntity> kmsSalesDataEntities = kmsSalesDataMapper.selectBatchIds(reqVo.getIds());
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        String menuObjectStr = (String) menuObject;
        if (CollectionUtils.isNotEmpty(kmsSalesDataEntities)) {
            kmsSalesDataEntities.forEach(o -> {
                KmsSalesDataEntity oldEntity = CrmBeanUtil.copy(o, KmsSalesDataEntity.class);
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                //修改日志
                if (menuObject != null) {
                    crmLogSendUtil.sendForUpdate(menuObjectStr, o.getId(), o.getOrderNumber(), oldEntity, o);
                }
            });
        }
        this.updateBatchById(kmsSalesDataEntities);

    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsSalesDataReqVo reqVo) {
        //设置状态为禁用
        List<KmsSalesDataEntity> kmsSalesDataEntities = kmsSalesDataMapper.selectBatchIds(reqVo.getIds());
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        String menuObjectStr = (String) menuObject;
        if (CollectionUtils.isNotEmpty(kmsSalesDataEntities)) {
            kmsSalesDataEntities.forEach(o -> {
                KmsSalesDataEntity oldEntity = CrmBeanUtil.copy(o, KmsSalesDataEntity.class);
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                //修改日志
                if (menuObject != null) {
                    crmLogSendUtil.sendForUpdate(menuObjectStr, o.getId(), o.getOrderNumber(), oldEntity, o);
                }
            });
        }
        this.updateBatchById(kmsSalesDataEntities);
    }

    /**
     * 自动转换销售数据
     *
     * @param dwRespVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoTransData(KmsSaleDataFormDetailDwRespVo dwRespVo) {
        //匹配处理销售版本信息
        KmsSalesDataEntity salesDataEntity = kmsSalesDataHelper.transFormVersion(dwRespVo);
        if (!ObjectUtils.isEmpty(salesDataEntity)) {
            //数据转换
            kmsSalesDataHelper.transData(salesDataEntity);
            //保存信息
            this.saveOrUpdate(salesDataEntity);
        }
    }

    /**
     * 手动转换销售数据
     *
     * @param kmsSalesDataReqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void manualTrans(KmsSalesDataReqVo kmsSalesDataReqVo) {
        List<String> ids = kmsSalesDataReqVo.getIds();
        AssertUtils.isNotEmpty(ids, "请选择待转换数据");
        //查询验收单状态，待确认、转换失败
        List<KmsSalesDataEntity> kmsSalesDataEntities = this.lambdaQuery()
                .in(KmsSalesDataEntity::getId, ids).list();
        List<KmsSaleDataFormDetailDwRespVo> detailDwRespVos = CrmBeanUtil.copyList(kmsSalesDataEntities, KmsSaleDataFormDetailDwRespVo.class);
        detailDwRespVos.forEach(dwRespVo -> {
            //匹配处理验收单版本信息
            KmsSalesDataEntity kmsSalesDataEntity = kmsSalesDataHelper.transFormVersion(dwRespVo);
            if (!ObjectUtils.isEmpty(kmsSalesDataEntity)) {
                //数据转换
                kmsSalesDataHelper.transData(kmsSalesDataEntity);
                //保存信息
                this.saveOrUpdate(kmsSalesDataEntity);
            }
        });
    }
}
