package com.biz.crm.supermarket.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.kms.supermarket.req.KmsSupplierReqVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsSupplierRespVo;
import com.biz.crm.supermarket.mapper.KmsSupplierMapper;
import com.biz.crm.supermarket.model.KmsSupplierEntity;
import com.biz.crm.supermarket.service.IKmsSupplierService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.KmsOrgUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 售达方与供应商关系表接口实现
 *
 * @author gavin
 * @date 2021-07-06 13:58:17
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsSupplierServiceExpandImpl")
public class KmsSupplierServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsSupplierMapper, KmsSupplierEntity> implements IKmsSupplierService {

    @Resource
    private KmsSupplierMapper kmsSupplierMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsSupplierRespVo> findList(KmsSupplierReqVo reqVo) {
        Page<KmsSupplierRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsSupplierRespVo> list = kmsSupplierMapper.findList(page, reqVo);
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            //查询当前创建者所属公司
            List<String> createOrgCodes = list.stream().map(KmsSupplierRespVo::getCreateOrgCode)
                    .distinct().collect(Collectors.toList());
            Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
            list.forEach(o -> {
                o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
            });
        }
        return PageResult.<KmsSupplierRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsSupplierRespVo
     */
    @Override
    public KmsSupplierRespVo query(KmsSupplierReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "请选择数据ID");
        KmsSupplierEntity byId = this.getById(reqVo.getId());
        AssertUtils.isNotNull(byId, "未查询到数据信息");
        KmsSupplierRespVo copy = CrmBeanUtil.copy(byId, KmsSupplierRespVo.class);
        return copy;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsSupplierReqVo reqVo) {
        this.checkParams(reqVo);
        KmsSupplierEntity entity = CrmBeanUtil.copy(reqVo, KmsSupplierEntity.class);
        this.save(entity);
        KmsSupplierRespVo newData = CrmBeanUtil.copy(entity, KmsSupplierRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    private void checkParams(KmsSupplierReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemId(), "直营体系Id[bsDirectSystemId]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系编码[bsDirectSystemCode]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemName(), "直营体系名称[bsDirectSystemName]不能为空");
        AssertUtils.isNotEmpty(reqVo.getSellPartyId(), "售达方ID[sellPartyId]不能为空");
        AssertUtils.isNotEmpty(reqVo.getSellPartyCode(), "售达方编码[sellPartyCode]不能为空");
        AssertUtils.isNotEmpty(reqVo.getSellPartyName(), "售达方名称[sellPartyName]不能为空");
        AssertUtils.isNotEmpty(reqVo.getSupplierCode(), "供应商编码[supplierCode]不能为空");
        //重复性校验
        List<KmsSupplierEntity> entities = this.lambdaQuery().eq(KmsSupplierEntity::getBsDirectSystemId, reqVo.getBsDirectSystemId())
                .eq(KmsSupplierEntity::getSellPartyId, reqVo.getSellPartyId())
                .eq(KmsSupplierEntity::getSupplierCode, reqVo.getSupplierCode()).list();
        if (!org.springframework.util.CollectionUtils.isEmpty(entities)) {
            List<KmsSupplierEntity> collect = entities.stream()
                    .filter(o -> !o.getId().equals(reqVo.getId())).collect(Collectors.toList());
            if (!org.springframework.util.CollectionUtils.isEmpty(collect)) {
                throw new BusinessException("已存在相同配置");
            }
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsSupplierReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "请选择数据ID");
        this.checkParams(reqVo);
        KmsSupplierEntity byId = this.getById(reqVo.getId());
        AssertUtils.isNotNull(byId, "未查询到数据信息");
        BeanUtils.copyProperties(reqVo, byId);
        KmsSupplierRespVo oldData = CrmBeanUtil.copy(byId, KmsSupplierRespVo.class);
        this.updateById(byId);
        KmsSupplierRespVo newData = this.query(reqVo);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(KmsSupplierReqVo reqVo) {
        List<KmsSupplierEntity> kmsSupplierEntities = kmsSupplierMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsSupplierEntities)) {
            kmsSupplierEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsSupplierEntities);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsSupplierReqVo reqVo) {
        //设置状态为启用
        List<KmsSupplierEntity> kmsSupplierEntities = kmsSupplierMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsSupplierEntities)) {
            kmsSupplierEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsSupplierEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsSupplierReqVo reqVo) {
        //设置状态为禁用
        List<KmsSupplierEntity> kmsSupplierEntities = kmsSupplierMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsSupplierEntities)) {
            kmsSupplierEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsSupplierEntities);
    }
}
